/*
 * libthor - Tizen Thor communication protocol
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <sys/types.h>
#include <errno.h>
#include <string.h>

#include "thor.h"
#include "thor_internal.h"
#include "thor_transport.h"


int t_thor_init(thor_device_handle *th)
{
	switch (th->type) {
	case THOR_TRANSPORT_USB:
		return thor_usb_init(th);
	case THOR_TRANSPORT_NET:
		return thor_net_init(th);
	default:
		break;
	}

	return -ENODEV;
}

void t_thor_cleanup(thor_device_handle *th)
{
	switch (th->type) {
	case THOR_TRANSPORT_USB:
		thor_usb_cleanup(th);
		break;
	case THOR_TRANSPORT_NET:
		thor_net_cleanup(th);
		break;
	default:
		break;
	}
}

int t_thor_open(thor_device_handle *th, struct thor_device_id *dev_id, int wait)
{
	if (!th->ops || !th->ops->open)
		return -ENOENT;

	return th->ops->open(th, dev_id, wait);
}

void t_thor_close(thor_device_handle *th)
{
	if (!th->ops || !th->ops->close)
		return;

	th->ops->close(th);
}

int t_thor_send(thor_device_handle *th, unsigned char *buf,
		off_t count, int timeout)
{
	if (!th->ops || !th->ops->send)
		return -ENOENT;

	return th->ops->send(th, buf, count, timeout);

}

int t_thor_recv(thor_device_handle *th, unsigned char *buf,
		off_t count, int timeout)
{
	if (!th->ops || !th->ops->recv)
		return -ENOENT;

	return th->ops->recv(th, buf, count, timeout);
}

int t_thor_send_raw_data(thor_device_handle *th,
			 struct thor_data_src *data,
			 off_t trans_unit_size,
			 thor_progress_cb report_progress,
			 void *user_data)
{
	if (!th->ops || !th->ops->send_data)
		return -EIO;

	return th->ops->send_data(th, data, trans_unit_size,
				  report_progress, user_data);
}
