var EventEmitter = require('eventemitter2').EventEmitter2;
var logger = require('../logging/index').createNamedLogger('Device');
var request = require('request');
var createError = require('http-errors');
var msfUtils = require('../utils/index');
var async = require('async');

var LAUNCHER_PROXY_URL = 'http://multiscreen.samsung.com/launchservice/v2/launch';

var device = new EventEmitter();
device.lastLaunchData = {};
device.allowAllContent = false;

/*
 A private reference to the underlying device implementation
 set by the registerDevice method
*/
var _device;


/*
 Use a getter for attributes so that we can see errors
 if anything is accessing them before registerDevice is called;
 */
Object.defineProperty(device, 'attributes', {
    get: function() {
        if(!_device) logger.warn('Access of device attribute before device implementation was registered');
        return _device.attributes;
    }
});


/*
 Registers a device implementation (typically a device plugin)
 that handles the underlying implementation for providing attributes about the device
 and implements the pincode, application, and webapplication methods.
 The preferred way to register a device implementation is using the device plugin base
*/

device.registerDevice = function(impl){
    if(!_device){
        _device = impl;
        logger.info(impl.constructor.name + " is now registered and ready");
        device.emit('ready');
    }
    else logger.error('The device ('+impl.constructor.name+') has already been registered');
};



/*
 PinCodes
 */

device.showPinCode = function(options, callback){
    logger.verbose("showPinCode", options);
    if(!options || !options.code){
        options = msfUtils.getPinCode();
    }
    _device.showPinCode(options, callback);
    device.emit('showPinCode', options);
};

device.hidePinCode = function(options, callback){
    logger.verbose("hidePinCode", options);
    _device.hidePinCode(options, callback);
    device.emit('hidePinCode', options);
};

device.getPinCode = function(options, callback){
    logger.verbose("getPinCode", options);
    callback(null, msfUtils.getPinCode());
    device.emit('getPinCode', options);
};


/*
  Applications
*/

device.getApplication = function(options, callback){
    logger.verbose("getApplication", options);
    _device.getApplication(options, callback);
    device.emit('getApplication', options);
};

device.getApplicationData = function(options, callback){
    logger.verbose("getApplicationData", options);
    _device.getApplicationData(options, callback);
    device.emit('getApplicationData', options);

};

device.launchApplication = function(options, callback){
    logger.verbose("launchApplication", options);
    _device.launchApplication(options, callback);
    device.emit('launchApplication', options);

};

device.terminateApplication = function(options, callback){
    logger.verbose("terminateApplication", options);
    _device.terminateApplication(options, callback);
    device.emit('terminateApplication', options);
};

device.installApplication = function(options, callback){
    logger.verbose("installApplication", options);
    _device.installApplication(options, callback);
    device.emit('installApplication', options);
};


device.getWebApplication = function(options, callback){
    logger.verbose("getWebApplication", options);
    _device.getWebApplication(options, callback);
    device.emit('getWebApplication', options);

};

device.terminateWebApplication = function(options, callback){
    logger.verbose("terminateWebApplication", options);
    _device.terminateWebApplication(options, callback);
    device.emit('terminateWebApplication', options);
};


device.launchWebApplication = function(options, callback){
    logger.verbose("launchWebApplication", options);

    if(this.allowAllContent){

        _device.launchWebApplication(options, callback);
        device.emit('launchWebApplication', options);

    }else{

        /*
         * Check for valid response from the url before we attempt to launch it
         * Use only HEAD method to reduce load
         * Use jar for cookie support (tough-cookie is a required module) otherwise some sites will infinite redirect (netflix)
         */
        var tough = require('tough-cookie');
        var jar = request.jar();

        request({ method:'GET', uri:LAUNCHER_PROXY_URL, qs:{url:options.url}, followAllRedirects: true, jar:jar }, function (error, response, body) {

            // Unexpected error (could be DNS, no handling by the server, etc...
            if (error) return callback(createError(400,error.message || 'the url could not be resolved'));

            // Respond with the servers error status code if it is an higher level status
            // This is better than !== 200 so we can handle other valid codes such as cache responses
            if (response.statusCode >= 400) return callback(createError(response.statusCode,body));

            _device.launchWebApplication(options, callback);
            device.emit('launchWebApplication', options);

        });
    }
};

module.exports = device;
