var _ = require('lodash');
var util = require('util');
var fs = require('fs');
var path = require('path');
var express = require('express');
var createError = require('http-errors');
var MSFDevicePlugin = require('../device-plugin');


/*
 Constants
 */
var TEST_APP_ID      = 'e16522e6-1dbb-11e4-9ae8-b2227cce2b54';
var APPS_FOLDER      = 'multiscreen-apps';
var WEBLAUNCHER_ID   = 'weblauncher';
var WEBLAUNCHER_PATH = '/resources/webapplauncher/';

function TizenWebSimulator(config){
    /*
     Call the super constructor
     */
    MSFDevicePlugin.apply(this, arguments);

    this.appsPath = path.join(process.env.HOME || process.env.HOMEPATH || process.env.USERPROFILE,'/'+APPS_FOLDER+'/');

    /*
     Set the udn and deviceId in settings if none exist
     Then save the settings to disk
     */
    var settings = this.service.settings;
    if(!settings.data.udn) settings.data.udn = this.service.utils.generateUdn();
    if(!settings.data.deviceId) settings.data.deviceId = this.service.utils.generateDeviceId();
    settings.save();

    /*
     * Initially set the device attributes
     */
    this.attributes.type = 'Tizen Simulator';
    this.attributes.ip   = this.utils.getInternalIP();
    this.attributes.id   = settings.data.deviceId;
    this.attributes.duid = settings.data.deviceId;
    this.attributes.udn  = settings.data.udn;


    /*
     * Setup route handlers
     */

    // Map the apps path to the apps folder on the users system
    this.app.use('/apps',express.static(this.appsPath));

    // Built in app for unit test
    this.app.use('/apps/'+TEST_APP_ID+'/',express.static(path.join(__dirname,'testapp')));

    /*
     * wait 1 second and register this plugin as the device (will also fire the device ready event)
     * we wait one second in case other plugins were loaded after this and they listen for the ready event
     */
    setTimeout(this.register.bind(this), 1000);

    this.runningApp = {
        id : "9804752019347820",
        version : "1.2.4",
        name : "FakeApplication",
        running : false
    };

}

MSFDevicePlugin.extend(TizenWebSimulator);

TizenWebSimulator.prototype.resetRunningApp = function(){
    window.document.getElementById('document').src = 'about:blank';
    this.runningApp = {
        id : "9804752019347820",
        version : "1.2.4",
        name : "FakeApplication",
        running : false
    };
};


TizenWebSimulator.prototype.showPinCode = function(pinCode, callback) {
    this.logger.info("SHOWING PIN CODE : "+pinCode.code);
    callback(null,true);
};

TizenWebSimulator.prototype.hidePinCode = function(options, callback) {
    this.logger.info("HIDING PIN CODE");
    callback(null,true);
};

TizenWebSimulator.prototype.launchApplication = function(options, callback) {
    this.logger.verbose('launchApplication', options);
    if (this.findAppById(options.id)){
        window.document.getElementById('document').src = 'http://127.0.0.1:' + this.service.port + '/apps/'+options.id+'/index.html';
        this.runningApp = {
            id : options.id,
            version : options.version || '0.0.0',
            name : options.name || 'App('+options.id+')',
            running : true
        };
        return callback(null,true);
    }else{
        return callback(createError(404,"Application not found"));
    }

};

TizenWebSimulator.prototype.terminateApplication = function(options, callback) {

    if(options.id === this.service.device.webLauncherId || this.findAppById(options.id)){
        window.document.getElementById('document').src = 'about:blank';
        this.resetRunningApp();
        callback(null, true);

    }else{
        callback(createError(404,"Application not found"));
    }
};

TizenWebSimulator.prototype.getApplication = function(options, callback) {

    if(options.id === this.service.device.webLauncherId || this.findAppById(options.id)){
        callback(null,{
            id : options.id,
            version : "1.2.4",
            name : "App-"+options.id,
            running : this.runningApp.id === options.id
        });
    }else{
        return callback(createError(404,"Application not found"));
    }

};

TizenWebSimulator.prototype.installApplication = function(options, callback) {
   callback(null,true);
};

TizenWebSimulator.prototype.getApplicationData = function(options, callback) {
    this.logger.verbose("getApplicationData", options);
    callback(null, this.runningApp);
};


TizenWebSimulator.prototype.findAppById = function(id){
    this.logger.verbose("findAppById : ", id);
    var appPath = path.normalize(this.appsPath + '/'+id+'/index.html');
    if(id === TEST_APP_ID || fs.existsSync(appPath)){
        return appPath;
    }else{
        return false;
    }
};

TizenWebSimulator.prototype.getWebApplication = function(options, callback) {
    this.logger.verbose('getWebApplication', options);
    callback(null,{
        id : WEBLAUNCHER_ID,
        version : "0.0.0",
        name : 'MultiScreen Web Launcher',
        running : this.runningApp.id === WEBLAUNCHER_ID && this.runningApp.url === options.url
    });
};

TizenWebSimulator.prototype.launchWebApplication = function(options, callback) {
    this.logger.verbose('launchWebApplication', options);
    if (window.document.getElementById('document').src !== options.url){
        window.document.getElementById('document').src = options.url;
    }
    this.runningApp = {
        id : WEBLAUNCHER_ID,
        version : '0.0.0',
        name : 'MultiScreen Web Launcher',
        url  : options.url,
        running : true
    };
    return callback(null,true);
};

TizenWebSimulator.prototype.terminateWebApplication = function(options, callback) {
    this.logger.verbose('terminateWebApplication', options);
    if(options.id === this.service.device.webLauncherId || this.findAppById(options.id)){
        this.resetRunningApp();
        callback(null, true);

    }else{
        callback(createError(404,"Application not found"));
    }
};

module.exports = TizenWebSimulator;
