var _ = require('lodash');
var util = require('util');
var channelManager = require("./channels/index").ChannelManager;
var createError = require('http-errors');
var MSFPlugin = require('../plugin');


/*
 Constants
 */
var URI_API = '/ms/1.0/';
var CAPABILITY = 'samsung:multiscreen:1';


/*
 Maps device attributes to legacy device info
 */
var legacyAttributesMap = {
    duid            : 'DUID',
    model           : 'Model',
    networkType     : 'NetworkType',
    ssid            : 'SSID',
    ip              : 'IP',
    firmwareVersion : 'FirmwareVersion',
    countryCode     : 'CountryCode',
    name            : 'DeviceName',
    id              : 'DeviceID',
    description     : 'ModelDescription',
    modelName       : 'ModelName',
    udn             : 'UDN',
    resolution      : 'Resolution',
    serviceUri      : 'ServiceURI',
    dialUri         : 'DialURI'
};

/*
 Maps RPC method names to class methods
 */
var methodMap = {
    'ms.device.getInfo' : 'getDeviceInfo',
    'ms.device.showPinCode' : 'showPinCode',
    'ms.device.hidePinCode' : 'hidePinCode',
    'ms.device.getPinCode' : 'getPinCode',
    'ms.device.createChannel' : 'createChannel',
    'ms.device.getChannelInfo' : 'getChannelInfo'
};



function ApiV1(config){

    /* Call the super constructor */
    MSFPlugin.apply(this, arguments);


    var self = this;

    /*
    Root of API v1 returns device information
    */
    this.app.get(URI_API, function(req, res){
        res.send(self.getDeviceInfo({__req : req }));
    });

    /*
    Processes all JSON RPC methods
    */
    this.app.post(URI_API, this.processRpcRequest.bind(this));

}

MSFPlugin.extend(ApiV1);

ApiV1.prototype.formatDeviceInfo = function(info){

    /*
     Map the new device attributes to the old format
     */
    var oldAttr = {};

    _.forOwn(info, function(val, key) {
        if(legacyAttributesMap[key]){
            oldAttr[legacyAttributesMap[key]] = val;
        }
    });

    oldAttr.Capabilities = [
        {
            name: CAPABILITY,
            port: this.service.port,
            location: URI_API
        }
    ];

    return oldAttr;
};


ApiV1.prototype.getDeviceInfo = function(options, callback){

    options = options || {};

    // Clone and rewrite ip based properties
    var info = _.clone(this.device.attributes);

    // Use the incoming host address as the ip if it not loopback (mainly to address evolution kit & widi)
    if(options.__req.host && options.__req.host.indexOf("127.") !== 0){
        info.ip = options.__req.host;
    }else{
        info.ip = this.device.attributes.ip;
    }
    info.serviceUri = "http://"+ options.__req.get('host') + URI_API;

    // TODO : Remove the dial information from here (hardcoded for now) and just move the dial proxy plugin code to here
    info.dialUri = "http://" + options.__req.get('host') + '/ws/apps/';

    var formattedInfo = this.formatDeviceInfo(info);

    // Support both sync and async calls to this function
    if(_.isFunction(callback)){
        callback(null, formattedInfo);
    }

    return formattedInfo;
};

ApiV1.prototype.showPinCode = function(options, callback){
    var self = this;
    if(this.device.getCloudPinCode){
        this.device.getCloudPinCode(options, function(err, pincode){
            if(err) return callback(err);
            self.device.showPinCode(pincode, callback);

        });
    }else{
        this.device.showPinCode({}, callback);
    }
};

ApiV1.prototype.hidePinCode = function(options, callback){
    return this.device.hidePinCode({}, callback);
};

ApiV1.prototype.getPinCode = function(options, callback){

    // Check for existance of the method because clouddiscovery v1 adds it to device;
    if(this.device.getCloudPinCode){

        // Only allow the host device to make this request
        if(!this.utils.isInternalIp(options.__req.ip)) return callback(createError(401));
        return this.device.getCloudPinCode(options, callback);

    }else{
        return callback(createError(404,'Method not found'));
    }
};

ApiV1.prototype.createChannel = function(options, callback){

    options = options || {};

    var id = options.id;

    if(typeof id !== 'string') return callback(createError(400,'`id` is required'));

    var channel = channelManager.getChannel(id) || channelManager.createChannel(id);

    if(channel){

        var info = {
            id:channel.id,
            endpoint: "ws://"+options.__req.get('host')+channel.endpoint,
            // This is here for legacy support
            __token: "deprecated"
        };
        callback(null, info);

    }else{
        callback(createError(500,'Unable to create channel'));

    }

};

ApiV1.prototype.getChannelInfo = function(options, callback){

    options = options || {};

    var id = options.id;

    if(typeof id !== 'string') return callback(createError(400,'`id is required`'));

    var channel = channelManager.getChannel(id);

    if(channel && channel.endpoint){
        var info = {
            id:channel.id,
            endpoint: "ws://"+options.__req.get('host')+channel.endpoint,
            hostConnected:channel.hostConnection ? true : false
        };

        callback(null,info);

    }else{
        callback(createError(404,'Channel not found'));
    }

};


ApiV1.prototype.processRpcRequest = function(req, res, next){

    try{

        var message = req.body;
        var methodName = methodMap[message.method];

        // Verify the method exist
        if(!methodName) return next(createError(404,'Method not found'));

        // Let the message be processed
        else{

            var responseMessage;

            // Add the request object to the params so methods have it if needed
            message.params.__req = req;

            this[methodName](message.params, function(err, result){
                if(err){
                    return next(err);
                }else{
                    responseMessage = {
                        result : result,
                        id : message.id
                    };
                    res.send(responseMessage);
                }
            });
        }

    }catch (e){
        this.logger.error(e.message, e.stack);
        return next(createError(500));
    }

};

module.exports = ApiV1;