var _ = require('lodash');
var util = require('util');
var url = require('url');
var Channel = require('./channels/index').Channel;
var request = require('request');
var createError = require('http-errors');
var MSFPlugin = require('../plugin');


/*
 Constants
 */
var URI_PATH = '/api/v2/';

/*
 Utility function for create REST callback responders from route handlers
 */
function createCallbackResponder(req, res, next){

    return function(err, result){

        if(err){
            return next(err);
        }else{
            // ensure standardized json
            if(_.isBoolean(result)) result = {ok:result};
            res.send(result);
        }
    };
}


function ApiV2(config){

    /* Call the super constructor */
    MSFPlugin.apply(this, arguments);

    this.channels = {};

    // REST API : Service
    this.app.get(URI_PATH,this.getServiceInfo.bind(this));

    // REST API : PinCode
    this.app.post(URI_PATH + 'pincode',this.showPinCode.bind(this));
    this.app.delete(URI_PATH + 'pincode',this.hidePinCode.bind(this));

    // REST API : Application
    this.app.get(URI_PATH + 'applications/:id',this.getApplication.bind(this));
    this.app.post(URI_PATH + 'applications/:id',this.launchApplication.bind(this));
    this.app.delete(URI_PATH + 'applications/:id',this.terminateApplication.bind(this));
    this.app.put(URI_PATH + 'applications/:id',this.installApplication.bind(this));

    // REST API : Web Application
    this.app.get(URI_PATH + 'webapplication/',this.getWebApplication.bind(this));
    this.app.post(URI_PATH + 'webapplication/',this.launchWebApplication.bind(this));
    this.app.delete(URI_PATH + 'webapplication/',this.terminateWebApplication.bind(this));

    // TODO : Remove this once we can pass parameters to all devices including Tizen
    this.app.get(URI_PATH + 'webapplication/data',function(req, res, next){
        this.device.getApplicationData(null, createCallbackResponder(req, res, next));
    }.bind(this));

    this.configureChannels();

}


MSFPlugin.extend(ApiV2);


/*
 Sets up the channel container and creates channels on the fly for upgrade requests
 */
ApiV2.prototype.configureChannels = function(){

    var self = this;

    this.server.on('upgrade',function(req, socket, upgradeHead){

        var u = url.parse(req.url);

        /*
         * Check is it is a channel request
         */

        // Verify that the url path matches this api path plus `channels`
        if(u && u.pathname.indexOf(URI_PATH+'channels') === 0){

            var path = u.pathname;

            if(!self.channels[path]){

                var channel = self.channels[path] = new Channel(path, self.server);

                if(channel){

                    channel.server._onHandleUpgrade(req, socket, upgradeHead);
                    // If the server shutsDown remove it from the mapped channels
                    channel.on("shutDown", function(){
                        delete self.channels[path];
                        self.channels[path] = void 0;
                    });

                }else{
                    socket.end();
                }
            }
        }

    });

};



ApiV2.prototype.getServiceInfo = function(req, res, next){

    this.logger.verbose("getServiceInfo");

    var self = this;

    var infoHandler = function(err, callback){
        var info = {
            id      : self.device.attributes.id,
            name    : self.device.attributes.name,
            version : self.service.version,
            device  : self.device.attributes,
            type    : self.device.attributes.type,
            uri     : "http://"+ req.get('host') + URI_PATH
        };
        callback(null,info);
    };

    infoHandler({}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.showPinCode = function(req, res, next){
    this.logger.verbose("showPinCode");
    this.device.showPinCode({}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.hidePinCode = function(req, res, next){
    this.logger.verbose("hidePinCode");
    this.device.hidePinCode({}, createCallbackResponder(req, res, next));
};


ApiV2.prototype.getApplication = function getApplication(req, res, next){
    this.logger.verbose("getApplication : ", req.params.id);
    this.device.getApplication( {id:req.params.id}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.launchApplication = function launchApplication(req, res, next){
    this.logger.verbose("launchApplication : ", req.params.id);
    var data = req.body;
    this.device.launchApplication({id:req.params.id, data:data}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.terminateApplication = function stopApplication(req, res, next){
    this.logger.verbose("stopApplication : ", req.params.id);
    this.device.terminateApplication({id:req.params.id}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.installApplication = function installApplication(req, res, next){
    this.logger.verbose("installApplication : ", req.params.id);
    this.device.installApplication({id:req.params.id}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.getWebApplication = function getWebApplication(req, res, next){
    this.logger.verbose("getWebApplication");
    this.device.getWebApplication({url:req.body.url}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.launchWebApplication = function launchWebApplication(req, res, next){
    this.logger.verbose("launchWebApplication", req.body);
    this.device.launchWebApplication({url:req.body.url}, createCallbackResponder(req, res, next));
};

ApiV2.prototype.terminateWebApplication = function stopWebApplication(req, res, next){
    this.logger.verbose("stopWebApplication", req.body);
    this.device.terminateWebApplication({url:req.body.url}, createCallbackResponder(req, res, next));
};




module.exports = ApiV2;