var util = require('util');
var uuid = require('node-uuid');
var EventEmitter = require('eventemitter2').EventEmitter2;
var url = require("url");
var _ = require("lodash");
var forge = require("node-forge");

/* TODO : Need to clean up these path dependencies... perhaps a global msfRootRequire method */
var service = require('../../../lib/multiscreen-service');
var WebSocketServer = require('../../../lib/ws').Server;

var config = service.config;
var logger = service.logger.createNamedLogger('Channel v1');



var privatePEM = "-----BEGIN RSA PRIVATE KEY-----" +
    "MIICWwIBAAKBgQDfaldZKOKdkfvfYiFgX/ZRHdQwNrb8U8imZ9gNOBXtrDu/hGxH" +
    "EgyrZ9iMqoIcIhgxBzcKwKBAp4xu6yB3AOZiBwLI73ajox/CpIzXE9yPevd5wQ+X" +
    "HctIQazp0qrE9Py5Q5Ox7HB9rmKjSISKQ3A1JtEVbl0bI0iMf4QCtl/FdQIDAQAB" +
    "AoGAcRYt0pasZp/fQ0ozpMnOL28O2bzIUN7EAN8tcDuEdgKpV57bA/px6U0uQr2I" +
    "EF99qUuCo/Gu7CsjHX3st+//rIqaopZKeKA7Pcf5DndotFmgY24qyTzAZIM+FpXD" +
    "UXDswr49Sd1lxM6KYCD19qjufiGpP2gjI8IzMPT/CtiqeX0CQQD5Foh8iIdaKhus" +
    "uJsm3PTYRjyB6VwwDUrVK4nyk3MpDUt9wSw0D+ogRaPCPl3NT/d9gz5WkJugpBCe" +
    "WhUtOMZ3AkEA5Z1vcl/A4MWTEGDiBoSEkdkPI+vyh714ibHQ8p7ic8AMWhFmj5hY" +
    "r8Iti/vsJ83AiOeYdy+YK+7v/WiPrLfScwJAYEW3Rvq15cF0pNNNFD4+XAD5jfSV" +
    "7vSUQcPsM1eOIQXEcbxhy1WDVinUW1UjiCEqNNavF2IY5IPE8I88tBfHjQJAQHAm" +
    "TprArlPEKdyGkf2ulp+ruBEHR0DNCxdz5CLiukkzLjOj7Lh8axa7YYWZiLIdTk5w" +
    "Y0JgGjQ79YnaiEYuMQJAQyNXB4EvRoVuho4BVyQHiNKBDJ1YfyQT82u76L4Hf5Qb" +
    "cWydrGHOM6T+zj/hUQv5C96ItnXRv6z97Ktx+2pJlQ==" +
    "-----END RSA PRIVATE KEY-----";

var publicPEM = "-----BEGIN PUBLIC KEY-----" +
    "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDfaldZKOKdkfvfYiFgX/ZRHdQw" +
    "Nrb8U8imZ9gNOBXtrDu/hGxHEgyrZ9iMqoIcIhgxBzcKwKBAp4xu6yB3AOZiBwLI" +
    "73ajox/CpIzXE9yPevd5wQ+XHctIQazp0qrE9Py5Q5Ox7HB9rmKjSISKQ3A1JtEV" +
    "bl0bI0iMf4QCtl/FdQIDAQAB" +
    "-----END PUBLIC KEY-----";

var privateKey = forge.pki.privateKeyFromPem(privatePEM);
//var publicKey = pki.publicKeyFromPem(publicPEM);


// Takes the string and pulls off 117 chars at a time ( leaving 11 for padding )
// Since we are using 1024 bit keys ( 1024 / 8 - 11 )
// Each chunk is encrypted using 'RSAES-PKCS1-V1_5' and concated to the final encrypted message
// The string is then converted to hex for safe transport over http

function encryptString(str, pubKey){

    // encode utf8 first in case we are dealing with multibyte characters
    var encoded = forge.util.encodeUtf8(str);
    var buffer = forge.util.createBuffer(encoded, 'raw');
    var outBuffer = forge.util.createBuffer('', 'raw');

    while(buffer.length() > 0){
        var chunk = buffer.getBytes(117);
        var eChunk = pubKey.encrypt(chunk, 'RSAES-PKCS1-V1_5');
        outBuffer.putBytes(eChunk);
    }

    return forge.util.bytesToHex(outBuffer.bytes());
}

// Takes hex encrytped string and converts the hex to utf8 bytes
// Takes the encrypted bytes and pulls off 128 chars at a time ( 117 plus 11 for padding )
// Since we are using 1024 bit keys ( 1024 / 8 - 11 )
// Each chunk is decrypted using 'RSAES-PKCS1-V1_5' and concated to the final decrypted string
function decryptString(eStr, privKey){

    var bytes = forge.util.hexToBytes(eStr);
    var buffer = forge.util.createBuffer(bytes, 'raw');
    var outBuffer = forge.util.createBuffer('', 'raw');

    while(buffer.length() > 0){
        var chunk = buffer.getBytes(128);
        var dChunk = privKey.decrypt(chunk, 'RSAES-PKCS1-V1_5');
        outBuffer.putBytes(dChunk);
    }

    // decode the utf8 bytes in case their were multbyte chars and return
    return forge.util.decodeUtf8(outBuffer.bytes());

}


function Channel(channelId){

    logger.info("Creating channel", channelId);

    this.activeTimeout = null;
    this.id = channelId;    
    this.clientConnections = {};
    this.clients = {};
    this.hostConnection = null;
    this.hostToken = "deprecated";
    this.endpoint = "/api/v1/channels/"+this.id;
    this.server = new WebSocketServer({server: service.server, path: this.endpoint});


    // Check for host connection or connected clients every so often and shut down if no one is here
    this.activeTimeout = setTimeout(this.checkConnections.bind(this),30000);

    this.server.on('error', this.onServerError.bind(this));
    this.server.on('connection', this.onSocketConnection.bind(this));

}

util.inherits(Channel, EventEmitter);


Channel.prototype.checkConnections = function(){
    if(!this.hostConnection){
        this.shutDown();
    }
};

Channel.prototype.isMaxConnections = function(){

    return this.server.clients.length >= 50;

};

Channel.prototype.onServerError = function(err){

    logger.error('onServerError', err.message);

};


Channel.prototype.onSocketError = function(socket, err){

    logger.error('onSocketError', err.message);

};


Channel.prototype.onSocketConnection = function(socket){

    logger.verbose('API v1 : Channel client connecting...');

    /*
     *  Test if we are over the limit and disconnect the socket if so
     */
    if(this.isMaxConnections()){
        socket.close(4000,"Maximum number of connections");
        return;
    }


    /*
     *  Create client object with an id and connect time
     */

    var id = uuid.v1();
    var client = {id:id, connectTime:Date.now()};

    // TODO : Improve me
    // hate to store the id on the websocket but is the easiest way for now
    socket.id = id;


    /*
     *  Retrieve the connection attributes for the upgrade request query string
     */
    var connectionAttributes = url.parse(socket.upgradeReq.url, true).query;


    /*
     *  Copy all attributes to the client object except the ones with leading double underscores (__pem, __token)
     */

    client.attributes = {};
    for(var key in connectionAttributes){
        if(connectionAttributes.hasOwnProperty(key) && key.indexOf("__") !== 0){
            client.attributes[key] = connectionAttributes[key];
        }
    }


    /*
     *  Is this the host connecting?
     */
    if(this.hostToken === connectionAttributes.__token){
        this.hostConnection = socket;
        client.isHost = true;
        logger.info("host connected");
    }

    /*
     *  Assume this is client but host has not joined || The token was incorrect
     */
    else if(!this.hostConnection){
        socket.close();
        return;
    }

    /*
     *  This is a client
     */
    else{
        client.isHost = false;
        logger.info("client connected");
    }

    /*
     * Set the clients PEM (public key)
     */

    this.setSocketPublicKey(socket, connectionAttributes.__pem);


    /*
     *  Add the connection and info to the maps
     */
    this.clientConnections[id] = socket;
    this.clients[id] = client;


    /*
     *  Add our listeners to the socket
     */
    socket.on('error', this.onSocketError.bind(this, socket));
    socket.on('close', this.onSocketClose.bind(this, socket));
    socket.on('message', this.onSocketMessage.bind(this, socket));


    /*
     *  Notify the client he is connected providing the id and client list
     *  TODO : Make this more efficient : currently converting clients Map to Array
     */

    var clientList = _.map(this.clients, function(client){
        if(client){
            return client;
        }else{
            logger.debug("Found null client in list");
            return void 0;
        }

    });

    /*
     *  Notify the client they connected and give them the other clients
     */
    var msgConnect = {
        method: "ms.channel.onConnect",
        params: {
            clientId: client.id,
            clients: clientList
        }
    };

    this.send(msgConnect,socket);

    /*
     *  Notify everyone else the client connected
     */
    var msgClientConnect = {
        method: "ms.channel.onClientConnect",
        params: client
    };

    this.broadcast(msgClientConnect,socket);

};



Channel.prototype.onSocketMessage = function(socket, msg){

    logger.silly('APIv1 : Channel : client received message ' , msg);

    try{

        var command = JSON.parse(msg);

        switch(command.method){

            case "ms.channel.disconnectClient" :

                if(this.clientConnections[command.params.id]){
                    this.clientConnections[command.params.id].close();
                }
                break;

            case "ms.channel.sendMessage" :

                var targetId = command.params.to;
                var message = command.params.encrypted ? decryptString(command.params.message, privateKey) : command.params.message;

                if(command.params.encrypted){
                    logger.silly("Decrypted message body : ", message);
                }

                var msgCommand = {
                    method : "ms.channel.onClientMessage",
                    params : {
                        encrypted : command.params.encrypted,
                        message : message,
                        from    : socket.id
                    }
                };

                if(targetId === "all"){
                    this.broadcast(msgCommand);
                }else if(targetId === "broadcast"){
                    this.broadcast(msgCommand, socket);
                }else if(targetId === "host"){
                    this.send(msgCommand,this.hostConnection);
                }else if(typeof targetId === "string" && this.clientConnections[targetId]){
                    this.send(msgCommand,this.clientConnections[targetId]);
                }else if(_.isArray(targetId)){
                    targetId.forEach(function(cid){
                        if(this.clientConnections[cid]){
                            this.send(msgCommand,this.clientConnections[cid]);
                        }
                    }, this);
                }
                break;

        }

    }catch(e){
        // Fail silently
        logger.error("Invalid command sent", msg);
    }

};

Channel.prototype.onSocketClose = function(socket){

    logger.info("client is disconnecting");

    // If the host disconnected shut down the channel
    if(socket === this.hostConnection){

        logger.debug("disconnecting client was the host");
        this.shutDown();

    }
    // Else let everyone know about the client disconnecting
    else if(this.clientConnections && this.clients && socket){

        try{
            var command = {
                method: "ms.channel.onClientDisconnect",
                params: this.clients[socket.id]
            };
            this.broadcast(command,socket);
            delete this.clientConnections[socket.id];
            delete this.clients[socket.id];
        }catch(e){
            // Just a fail safe as there are times when multiple connections are closing at once
            logger.error("Issue sending clientDisconnect notification to client : ", e.message);
        }

    }

};

Channel.prototype.setSocketPublicKey = function(socket, encryptedPEM){
    /*
     *  Attempt to set the client / sockets public key
     */
    try{
        if(encryptedPEM){
            var pem = decryptString(encryptedPEM, privateKey);
            socket.publicKey = forge.pki.publicKeyFromPem(pem);
            logger.debug("Setting public key for socket",  pem);
        }else{
            logger.warn("Connection provided no PEM");
        }

    }catch(e){
        logger.error("Error parsing encryptedPEM ", e);
    }

};


Channel.prototype.send = function(msgCommand, socket){

    try{

        if(msgCommand.method === 'ms.channel.onClientMessage' &&  msgCommand.params.encrypted){

            // TODO : Review if this is the fastest way to deep clone
            var msgClone = JSON.parse(JSON.stringify(msgCommand));
            msgClone.params.message = encryptString(msgClone.params.message, socket.publicKey);
            socket.send(JSON.stringify(msgClone));

        }else{
            socket.send(JSON.stringify(msgCommand));
        }

        logger.silly('APIv1 : Channel sending message to  : ' + socket.id, msgCommand);

    }catch(e){

        logger.error('APIv1 Channel.send : failed : ', e.message);

    }

};


Channel.prototype.broadcast = function(msgCommand, excludedSocket){

    logger.silly('APIv1 : Channel : broadcasting ', msgCommand);

    if(excludedSocket){
        logger.silly('APIv1 : Channel : broadcast is excluding client ', excludedSocket.id);
    }

    this.server.clients.forEach(function(socket){
        if(socket !== excludedSocket){
            this.send(msgCommand, socket);
        }
    }, this);

};


Channel.prototype.shutDown = function(){

    logger.warn("Shutting down channel", this.id);

    // Clear the test interval
    clearTimeout(this.activeTimeout);

    delete this.activeTimeout;
    delete this.hostConnection;
    delete this.clientConnections;
    delete this.clients;

    if(this.server){
        this.server.close(4000);
        delete this.server;
    }

    this.emit("shutDown");

    this.removeAllListeners();

};


module.exports = Channel;