#!/usr/bin/env
# -*- coding: UTF-8 -*-
# vim: sw=4 ts=4 expandtab ai
#
# Copyright (c) 2013 Intel, Inc.
# License: GPLv2
# Author: Alexander Kanevskiy <alexander.kanevskiy@intel.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License, version 2,
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.

"""
Common code to run hooks from plugins
"""
import os
import glob
import sys
import types
import ConfigParser

from gerrithooks.misc import find_config, configure_logging, daemonize

from pkg_resources import iter_entry_points

def run_plugin_hooks(hook, parser):
    """
    Common code for every hook. Must provide parsed params and logger
    """

    # get parameters from command line
    params = parser(sys.argv[1:])
    if isinstance(params, types.TupleType):
        params, extra_params = params
    else:
        extra_params = None

    # set up logger
    logger = configure_logging(["gerrithooks","gerrithooks_debug"]\
                                                        [int(params.debug)])

    params_keys = sorted([ key for key in dir(params)
            if not (key.startswith("_") or key =="debug" or key == "comment") ])
    params_string = ", ".join(["%s: %s" % (key, getattr(params, key)) for key in
                                                            params_keys ])
    if not 'debug' in params or not params.debug:
        logger.debug("Daemonizing...")
        daemonize()

    logger.info("Event %s: %s", hook, params_string)

    logger.debug("looking for config")
    confname = "gerrithooks.conf"
    conf, confd = find_config(confname, subdir="gerrithooks")
    if not conf:
        logger.error("Log configuration file %s not found or not readable",
                        confname)
        return 1

    # read rules from configuration file
    config = ConfigParser.RawConfigParser()
    config.optionxform = str

    logger.debug("Reading main config from %s", conf)
    config.read(conf)

    if confd:
        conf = glob.glob(os.path.join(confd,"*.conf"))
        if conf:
            logger.debug("Reading additional configs from %s", conf)
            conf = config.read(conf)
            logger.debug("read from %s", conf)

    entry_point_group = "gerrithooks"

    for entry_point in iter_entry_points(group=entry_point_group, name=hook):
        logger.debug("Loading plugin %s from %s",
            entry_point.name, entry_point.module_name)
        plugin = entry_point.load()
        logger.debug("Executing plugin %s", entry_point.name)
        plugin(hook=hook, config=config, logger=logger,
                    params=params, extra_params=extra_params)

    logger.info("done")
    return 0

if __name__ == "__main__":
    raise SystemExit("This module does not supposed to be run directly!")
