import argparse
import json
import logging
import re
import sys
import requests

logging.basicConfig(
    format='%(name)s: %(levelname)s: %(message)s',
    level=logging.INFO,
)
logger = logging.getLogger('AHub-SubmitRequest-Script')


def init_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--host',
        help='AnalysisHub server host',
        default='https://----.com/',
    )
    parser.add_argument(
        '--token',
        help='AnalysisHub authentication token',
        required=True,
    )
    parser.add_argument(
        '--json',
        help='Sumbit Request information JSON file (etc.json)',
        required=True,
    )
    parser.add_argument(
        '--svace-id',
        help='SVACE analyzer policy ID',
        default=240,
        type=int,
    )
    parser.add_argument(
        '--coverity-id',
        help='Coverity analyzer policy ID',
        default=258,
        type=int,
    )
    parser.add_argument(
        '--defect-manager-id',
        help='AHub Defect Manager ID',
        default=8,
        type=int,
    )
    parser.add_argument(
        '--cov-connect-id',
        help='AHub Cov-Connect ID',
        default=7,
        type=int,
    )
    parser.add_argument(
        '--gbs-conf',
        help='GBS configuration',
        default='sr/tizen_7.0.conf',
    )
    parser.add_argument(
        '--gbs-profile',
        help='GBS profile',
        default='profile.tizen_7.0_standard',
    )
    parser.add_argument(
        '--review-system',
        help='AHub review system name',
        default='Public Gerrit',
    )

    return parser


def parse_project_commit_hash(project_string):
    projects = []
    commit_hashes = []
    for project in project_string.split(','):
        gerrit_project, commit_hash = project.split('@')
        projects.append(gerrit_project)
        commit_hashes.append(commit_hash[:8])
    return ','.join(projects), ','.join(commit_hashes)


class Ahub():
    def __init__(self, host, token):
        self.host = host
        self.headers = {'Authorization': 'Token {}'.format(token)}

    def get_analyzer_policy_id(self, analyzer_policy):
        api = '/v1/analyzer-policies'
        response = requests.get(self.host + api, headers=self.headers, verify=False,
                                params={'name': analyzer_policy})
        response.raise_for_status()
        return response.json()['results'][0]['id']

    def get_manager_id(self, manager):
        api = '/v1/managers'
        response = requests.get(self.host + api, headers=self.headers, verify=False,
                                params={'name': manager})
        response.raise_for_status()
        return response.json()['results'][0]['id']

    def get_review_system_id(self, review_system):
        api = '/v1/review-systems'
        response = requests.get(self.host + api, headers=self.headers, verify=False,
                                params={'name': review_system})
        response.raise_for_status()
        return response.json()['results'][0]['id']

    def run_analysis(self,
                     projects,
                     commit_hashes,
                     review_system_id,
                     gbs_conf,
                     gbs_profile,
                     analyzer_policy_id,
                     manager_id):
        api = '/v1/analyses/tizen-submit-request'
        data = {
            'clone_url': projects,
            'commit_hash': commit_hashes,
            'review_system_id': review_system_id,
            'gbs_conf': gbs_conf,
            'gbs_profile': gbs_profile,
            'analyzer_policy_id': analyzer_policy_id,
            'manager_id': manager_id,
        }
        response = requests.post(self.host + api, headers=self.headers, data=data, verify=False)
        response.raise_for_status()
        return response.json()['analysis_id']

    def create_tag(self, analysis_id, branch):
        api = '/v1/analysis-tags'
        data = {
            'analysis_id': analysis_id,
            'name': branch,
        }
        response = requests.post(self.host + api, headers=self.headers, data=data, verify=False)
        response.raise_for_status()
        return response.json()['id']


VALID_TAG_PATTERN = r'TIZEN_Tizen_Tizen-Unified.*'
OLD_TIZEN_TAG_PATTERN = r'TIZEN_Tizen-\d\.\d_Tizen-\d\.\d-Unified.*'


def is_valid_tag_pattern(tag):
    pattern = re.compile(VALID_TAG_PATTERN)
    old_pattern = re.compile(OLD_TIZEN_TAG_PATTERN)
    if bool(old_pattern.match(tag)):
        logger.warning('Old Tizen version is not supported')
        exit()
    if not bool(pattern.match(tag)):
        logger.error('Git tag has to match with pattern "{}"'.format(VALID_TAG_PATTERN))
        exit(1)


def main():
    logger.info('Start AHub Submit Request script')

    parser = init_parser()
    args = parser.parse_args()

    with open(args.json, 'r') as f:
        submit_request_info = json.load(f)

    projects, commit_hashes = parse_project_commit_hash(submit_request_info['packages'])
    if not projects:
        logger.error('There is no project to analyze.')
        return
    branch = submit_request_info['git_tag']
    is_valid_tag_pattern(branch)

    logger.info('Review System: %s', args.review_system)
    logger.info('Projects: %s', projects)
    logger.info('Commit Hash: %s', commit_hashes)
    logger.info('Branch(tag): %s', branch)
    logger.info('SVACE: %s', args.svace_id)
    logger.info('Coverity: %s', args.coverity_id)
    logger.info('Defect Manager: %s', args.defect_manager_id)
    logger.info('Cov-Connect: %s', args.cov_connect_id)

    ahub = Ahub(args.host, args.token)
    review_system_id = ahub.get_review_system_id(args.review_system)

    logger.info('Start trigger SVACE analysis')
    svace_analysis_id = ahub.run_analysis(projects,
                                          commit_hashes,
                                          review_system_id,
                                          args.gbs_conf,
                                          args.gbs_profile,
                                          args.svace_id,
                                          args.defect_manager_id)
    logger.info('SVACE analysis ID: %s', svace_analysis_id)
    ahub.create_tag(svace_analysis_id, branch)

    logger.info('Start trigger Coverity analysis')
    coverity_analysis_id = ahub.run_analysis(projects,
                                             commit_hashes,
                                             review_system_id,
                                             args.gbs_conf,
                                             args.gbs_profile,
                                             args.coverity_id,
                                             args.cov_connect_id)
    logger.info('Coverity analysis ID: %s', coverity_analysis_id)
    ahub.create_tag(coverity_analysis_id, branch)
    logger.info('Finish AHub Submit Request script')


if __name__ == "__main__":
    main()
