#!/usr/bin/python -tt
# vim: ai ts=4 sts=4 et sw=4
#
# Copyright (C) 2010, 2011, 2012, 2013, 2014 Intel, Inc.
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; version 2 of the License.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#

"""Unit tests for class BuildData"""

import os
import unittest
from collections import OrderedDict

from common.builddata import BuildData, BuildDataError


TEST_XML = """<?xml version="1.0" ?>
<build version="1.0">
  <id>test.id</id>
  <buildtargets>
    <buildtarget name="atom">
      <buildconf>
        <location href="3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8-build.conf"/>
        <checksum type="sh256">3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8</checksum>
      </buildconf>
      <archs>
        <arch>x86_64</arch>
        <arch>i586</arch>
      </archs>
      <repo type="binary">repos/atom/packages</repo>
      <repo type="debug">repos/atom/debug</repo>
      <repo type="source">repos/atom/sources</repo>
    </buildtarget>
    <buildtarget name="exynos">
      <buildconf>
        <location href="3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8-build.conf"/>
        <checksum type="sh256">3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8</checksum>
      </buildconf>
      <archs>
        <arch>armv7l</arch>
      </archs>
      <repo type="binary">repos/exynos/packages</repo>
      <repo type="debug">repos/exynos/debug</repo>
      <repo type="source">repos/exynos/sources</repo>
    </buildtarget>
  </buildtargets>
</build>
"""

class BuildDataTest(unittest.TestCase):
    '''Tests for BuildData functionality.'''

    def test_load(self):
        """Test loading of build data from xml string."""
        bdata = BuildData(build_id='test.id')
        bdata.load(TEST_XML)
        # pylint: disable=line-too-long
        # unsuitable line feed, line feed more unreadable
        self.assertEqual(bdata.targets, OrderedDict(
            [(u'atom',
              {"repos":
               [(u'binary', '', u'repos/atom/packages'),
                (u'debug', '', u'repos/atom/debug'),
                (u'source', '', u'repos/atom/sources')],
               'buildconf':
                 {'checksum': {'type': u'sh256',
                  'value': u'3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8'},
                  'location':  u'3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8-build.conf'},
               'name': u'atom', 'archs': [u'x86_64', u'i586']}),
             (u'exynos',
              {"repos":
               [(u'binary', '', u'repos/exynos/packages'),
                (u'debug', '', u'repos/exynos/debug'),
                (u'source', '', u'repos/exynos/sources')],
               'buildconf':
               {'checksum': {'type': u'sh256',
                             'value': u'3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8'},
                'location': u'3bd64bd5fa862d99dbc363ccb1557d137b5685bc3bfe9a86bcbf50767da5e2e8-build.conf'},
               'name': u'exynos', 'archs': [u'armv7l']})]))
        # pylint: enable=line-too-long

    def test_load_error(self):
        """Test rasing BuildDataError."""
        bdata = BuildData(1)
        self.assertRaises(BuildDataError, bdata.load, '<test/>')


    def test_add_target(self):
        """Test adding new target."""
        bdata = BuildData(1)
        target = {"name": "test_target",
                  "archs": ['i586'],
                  "buildconf": {
                      "checksum": {
                          "type": "md5",
                          "value": "45c5fb8bd2b9065bd7eb961cf3663b8c"
                       },
                      "location": 'build.conf'
                   },
                  "repos": [
                      ("binary", "arm7vl", "repos/exynos/packages"),
                      ("debug", None, "repos/exynos/debug"),
                      ("source", None, "repos/exynos/sources")]
                 }

        bdata.add_target(target)
        self.assertTrue(target["name"] in bdata.targets)


    def test_to_xml(self):
        """Test xml output."""
        bdata = BuildData(build_id='test.id')
        bdata.load(TEST_XML)
        self.assertEqual(bdata.to_xml(), TEST_XML)
        self.assertEqual(len(bdata.to_xml(hreadable=False)), 962)


    def test_save(self):
        """Test saving build data to file."""
        bdata = BuildData(build_id='test.id')
        bdata.load(TEST_XML)
        fname = 'test_save.tmp'
        bdata.save(fname)
        self.assertEqual(open(fname).read(), TEST_XML)
        os.unlink(fname)
