#!/usr/bin/python -tt
# vim: ai ts=4 sts=4 et sw=4
#
# Copyright (C) 2010, 2011, 2012, 2013, 2014 Intel, Inc.
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; version 2 of the License.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#

"""Unit tests for class ImageData"""

import os
import shutil
import unittest

from common.imagedata import ImageData, ImageDataError


TEST_XML = '''<?xml version="1.0" ?>
<image-configs>
  <config>
    <name>XX-00</name>
    <description>Image for device XX-00</description>
    <path>images/XX-00/</path>
    <ks>images/XX-00/xx-00.ks</ks>
    <buildtarget>arm</buildtarget>
    <arch>armv7l</arch>
  </config>
  <config>
    <name>YY-01</name>
    <description>Image for device YY-01</description>
    <path>images/YY-01/</path>
    <ks>images/YY-01/yy-01.ks</ks>
    <buildtarget>standard</buildtarget>
    <arch>ia32</arch>
  </config>
</image-configs>
'''

class ImageDataTest(unittest.TestCase):
    '''Tests for BuildData functionality.'''

    def test_load(self):
        """Test loading image data from xml."""
        idata = ImageData()
        idata.load(TEST_XML)
        self.assertEqual(dict(idata.images),
                         {'XX-00': {'name': 'XX-00',
                                    'description': 'Image for device XX-00',
                                    'path': 'images/XX-00/',
                                    'ks': 'images/XX-00/xx-00.ks',
                                    'buildtarget': 'arm',
                                    'arch': 'armv7l',
                                    },
                          'YY-01': {'name': 'YY-01',
                                    'description': 'Image for device YY-01',
                                    'path': 'images/YY-01/',
                                    'ks': 'images/YY-01/yy-01.ks',
                                    'buildtarget': 'standard',
                                    'arch': 'ia32'}})

    def test_load_error(self):
        """Test rasing ImageDataError."""
        idata = ImageData()
        self.assertRaises(ImageDataError, idata.load, '<test/>')

    def test_to_xml(self):
        """Test xml output."""
        idata = ImageData()
        idata.load(TEST_XML)
        self.assertEqual(len(idata.to_xml()), 509)
        self.assertEqual(len(idata.to_xml(hreadable=False)), 434)

    def test_save(self):
        """Test saving image data."""
        saved = ImageData()
        saved.load(TEST_XML)
        fname = 'test_save.tmp'
        saved.save(fname)
        loaded = ImageData()
        loaded.load(open(os.path.join(fname, 'images.xml')).read())
        self.assertEqual(saved.images, loaded.images)
        shutil.rmtree(fname)
