#!/usr/bin/env python
# vim: ai ts=4 sts=4 et sw=4
#
# Copyright (C) 2017 Samsung Electronics. Co,. Ltd.
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; version 2
#    of the License.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
"""
Interface of Tizen Studio
"""

import os
import sys
import shutil
import re
import stat
import ast
import subprocess
import glob
from datetime import datetime
from random import randint

sys.path.insert(1, os.path.join(sys.path[0], '..'))

from common.utils import wget_noproxy, list_files_in_url, tail
from common.buildtrigger import trigger_info, trigger_next
from common.git import Git, clone_gitproject
from common.gerrit import GerritEnv

class LocalError(Exception):
    """Local error exception."""
    pass

class TizenStudio(object):

    def __init__(self, vm_image, meta_script):

        self.vm_root = os.getenv('ABS_VM_IMAGE_ROOT')
        self.basedir = os.path.join(os.getenv('WORKSPACE'), os.getenv('JOB_NAME'))
        self.builddir = os.path.join(self.basedir, 'build')
        self.vm_image = os.path.join(self.vm_root, vm_image)
        assert os.path.isfile(self.vm_image)
        #if os.path.exists(self.basedir):
        #    shutil.rmtree(self.basedir)
        if not os.path.exists(self.basedir) or not os.path.exists(self.builddir):
            os.makedirs(self.builddir)

        # Fetch ABS cli wrapper from gerrit
        self.wrapper_path = self.fetch_resources(meta_script.split(','))
        print 'wrapper_path:%s' % self.wrapper_path

        self.build_root = '/home/build'
        self.sdk_path = os.path.join(self.build_root, 'tizen-sdk-cli')
        self.share_root = '/share/build'

    def get_template(self, mode=None):

        with open(os.path.join(self.wrapper_path, '%s.template' % mode), 'r') as rf:
            cmd_template = rf.read()
        return cmd_template

    def run_tizen_studio(self, cmd_to_run, vm_image=None, snapshot_on=True):

        with open(os.path.join(self.builddir, 'run'), 'w') as fcmdl:
            fcmdl.write('%s' % cmd_to_run)
        os.chmod(os.path.join(self.builddir, 'run'), 0o777)

        if vm_image is None:
            vm_image = self.vm_image

        ssh_connection = os.getenv('SSH_CONNECTION')
        try:
            lastip = int(ssh_connection.split(' ')[2].split('.')[3])
        except (IndexError, ValueError, AttributeError):
            print 'ssh_connection is %s, it is a incorrect format ,' \
                'random instead' % ssh_connection
            lastip = randint(0x00, 0xff)
        exeid = int(os.getenv('EXECUTOR_NUMBER', '1')) % 256
        processid = os.getpid() % 256
        buildid = int(os.getenv('BUILD_NUMBER', randint(0x00, 0xff))) % 256
        mac_address = '52:%02x:%02x:%02x:%02x:%02x' % \
            (lastip, exeid, processid, buildid, randint(0x00, 0xff))
        if snapshot_on:
            opt_snapshot = ",snapshot=on "
        else:
            opt_snapshot = ""
        cmd = 'qemu-system-x86_64 -machine accel=kvm:xen:tcg ' \
              '-name ubuntu -M pc -m %d -smp %d ' \
              '-drive file=%s%s '\
              '-virtfs local,id=test_dev,path=%s,security_model=mapped,mount_tag=share ' \
              '-net nic,macaddr=%s -net user -nographic' \
              % (int(os.getenv("ABS_VM_MEMORY")), int(os.getenv("ABS_VM_CPUS")), \
                vm_image, opt_snapshot, self.basedir, mac_address)
        print "Running kvm machine...\n %s" % (cmd)
    
        subprocess.call(cmd, stdout=sys.stdout,
                        stderr=sys.stderr, shell=True)
    
        #read testresult from file
        try:
            with open(os.path.join(self.basedir,'build','testresult')) as statusf:
                status = statusf.read().strip()
            print 'KVM returned [%s]' % (status)
            return int(status)
        except IOError, _err:
            raise LocalError('KVM Failed')

    def fetch_resources(self, rc_repo_branch=[]):

        gerrit_env = GerritEnv('PUBLIC_')

        rc_git_path, rc_git_branch = rc_repo_branch
        rc_dir = os.path.join(self.builddir, os.path.basename(rc_git_path))
        if not clone_gitproject(rc_git_path, rc_dir, \
                                gerrit_hostname=gerrit_env.hostname, \
                                gerrit_username=gerrit_env.username, \
                                gerrit_sshport=gerrit_env.sshport):
            print 'Error clone project %s' % rc_git_path 
        rc_git = Git(rc_dir)
        rc_git.checkout(rc_git_branch)
        return rc_git.path

    def update_rootstrap(self, rootstrap_src_url, rootstrap_version, profiles=[]):

        temp_image = os.path.join(self.vm_root, '.tmp.%s.v-%s' \
                     % (os.path.basename(self.vm_image), rootstrap_version))
        print 'Coping %s from %s...' % (temp_image, self.vm_image)
        sys.stdout.flush()
        shutil.copy(self.vm_image, temp_image)
        if not os.path.isfile(temp_image):
            os.remove(temp_image)
            print 'Temp file creation failed'
            return None
        os.chmod(temp_image, os.stat(temp_image).st_mode | stat.S_IWRITE)

        # Fetch sdk-tool from gerrit
        sdk_tool_path = self.fetch_resources(os.getenv('SDK_TOOL_GIT').split(','))
        print 'sdk_tool_path:%s' % sdk_tool_path

        update_cmd = self.get_template('update')
        update_cmd = update_cmd.replace( \
                     '__SDK_PATH__', self.sdk_path).replace( \
                     '__SHARE_ROOT__', self.share_root).replace( \
                     '__TOOL_PATH__', os.path.join(self.share_root, os.path.basename(sdk_tool_path))).replace( \
                     '__WRAPPER_PATH__', os.path.join(self.share_root, os.path.basename(self.wrapper_path))).replace( \
                     '__PROFILE__', ' '.join(profiles)).replace( \
                     '__PACKAGE_SERVER__', os.getenv('ABS_SDK_PACKAGE_SERVER')).replace( \
                     '__ROOTSTRAP_URL__', rootstrap_src_url)
        print 'UPDATE_CMD:\n%s' % update_cmd

        #### Running QEMU to launch Tizen Studio ####
        print '[ TizenStudio START ] %s' % (str(datetime.now()))
        sys.stdout.flush()
        ret = self.run_tizen_studio(update_cmd, vm_image=temp_image, snapshot_on=False)
        print '[ TizenStudio END ] %s' % (str(datetime.now()))

        if int(ret) != 0:
            os.remove(temp_image)
            print 'Updating failed, cause : %s' % ret 
            return None
        os.rename(temp_image, self.vm_image)

        return self.vm_image

    def gather_build_result(self):

        # Get installed rootstrap version
        built_version = None
        with open(os.path.join(self.builddir, 'rsver')) as rsverfile:
            rrrr = rsverfile.read()
            _built_version = rrrr.replace(' ', ', ').replace('\n', ' ').encode('utf8').split(',')[0]
            verregex = re.compile(r'(\d{8}_\d{1,2})')
            versearch = verregex.search(_built_version)
            if versearch is not None:
                if versearch.group() is not None:
                    built_version = versearch.group()
            print 'Installed RS version... %s' % built_version
        if built_version is None:
            print 'Not able detect installed Rootstrap version'

        tizen_studio_version = 'Tizen CLI 0.0.0'
        tizen_studio_distribution = 'default'
        with open(os.path.join(self.builddir, 'tizen_studio_version')) as tizenstudioverfile:
            tizen_studio_version = tizenstudioverfile.read().strip()
            print 'Installed Tizen Studio version... %s' % tizen_studio_version
        with open(os.path.join(self.builddir, 'tizen_studio_distribution')) as tizenstudiodistributionfile:
            tizen_studio_distribution = tizenstudiodistributionfile.read().strip()
            print 'Installed Tizen Studio Distribution... %s' % tizen_studio_distribution

        self.built_version = built_version
        self.tizen_studio_version = tizen_studio_version
        self.tizen_studio_distribution = tizen_studio_distribution

        # Get building log
        self.buildlog = ''
        mtime = lambda f: os.stat(os.path.join(self.builddir, f)).st_mtime
        for filename in list(sorted(os.listdir(self.builddir), key=mtime)):
            if re.match('build.*\.log', filename):
                print 'Log file [%s]' % filename
                onefile = tail(os.path.join(self.builddir, filename), c=4096)
                self.buildlog += "\n\n" + onefile[:onefile.rfind("Finished build-native")]
                if 'Exception ' in onefile:
                    break

        if self.build_result != 0:
            return {'built_version': built_version, \
                    'title': 'FAIL::CLI BUILD', \
                    'buildlog': self.buildlog, \
                    'tizen_studio_version': tizen_studio_version, \
                    'tizen_studio_distribution': tizen_studio_distribution}

        # Get packaging log
        if not glob.glob(os.path.join(self.builddir, '*.tpk')):
            for filename in os.listdir(self.builddir):
                if re.match('pkg.*\.log', filename):
                    self.buildlog = self.buildlog + tail(os.path.join(self.builddir, filename))
            return {'built_version': built_version, \
                    'title': 'FAIL::CLI PKG', \
                    'buildlog': self.buildlog, \
                    'tizen_studio_version': tizen_studio_version, \
                    'tizen_studio_distribution': tizen_studio_distribution}

        return {'built_version': built_version, \
                'title': None, \
                'buildlog': None, \
                'tizen_studio_version': tizen_studio_version, \
                'tizen_studio_distribution': tizen_studio_distribution}

    def build_app_source(self, package, profile=None, gitpath=None, build_mode='Release', \
                         parallel_jobs='', build_type='default', app_type='tpk'):

        PROFILE = profile

        if app_type == 'wgt':
            build_cmd = self.get_template('build-wgt')
        else:
            build_cmd = self.get_template('build')
        build_cmd = build_cmd.replace( \
                     '__SDK_PATH__', self.sdk_path).replace( \
                     '__SHARE_ROOT__', self.share_root).replace( \
                     '__BUILD_ROOT__', self.build_root).replace( \
                     '__PROFILE__', PROFILE).replace( \
                     '__PACKAGE__', package).replace( \
                     '__PARALLEL_JOBS__', parallel_jobs).replace( \
                     '__BUILD_MODE__', build_mode).replace( \
                     '__BUILD_TYPE__', build_type)
        print 'BUILD_CMD:\n%s' % build_cmd

        #### Running QEMU to launch Tizen Studio ####
        print '[ TizenStudio START ] %s' % (str(datetime.now()))
        sys.stdout.flush()
        ret = self.run_tizen_studio(build_cmd, snapshot_on=True)
        print '[ TizenStudio END ] %s' % (str(datetime.now()))

        self.build_result = int(ret)
        return self.build_result

if __name__ == '__main__':
    print 'This is not callable module'
    sys.exit(-1)

