#!/usr/bin/env python
#
# Copyright (c) 2014, 2015, 2016 Samsung Electronics.Co.Ltd.
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; version 2 of the License
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#

"""
This job is triggered by base verification process
"""

import os
import re
import sys
import base64
import json
import requests
import urlparse
import ast
from time import sleep
from datetime import datetime
from datetime import timedelta

import urllib2
import copy

from osc import conf, core
from common.buildservice import BuildService
from common.buildtrigger import trigger_info, trigger_next, remote_jenkins_build_job
from gitbuildsys.errors import ObsError
import xml.etree.ElementTree as ET
import xml.etree.cElementTree as ElementTree
from common.mapping import get_ref_map, get_sync_map_list
from common.gerrit import GerritEnv
from common.upload_service import UploadError

class LocalError(Exception):
    """Local error exception."""
    pass

class CreateProjectObs(object):
    """ create project to obs
        PreRelease Project fotmat is :
                home:prerelease:<origin_target_obs_project>:submit:basechecker """

    def __init__(self):

        self.args = trigger_info(os.getenv('TRIGGER_INFO'))

        obs = {
               "url": os.getenv("REF_TARGET_OBS_API_URL"),
               "user": os.getenv("REF_TARGET_OBS_API_USERNAME"),
               "pass": os.getenv("REF_TARGET_OBS_API_PASSWD")
              }
        obs_remote = {
                      "url": os.getenv("REF_REMOTE_OBS_API_URL"),
                      "user": os.getenv("REF_REMOTE_OBS_API_USERNAME"),
                      "pass": os.getenv("REF_REMOTE_OBS_API_PASSWD")
                     }

        self.obs = BuildService(obs["url"], obs["user"], obs["pass"])
        self.obs_remote = BuildService(obs_remote["url"], obs_remote["user"], obs_remote["pass"])

    def re_format_obs_description(self, target):

        obs_target = self.args["platform_to_verify"]
        ref_profile_prj = "{}:ref:{}".format(obs_target["obs_project_name"],
                                             obs_target["snapshot_name"].split("_")[-1])
        bpi = self.args["target_base_projects"][self.args["target_base_projects"].keys()[-1]]
        bpn = bpi["project"]
        info = {
                "obs_target_prj": obs_target["obs_project_name"],
                "request_sr": self.args["sr_list"],
                "ref_base_prj": "{}:ref:{}".format(bpn, bpi["ref"].split("_")[-1]),
                "sr_list": [x["sr_name"] for x in self.args["sr_list"]],
                "latest_base_prj": "{}:ref:{}".format(bpn, bpi["latest"].split("_")[-1]),
                "ref_profile_prj": ref_profile_prj,
                "git_tag": self.args["virtual_sr"],
                "base_target_prj": bpn,
                "build_step": self.args["build_step"],
                "submitter": self.args["submitter"],
                "virtual_sr": self.args["virtual_sr"],
                "obs_url": os.path.join(os.getenv("OBS_URL_EXTERNAL"), \
                    "project/show?project=%s" % target)
               }
        self.obs_description = info


    def post_project_create(self, project_name):

        bm_start_datetime = datetime.now()
        # get vars
        bm_stage = "Sync_SR_Submit_BaseCheck"
        bm_sync_git_tag = self.args.get('virtual_sr')
        bm_end_datetime = datetime.now()
        triggered_by = self.args.get('submitter')
        bm_data = {"bm_stage": bm_stage,
                   "commit_date": str(bm_end_datetime),
                   "commit_msg": "N/A",
                   "submit_date": str(bm_end_datetime),
                   "submit_msg": "N/A",
                   "submitter": "<%s>" % triggered_by,
                   "bm_sync_git_tag": bm_sync_git_tag,
                   "gerrit_project": "N/A",
                   "gerrit_newrev": "N/A",
                   "gerrit_account_name": "<%s>" % triggered_by,
                   "bm_start_datetime": str(bm_start_datetime),
                   "bm_end_datetime": str(bm_end_datetime),
                   "bm_src_project_lst": [project_name],
                   "git_tag_list": self.obs_description.get('sr_list')
                   }
        trigger_next("BUILD-MONITOR", bm_data)

    def generate_project_config(self, build, base_projects, profile_project, git_tag):
        """
        Change release name from project config in OBS
        Add the datetime into release name.
        Eg: 'Release: <CI_CNT>.<B_CNT>' ----> 'Release: 20141010.<CI_CNT>.<B_CNT>'
        """

        note = '#Add release name into prjconf\n'
        release_name = '{}\n{}.{}'.format(note, 'Release: %s' % (git_tag.split(':')[-1]), '<CI_CNT>.<B_CNT>')

        # get project config
        config = ''
        for base_project in base_projects:
            config = '{}\n{}'.format(config, build.get_project_config(base_project))
        config = '{}\n{}\n{}'.format(config, build.get_project_config(profile_project), release_name)

        # Add rpmbuild stage option
        if os.getenv('PRERELEASE_RPMBUILD_STAGE'):
            # Check if we've got required fields in TRIGGER_INFO
            if not os.getenv('PRERELEASE_RPMBUILD_STAGE') in ('ba', 'bb'):
                print 'Error: PRERELEASE_RPMBUILD_STAGE %s' % (os.getenv('PRERELEASE_RPMBUILD_STAGE'))
            else:
                rpmbuildstage = 'Rpmbuildstage: -%s' % (os.getenv('PRERELEASE_RPMBUILD_STAGE'))
                note_rpmbuildstage = '#Add RpmbuildStage option into prjconf'
                config = '{}\n{}\n{}'.format(config, note_rpmbuildstage, rpmbuildstage)

        # Add "CopyLinkedPackages: yes" for prerelease projects.
        if not re.search("CopyLinkedPackages:", config):
            config = config + "\nCopyLinkedPackages: yes\n"

        return config

    def run_create_staging_base_project_obs(self):

        print("[JOB STARTED : RUN CREATE STAGING BASE]")

        # Tizen:Unified
        platform_obs_project_name = self.args.get("platform_to_verify").get("obs_project_name")
        # Tizen:Unified:ref:20200729.2
        platform_ref_obs_project_name = "{}:ref:{}".format( \
            platform_obs_project_name, \
            self.args.get("platform_to_verify").get("snapshot_name").split("_")[-1])

        # submit/basechecker/20200730.144526
        sr_tag_to_create = self.args.get("virtual_sr")

        # home:prerelease:Tizen:Unified:submit:basechecker:20200730.144526
        obs_prerelease_project_to_create = "home:prerelease:{}:{}".format( \
            platform_obs_project_name, sr_tag_to_create.replace("/", ":"))

        # ["Tizen:Base:Tool:ref:20200724.1", "Tizen:Base:ref:20200723.1"]
        # Should be sorted
        tbps = self.args.get("target_base_projects")
        tool_project_included = False
        base_ref_obs_project_names = []
        for tbp in tbps:
            _target_name = "{}:ref:{}".format(tbp, tbps[tbp]["ref"].split("_")[-1])
            base_ref_obs_project_names.append(_target_name)
            if ':Base:Tool:ref:' in _target_name:
                tool_project_included = True
        if tool_project_included is False and len(base_ref_obs_project_names) > 0:
            print('Force include tool project...')
            base_ref_obs_project_names = [base_ref_obs_project_names[0].replace(':Base:ref:', ':Base:Tool:ref:')] \
                                         + base_ref_obs_project_names

        self.re_format_obs_description(obs_prerelease_project_to_create)

        meta = self.obs.get_meta(platform_ref_obs_project_name)
        config = self.generate_project_config( \
            self.obs, \
            base_ref_obs_project_names, \
            platform_ref_obs_project_name, \
            sr_tag_to_create.replace("/", ":"))

        self.create_basechecker_project( \
            self.obs, \
            obs_prerelease_project_to_create, \
            meta=meta, \
            config=config, \
            base_projects=base_ref_obs_project_names, \
            linked='all')

        request_submits = self.args.get("sr_list")

        self.copy_request_submits_package(self.obs, obs_prerelease_project_to_create, request_submits)

        for base_project in base_ref_obs_project_names:
            for idx, val in enumerate(request_submits):
                if val and request_submits[idx]:
                    self.create_link_aggregate_package(
                        self.obs, base_project, obs_prerelease_project_to_create, request_submits[idx])

        self.post_project_create(obs_prerelease_project_to_create)


    def create_basechecker_project(self, build, target, meta=None, config=None, base_projects=None, linked=''):
        """
        create project
        """

        if not build.exists(target):
            build.create_project(target, None, description=json.dumps(self.obs_description))
        else:
            print("Project {} already exists.".format(target))

        if meta:
            xml_meta = ElementTree.fromstringlist(meta)
            xml_meta.set('name', target)
            for person_element in xml_meta.findall('person'):
                xml_meta.remove(person_element)
            for link in xml_meta.findall('link'):
                xml_meta.remove(link)

            for base_project in base_projects:
                for repo_element in xml_meta.findall('repository'):
                    if linked:
                        repo_element.set('linkedbuild', linked)
                    for element in repo_element.findall('path'):
                        element.set('project', base_project)
                element = ElementTree.Element('link', {"project": "%s" % (base_project)})
                xml_meta.append(element)

            element = ElementTree.Element('person', {"userid": 'hyokeun', "role": "maintainer"})
            xml_meta.append(element)
            build.set_meta(ElementTree.tostring(xml_meta), target)

        build.set_project_config(target, config)
        build.set_description(json.dumps(self.obs_description), target)
        build.set_global_flag('build','enable', target)
        build.set_global_flag('publish','enable', target)

        print "\nTarget project %s created" % (target)


    def copy_request_submits_package(self, build, project_name, request_submits):
        sr_list = sorted(request_submits, key=lambda submit: (submit["project"]))
        for submit in sr_list:
            src_project = submit['project']
            for package in submit['packages'].split(','):
                build.create_copy_pac(
                    src_project, package, project_name, package)


    def add_multi_link_project(self, build, project, link_project):
        """ add link project """
        path = core.quote_plus(project)
        kind = 'prj'
        data = core.meta_exists(
            metatype=kind, path_args=path, template_args=None, create_new=False)
        if not data:
            return

        root = ElementTree.fromstring(''.join(data))
        if link_project not in [x.get('project') for x in root.findall('link')]:
            root.insert(2, ElementTree.Element('link', project=link_project))
            res_data = ElementTree.tostring(root)
            build.set_meta(res_data, project)

    def create_link_aggregate_package(self, build, src_project, dst_project, target_package):
        sourceinfo = build.get_sourceinfo_list(src_project)
        for package in sourceinfo:
            if sourceinfo[package]:
                link_prj, link_pkg = sourceinfo[package][-1].split('/')
                if link_prj == src_project and link_pkg == target_package:
                    build.create_link_pac(
                        dst_project, target_package, dst_project, package)

            if re.search("_aggregate", package) and \
               not re.search("base_tool_aggregate", package):
                print("Copypac aggregate package: %s/%s" %
                      (dst_project, package))
                build.create_copy_pac(
                    src_project, package, dst_project, package)
                aggregate_file_name = "_aggregate"
                build.get_source_file(
                    src_project, package, aggregate_file_name)
                content = ""
                with open(aggregate_file_name, 'r') as f:
                    content = f.read()

                if not re.search("qemu_aggregate", package) and \
                   not re.search("java-1_6_0-sun_aggregate", package) and \
                   not re.search("jpackage-utils_aggregate", package):
                    content_xml_root = ElementTree.fromstringlist(content)
                    for element in content_xml_root.findall('aggregate'):
                        #TODO: Set aggregate project only for matched project
                        if element.get('project') == src_project:
                            element.set('project', dst_project)
                    content = ElementTree.tostring(content_xml_root)
                    with open(aggregate_file_name, 'w') as f:
                        f.write(content)
                    commit_msg = "uploaded to copy pac %s/%s from %s" % \
                                 (dst_project, package, src_project)
                    try:
                        print src_project, dst_project, package
                        build.commit_files(dst_project, package,
                                           [(aggregate_file_name, True)], commit_msg)
                    except ObsError, error:
                        raise UploadError("Unable to upload _aggregate to %s: %s" % \
                             (dst_project, error))

                print("aggregate Copy Done.")

    def check_build_status(self, build, project):
        sleep(30)
        status = build.getbuildstatus(project)

        for repo in status.get('buildstatus'):
            if repo.get('code') != 'succeeded':
                return False

        return True

    def add_linked_project_profile(self):
        project = self.args.get("project") or self.args.get('sourceproject')
        info = json.loads(self.obs.get_description(project).replace("'", "\""))
        source_project = info.get('ref_profile_prj')
        event_type = self.args.get("event_type")

        checked_repos = self.check_build_status(self.obs, project)

        if not checked_repos:
            print("Build not succeeded.")
            trigger_next('#PRERELEASE#%s#%s' % (project, event_type), self.args)
            return

        if self.obs.exists(project):
            self.remove_repos_path_project(self.obs, project)
            self.add_multi_link_project(self.obs, project, source_project)
            info["build_step"] = "profile_check"
            self.obs.set_description(json.dumps(info), project)


    def remove_repos_path_project(self, build, project_name):

        path = core.quote_plus(project_name)
        kind = 'prj'
        data = core.meta_exists(
            metatype=kind, path_args=path, template_args=None, create_new=False)

        if not data:
            return

        root = ElementTree.fromstring(''.join(data))

        for repo in root.iter('repository'):
            for repo_path in repo.findall('path'):
                repo.remove(repo_path)

        res_data = ElementTree.tostring(root)
        build.set_meta(res_data, project_name)

    def create_project(self, build, project_name, info, base_ref_project, source_project=None):

        try:
            if not build.exists(project_name):
                try:
                    build.create_project(
                        project_name, src=source_project, description=json.dumps(info), linkto=base_ref_project, linkedbuild='all')
                except ObsError, error:
                    raise LocalError(
                        "Unable to create project %s: %s" % (project_name, error))

            else:
                print "\nTarget project %s exist" % (project_name)
                return False
        except ObsError, error:
            raise LocalError("Unable to create project %s: %s" %
                             (project_name, error))

    def main(self, action=None):
        """
        main
        """

        if action == 'base_check' or action is None:
            return self.run_create_staging_base_project_obs()
        elif action == 'profile_check':
            return self.add_linked_project_profile()


if __name__ == '__main__':
    trigger = CreateProjectObs()
    build_step = trigger.args.get('build_step', 'profile_check')
    sys.exit(trigger.main(build_step))
