#!/usr/bin/python
import os
import sys
import shutil
import json
import base64
import zipfile
import MySQLdb
import MySQLdb.cursors


UNZIP_DIR_NAME = "unzip"
FILE_COVERAGE_NAME = "merged_coverage.zip"


def unzip_coverages():
    # Prepare dir
    root_dir = os.getenv('WORKSPACE')
    zip_file = os.path.join(root_dir, FILE_COVERAGE_NAME)
    unzip_dir = os.path.join(root_dir, UNZIP_DIR_NAME)

    shutil.rmtree(unzip_dir, ignore_errors=True)
    if not os.path.exists(unzip_dir):
        os.makedirs(unzip_dir)

    with zipfile.ZipFile(zip_file, "r") as file:
        file.extractall(unzip_dir)

    return unzip_dir


if __name__ == "__main__":
    print("=====[ Fuzzing Coverage receiving started ]=====\n")

    data = os.getenv("FUZZ_DATA")

    print("FUZZ_DATA: {}\n".format(data))

    try:
        if data:
            data_json = json.loads(
                base64.urlsafe_b64decode(data.encode("ascii")).decode("ascii")
            )
        else:
            sys.exit("[Error] FUZZ_DATA is empty!\n")

        print("FUZZ_DATA encoded: {}\n".format(data_json))

        print("=====[ Fuzzing Coverage receiving finished ]=====\n")

        CONFIG = {"BASE": {}, "MYSQL": {}}

        RAW = {}
        MYSQL = {"query": {}, "fetchall": {}}
        RESULT = {}

        table = {
            "project": "fuzz_project",
            "job": "fuzz_job",
            "coverage": "fuzz_coverage",
        }

        # load config data
        CONFIG["MYSQL"]["host"] = os.getenv("BUILDMONITOR_IP")
        CONFIG["MYSQL"]["user"] = os.getenv("BUILDMONITOR_USER")
        CONFIG["MYSQL"]["password"] = os.getenv("BUILDMONITOR_PASS")
        CONFIG["MYSQL"]["database"] = os.getenv("BUILDMONITOR_NAME")

        # Connect MySQL
        MYSQL["connection"] = MySQLdb.connect(
            host=CONFIG["MYSQL"]["host"],
            user=CONFIG["MYSQL"]["user"],
            passwd=CONFIG["MYSQL"]["password"],
            db=CONFIG["MYSQL"]["database"],
            cursorclass=MySQLdb.cursors.DictCursor,
        )
        MYSQL["cursor"] = MYSQL["connection"].cursor()

        MYSQL["query"][
            "select_project"
        ] = "SELECT * FROM fuzz_project WHERE id = %(id)s"
        MYSQL["query"]["update_project"] = (
            "UPDATE fuzz_project SET "
            "name=%(name)s, "
            "url=%(url)s, "
            "branch=%(branch)s, "
            "platform=%(platform)s, "
            "active=%(active)s, "
            "ftb_name=%(ftb_name)s "
            "WHERE id = %(id)s"
        )
        MYSQL["query"][
            "update_project_actual_job"
        ] = "UPDATE fuzz_project SET actual_job=%(actual_job)s WHERE id = %(id)s"
        MYSQL["query"]["insert_project"] = (
            "INSERT INTO fuzz_project "
            "(id, name, url, branch, platform, active, ftb_name) "
            "VALUES (%(id)s, %(name)s, %(url)s, %(branch)s, %(platform)s, %(active)s, %(ftb_name)s)"
        )
        MYSQL["query"][
            "delete_project"
        ] = "DELETE FROM fuzz_project WHERE id = %(id)s"

        MYSQL["query"]["select_job"] = "SELECT * FROM fuzz_job WHERE id = %(id)s"
        MYSQL["query"]["update_job"] = (
            "UPDATE fuzz_job SET "
            "project_id=%(project_id)s, "
            "revision=%(revision)s, "
            "state=%(state)s, "
            "edge_cov=%(edge_cov)s, "
            "func_cov=%(func_cov)s, "
            "commit_date=%(commit_date)s, "
            "commit_url=%(commit_url)s, "
            "commit_message=%(commit_message)s "
            "WHERE id = %(id)s"
        )
        MYSQL["query"]["insert_job"] = (
            "INSERT INTO fuzz_job "
            "(id, project_id, revision, state, edge_cov, func_cov, commit_date, commit_url, commit_message) "
            "VALUES (%(id)s, %(project_id)s, %(revision)s, %(state)s, %(edge_cov)s, "
            "%(func_cov)s, %(commit_date)s, %(commit_url)s, %(commit_message)s)"
        )
        MYSQL["query"][
            "delete_job"
        ] = "DELETE FROM fuzz_job WHERE id = %(id)s"

        MYSQL["query"][
            "delete_coverage"
        ] = "DELETE FROM fuzz_coverage WHERE job_id = %(id)s"
        MYSQL["query"]["insert_coverage"] = (
            "INSERT INTO fuzz_coverage "
            "(id, type, parent_id, job_id, name, edge_cov, func_cov, full_path, html_file) "
            "VALUES (%(id)s, %(type)s, %(parent_id)s, %(job_id)s, %(name)s, "
            "%(edge_cov)s, %(func_cov)s, %(full_path)s, %(html_file)s)"
        )

        # Define an action
        action = data_json["action"] if "action" in data_json else "add"
        print("Action: {}".format(action))

        if action == "add":
            unzip_dir = unzip_coverages()

            # 1. Update project
            MYSQL["cursor"].execute(MYSQL["query"]["select_project"], data_json["project"])
            if MYSQL["cursor"].rowcount == 1:
                try:
                    MYSQL["cursor"].execute(
                        MYSQL["query"]["update_project"], data_json["project"]
                    )
                    MYSQL["connection"].commit()
                except MySQLdb.IntegrityError as e:
                    if "Duplicate" not in e.args[1]:
                        raise e
                print("1. Updated existing project:")
            elif MYSQL["cursor"].rowcount == 0:
                MYSQL["cursor"].execute(
                    MYSQL["query"]["insert_project"], data_json["project"]
                )
                MYSQL["connection"].commit()
                print("1. Inserted project:")
            print(data_json["project"])

            # 2. Update job
            MYSQL["cursor"].execute(MYSQL["query"]["select_job"], data_json["job"])
            if MYSQL["cursor"].rowcount == 1:
                try:
                    MYSQL["cursor"].execute(MYSQL["query"]["update_job"], data_json["job"])
                    MYSQL["connection"].commit()
                except MySQLdb.IntegrityError as e:
                    if "Duplicate" not in e.args[1]:
                        raise e
                print("2. Updated existing job:")
            elif MYSQL["cursor"].rowcount == 0:
                MYSQL["cursor"].execute(MYSQL["query"]["insert_job"], data_json["job"])
                MYSQL["connection"].commit()
                print("2. Inserted job:")
            print(data_json["job"])

            # 3. Update coverage
            print("3. Insert coverage:")
            MYSQL["cursor"].execute(MYSQL["query"]["delete_coverage"], data_json["job"])
            for c in data_json["coverage"]:
                if c["parent_id"] == "None":
                    c["parent_id"] = None
                c["html_file"] = None
                if c["type"] == "FILE":
                    html_file_path = unzip_dir + "/" + c["full_path"] + ".html"
                    try:
                        with open(html_file_path, "rb") as input_file:
                            c["html_file"] = input_file.read()
                    except Exception:
                        print("[ERROR] Failed to read '{}'".format(html_file_path))
                MYSQL["cursor"].execute(MYSQL["query"]["insert_coverage"], c)
                print("\tinserted {}".format(c["full_path"]))
            MYSQL["connection"].commit()

            # 4. Update actual_job
            actual_job = data_json["project"]["actual_job"]
            MYSQL["cursor"].execute(MYSQL["query"]["select_job"], {"id": actual_job})
            if MYSQL["cursor"].rowcount == 0:
                actual_job = data_json["job"]["id"]
            try:
                MYSQL["cursor"].execute(
                    MYSQL["query"]["update_project_actual_job"],
                    {"id": data_json["project"]["id"], "actual_job": actual_job},
                )
                MYSQL["connection"].commit()
                print("4. Actual job: {}".format(actual_job))
            except MySQLdb.IntegrityError as e:
                if "Duplicate" not in e.args[1]:
                    raise e
        elif action == "remove_project":
            # Remove project cascade
            MYSQL["cursor"].execute(MYSQL["query"]["delete_project"], data_json["project"])
            MYSQL["connection"].commit()
            print("Removed project: {}".format(data_json["project"]['id']))
        elif action == "remove_job":
            # Remove job cascade
            MYSQL["cursor"].execute(MYSQL["query"]["delete_job"], data_json["job"])
            MYSQL["connection"].commit()
            print("Removed job: {}".format(data_json["job"]['id']))
        else:
            raise Exception("Unknown action")

        MYSQL["cursor"].close()
        MYSQL["connection"].close()

    except Exception as e:
        print("[ERROR] {}".format(e))
