#!/usr/bin/env python
#
# Copyright (C) 2010, 2011, 2012, 2013, 2014 Intel, Inc.
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; version 2
#    of the License.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
#
"""
job_release_snapshot.py
This job is created to RE to release snapshot through jenkins UI, by trigger
it manually. The release target can be weekly or daily
"""

import os
import sys
import shutil
import tempfile
import subprocess
from time import sleep

from common.utils import sync
from common.utils import make_latest_link
from common.backenddb import BackendDB
from common.buildtrigger import trigger_next


def extract_release_notes(snapshot_dir):
    print 'Extract release notes in case of Tizen tools/services release with %s' %snapshot_dir
    release_note_list = []
    for root, dirs, files in os.walk(snapshot_dir):
        for file in files:
            if not file.endswith('.rpm'):
                continue
            # find rpm file which contains RELEASE_NOTES inside
            rpm_file = os.path.join(root, file)
            rpm_qpl_cmd = 'rpm -qpl ' + rpm_file
            output = subprocess.check_output(rpm_qpl_cmd,shell=True)
            for inner_file in output.splitlines():
                release_notes_file = ''
                if inner_file.endswith('RELEASE_NOTES'):
                    print 'RELEASE_NOTES exists in rpm_file: %s' %rpm_file
                    release_notes_file = inner_file
                    new_release_note_file_name = 'RELEASE_NOTES_' + inner_file.split('/')[-2].upper() + '.txt'
                    new_release_note_file = os.path.join(snapshot_dir, new_release_note_file_name)
                # copy RELEASE_NOTES_xxx file to snapshot_dir
                if release_notes_file and (new_release_note_file_name not in release_note_list):
                    release_note_list.append(new_release_note_file_name)
                    print 'orignal release_notes_file: %s, new_release_note_file: %s' \
                        %(release_notes_file, new_release_note_file)
                    tmpdir = tempfile.mkdtemp(prefix=snapshot_dir+'/')
                    curr_dir = os.getcwd()
                    os.chdir(tmpdir)
                    rpm_file_unextract_cmd = 'rpm2cpio < ' + rpm_file +'|cpio -idmv '
                    subprocess.call(rpm_file_unextract_cmd, stdout=sys.stdout,stderr=sys.stderr, shell=True)
                    os.chdir(curr_dir)
                    shutil.move(os.path.join(tmpdir, release_notes_file.lstrip('/')), new_release_note_file)
                    shutil.rmtree(tmpdir)


def main():
    """The main body"""

    path_repo_base = os.getenv('PATH_REPO_BASE')

    release_type = os.getenv('release_type')
    #release_profile = os.getenv('profile')
    obs_project = os.getenv('obs_project')
    release_id = os.getenv('release_id')
    version = os.getenv('version')
    latest_release = os.getenv('latest')

    # only supports the snapshot repo created by Jenkins
    backend = BackendDB(os.getenv('REDIS_HOST'),
                        int(os.getenv("REDIS_PORT")))

    obs_repo_map = backend.get_obs_repo_map()
    if obs_project not in obs_repo_map:
        print "Internal error, the repository corresponding obs project %s does not exist!" % obs_project
        return -1

    release_profile = obs_repo_map[obs_project]

    repos = backend.get_repos()
    if release_profile not in repos:
        print "Internal error, the profile %s doesn't exists" % release_profile
        return -1

    profile_info = repos[release_profile]

    # figure out the snapshot dir
    path_repo_snapshot = os.path.join(path_repo_base,
                                      profile_info['SnapshotDir'],
                                      release_id)

    # make sure the source dir is exists

    if not os.path.exists(path_repo_snapshot):
        path_repo_snapshot = path_repo_snapshot.replace('snapshots','releases/daily')
        if not os.path.exists(path_repo_snapshot):
            print("The snapshot %s of %s doesn't exists on download server"
                  %(release_id, release_profile))
            return -1

    # figure out the release destination dir
    release_dir = "releases/%s" % release_type

    path_release_snapshot = os.path.join(path_repo_base,
                                         profile_info['SnapshotDir'],
                                         release_id)\
                                         .replace('snapshots', release_dir)
    # exists, it will be replaced
    if os.path.exists(path_release_snapshot):
        print "The repo %s already exists, deleted and sync again" % release_id
        shutil.rmtree(path_release_snapshot)

    # sync snapshot to release
    sync(path_repo_snapshot, path_release_snapshot, remove=False, hardlinks=True)

    # create release notes in case of milestone release of tizen-devbase-tools/services 
    if release_id.find('tizen-devbase-') != -1 \
        and release_type == 'milestone':
        extract_release_notes(path_release_snapshot)

    # find the real latest repo by timestamp
    repo_list = os.listdir(os.path.join(path_release_snapshot, "../"))
    repo_list.sort()

    path_latest_release_snapshot = os.path.join(path_release_snapshot, "../",
                                           repo_list[-1])

    # update the latest link
    print 'path_latest_release_snapshot:%s' % path_latest_release_snapshot
    for x in [1, 2, 3]:
        try:
            make_latest_link(path_latest_release_snapshot)
        except Exception as err:
            sleep(1)
        break

    if latest_release and 'snapshots' in profile_info['SnapshotDir']:
        base_path = profile_info['SnapshotDir'].split('snapshots')
        path_latest_release = os.path.join(path_repo_base, base_path[0], latest_release)
        if os.path.lexists(path_latest_release):
            print(path_release_snapshot, "->", path_latest_release)
            if os.path.islink(path_latest_release):
                os.unlink(path_latest_release)
            else:
                shutil.rmtree(path_latest_release)
            os.symlink(path_release_snapshot, path_latest_release)

    # update reference project information with release id
    if os.getenv("REF_MAP_UPDATE_ENABLED", "0") != "0" \
        and os.getenv("REF_MAPPING_PRJ") is not None \
        and release_type == 'daily':
        trigger_next("ref-snapshot-info-update", \
                        {'profile':release_profile, \
                         'base_project':profile_info['Project'], \
                         'release_id':release_id})

    trigger_next("BUILD-MONITOR", \
            {'bm_stage':'RELEASE_SNAPSHOT',
                 'snapshot_name':release_id,
                 'release_type':release_type,
                 'version':version})

    # TRIGGER NEXT RSYNC-DOWNLOAD (MIRROR)
    if os.getenv("RSYNC_DOWNLOAD_ENABLED", "0") != "0":
        trigger_next('RSYNC_DOWNLOAD', {'repo_path': path_release_snapshot,
                                        'project': obs_project})

if __name__ == '__main__':

    sys.exit(main())
