#!/usr/bin/env python
# vim: ai ts=4 sts=4 et sw=4
#
# Copyright (C) 2016 Samsung
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; version 2
#    of the License.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
"""
This code is called by repa webpage
"""

import os
import re
import sys
import shutil

from common.buildservice import BuildService
from common.buildtrigger import trigger_info, trigger_next
from common import runner
from datetime import datetime
from common.buildmonitor_extention import BuildMonitorExtention

# set default char-set endcoding to utf-8
reload(sys)
sys.setdefaultencoding('utf-8') # pylint: disable-msg=E1101

class LocalError(Exception):
    """Local error exception."""
    pass

#-------------------------------------------------------------------------------
# global variables
status_str = ""
error_string = ""
obs_api = os.getenv("OBS_API_URL")
obs_user = os.getenv("OBS_API_USERNAME")
obs_passwd = os.getenv("OBS_API_PASSWD")

OBS_PROJECT_PREFIX = "home:prerelease:"
#-------------------------------------------------------------------------------

def tag_time_format(tag_info):
    _YYYYmmdd = tag_info.split('.')[0]
    _HHMMSS = tag_info.split('.')[-1]

    _YYYY = int(_YYYYmmdd) / 10000
    _mm = int(_YYYYmmdd) / 100 % 100
    _dd = int(_YYYYmmdd) % 100
    _HH = int(_HHMMSS) / 10000
    _MM = int(_HHMMSS) / 100 % 100
    _SS = int(_HHMMSS) % 100

    if _mm > 12: _mm = 12
    if _dd > 31: _dd = 31
    if _HH > 23: _HH = 23
    if _MM > 59: _MM = 59
    if _SS > 59: _SS = 59

    return '%04d%02d%02d.%02d%02d%02d' % (_YYYY, _mm, _dd, _HH, _MM, _SS)

def get_projects(build, regexp=''):
    """List projects with attributes."""
    projects = list(build.get_project_list())

    for proj in projects:
        if regexp and re.match(regexp, proj):
            yield proj

def get_tag_from_project_name(project):
    """ Get tag from OBS project name """
    for tag in (':submit:', ':submitgroup:'):
        splitted = project.split(tag)
        if len(splitted) > 1:
            return ('submit/%s' %(splitted[1].replace(':','/')))

def auto_reject_submission_list(build, target, purge_day=14, processes=20):
    """List submissions and groups."""
    # submissions
    purge_sr_tag = []

    now_datetime = datetime.now()

    for project in get_projects(build, '^%s%s:submit' % (OBS_PROJECT_PREFIX, target)):
        if ':submitgroup:' in project:
            continue

        tag = get_tag_from_project_name(project)
        # check for day
        newtime = tag_time_format(project.split(':')[-1])
        create_datetime = datetime.strptime(newtime, "%Y%m%d.%H%M%S")
        #print create_datetime
        post_day = divmod((now_datetime - create_datetime).total_seconds(), 3600 * 24)[0]
        if int(post_day) > int(purge_day):
            purge_sr_tag.append(tag)
    
    return purge_sr_tag

def trigger_next_to_repa(project, tag, obs_user, decision, comment, count):
    """ trigger next to repa """

    data = ""
    data += 'OBS_PROJECT=%s\n' %(project)
    data += 'SR_TAG=%s\n' %(tag)
    data += 'REPA_USER=%s\n' %(obs_user)
    data += 'REPA_DECISION=%s\n' %(decision)
    data += 'REPA_COMMENT=%s\n' %(comment)

    with open('REPA_%d.env' % (count), 'w') as info_f:
        info_f.write(data)
        print data

def main():
    """Script entry point.
    """

    print '---[JOB STARTED: repa auto reject]-------------------------'

    build = BuildService(obs_api, obs_user, obs_passwd)
    bm_ext = BuildMonitorExtention()

    count = 0
    repa_decision = 'declined'

    bm_list = bm_ext.get_auto_reject_projects()
    print bm_list
    for project in bm_list:
        num_of_days = bm_list[project]
        for sr_tag in auto_reject_submission_list(build, project, purge_day=num_of_days):
            print 'sr_tag : %s' %(sr_tag)
            repa_comment = "Auto Rejected Submission " + sr_tag
            bm_ext.update_sr_accept_log( project, sr_tag, obs_user, repa_comment, repa_decision)
            trigger_next_to_repa(project, sr_tag, obs_user, repa_decision, repa_comment, count)
            count += count + 1

    return 0

if __name__ == '__main__':
    try:
        exit_status = main()
    except Exception as err:
        print err
        status_str = "Failed"
        error_string = str(err)
        exit_status = 1

sys.exit(exit_status)
