#!/usr/bin/env python
#
# This file is part of REPA: Release Engineering Process Assistant.
#
# Copyright (C) 2013 Intel Corporation
#
# REPA is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA 02110-1301, USA.

"""
REPA: Release Engineering Process Assistant.

Copyright (C) Intel Corporation 2013
Licence: GPL version 2
Author: Ed Bartosh <eduard.bartosh@intel.com>

List module.
Get list of submissions.
"""

import sys
import json
import re

from repa.common import (OBS_PROJECT_PREFIX, Colorizer,
                         get_download_url, get_obs_url, get_prerelease)
from repa.obs import OBS
from repa.main import sub_main

def get_status(meta, colorizer, build_results=None, ignore='',
               obs=None, showtime=False):
    """Get overall status by analyzing package and image build status."""
    if build_results:
        codes = set()
        pkgstatus = {}
        for (repo, arch), target in build_results.items():
            codes.add(target.get('code'))
            codes.add(target.get('state'))
            for pkginfo in target['packages']:
                # ignore packages if they match ignore regexp
                if not (ignore and re.match(ignore, "%s/%s/%s" % \
                                           (repo, arch, pkginfo[0]))):
                    codes.add(pkginfo[1])

        statuses = [('broken', ('red', 'broken source')),
                    ('unresolvable', ('red', 'unresolvable packages')),
                    ('failed', ('red', 'package build failed')),
                    ('building', ('blue', 'package building')),
                    ('blocked', ('blue', 'blocked'))]
        for pkgstatus, (color, status) in statuses:
            if pkgstatus in codes:
                return getattr(colorizer, color)(status)

    if 'images' in meta:
        for img in meta['images']:
            statuses = [('failed', ('red', 'image build failed')),
                        ('building', ('blue', 'image building'))]
            for imgstatus, (color, status) in statuses:
                if img['status'] == imgstatus:
                    return getattr(colorizer, color)(status)
    else:
        return colorizer.blue('waiting for images')

    status = 'ready'
    if obs and showtime:
        # Add build time to the status
        project = get_prerelease(meta['git_tag'], meta['obs_target_prj'])
        btime = 0
        for (repo, arch), target in build_results.items():
            btime = max(btime, obs.get_build_time(str(project), str(repo),
                                                  str(arch)))
        minutes, seconds = divmod(btime, 60)
        hours, minutes = divmod(minutes, 60)
        status += " %02d:%02d:%02d" % (hours, minutes, seconds)

    return colorizer.green(status)


def show_urls(meta):
    """Print OBS and download urls."""
    download_url = get_download_url(meta)
    if download_url:
        print('    download url: ', download_url)
    obs_url = get_obs_url(meta)
    if obs_url:
        print('    obs url: ', obs_url)
    print()

def get_sr(obs, project, package, tag, status):
    """Check if tag is in comments of accepted/declined SRs."""
    for sreq in obs.get_srs(project, status, str(package)):
        descr = sreq[-1]
        if descr and 'Tag:' in descr:
            if tag == descr.split('Tag: ')[-1].split()[0]:
                return sreq[0]

def list_submissions(obs, target, processes, base, is_colorize=False,
                     showurls=False, ignore='', showtime=False, refprj=False):
    """List submissions and groups."""
    colorizer = Colorizer(is_colorize)
    # submissions
    groups = []
    for project, desc, build_results in \
                obs.get_projects('^%s%s:submit' % (OBS_PROJECT_PREFIX, target),
                                 processes):
        meta = json.loads(desc)
        if ':submitgroup:' in project:
            groups.append(meta)
            continue
        projects = [project.split('/')[-1] for project in meta['projects']]
        if base:
            tag = str(meta['git_tag'])
            base_status = colorizer.white('')
            prerelease = get_prerelease(tag, base)
            if obs.exists(prerelease):
                base_meta = json.loads(obs.get_descr(prerelease))
                base_build_results = obs.get_build_results(prerelease)
                base_status = get_status(base_meta, colorizer,
                                         base_build_results, ignore)
            else:
                asr = get_sr(obs, base, projects[0], tag, 'accepted')
                if asr:
                    base_status = colorizer.green('accepted. SR %s' % asr)
                else:
                    rsr = get_sr(obs, base, projects[0], tag, 'revoked')
                    if rsr:
                        base_status = colorizer.red('rejected. SR %s' % rsr)
            if refprj:
                print('%-37s %-37s %-37s %-37s %s' % \
                      (meta['git_tag'],
                       meta['ref_obs_target_prj'],
                       get_status(meta, colorizer, build_results,
                                  ignore, obs, showtime),
                       base_status, ','.join(projects)))
            else:
                print('%-37s %-37s %-37s %s' % \
                      (meta['git_tag'],
                       get_status(meta, colorizer, build_results,
                                  ignore, obs, showtime),
                       base_status, ','.join(projects)))
        else:
            if refprj:
                print('%-37s %-37s %-37s %s' % \
                      (meta['git_tag'],
                       meta['ref_obs_target_prj'],
                       get_status(meta, colorizer, build_results,
                                  ignore, obs, showtime),
                       ','.join(projects)))
            else:
                 print('%-37s %-37s %s' % \
                      (meta['git_tag'],
                       get_status(meta, colorizer, build_results,
                                  ignore, obs, showtime),
                       ','.join(projects)))
        if showurls:
            show_urls(meta)

    # groups
    if groups:
        print()
        for meta in groups:
            print('%-37s %-37s' % (meta['name'], get_status(meta, colorizer)))
            if showurls:
                show_urls(meta)


class List(object):
    """Subcommand: List submissions."""

    name = 'list'
    description = 'List submissions'
    help = description

    @staticmethod
    def add_arguments(parser, config):
        """
        Add arguments to the parser. Called from [sub_]main.
        Set defaults for arguments from config.
        """
        parser.add_argument('--processes', type=int,
                            help='amount of parallel processes to use',
                            default=config.get('processes'))
        parser.add_argument('--showurls', action='store_true',
                            help='show OBS and download urls',
                            default=config.get('showurls', '').lower() == 'on')
        parser.add_argument('--ignore', default=config.get('ignore', ''),
                            help='ignore package failures by regexp')
        parser.add_argument('--base', default=config.get('base', ''),
                            help='Show submission status in base project')
        parser.add_argument('--showtime', action='store_true',
                            help='show build time',
                            default=config.get('showtime', '').lower() == 'on')
        parser.add_argument('--refprj', action='store_true',
                            help='Show submission reference project in project',
                            default=config.get('refprj', '').lower() == 'on')

    @staticmethod
    def run(argv):
        """Command line entry point. Called from [sub_]main"""
        obs = OBS(argv.apiurl, argv.apiuser, argv.apipasswd)
        return list_submissions(obs, argv.project, argv.processes, argv.base,
                                argv.colorize, argv.showurls, argv.ignore,
                                argv.showtime, argv.refprj)


if __name__ == '__main__':
    sys.exit(sub_main(sys.argv[1:], List()))
