#!/bin/bash

pname="${0##*/}"
args=("$@")
cur_dir="$(pwd)"

# file names:
decompression_code="decompression_code"
piggy_gz_piggy_trailer="piggy.gz+piggy_trailer"
piggy="piggy"
piggy_gz="piggy.gz"
padding_piggy="padding_piggy"
piggy_trailer="piggy_trailer"
padding3="padding3"
sizes="sizes"

# We dup2 stderr to 3 so an error path is always available (even
# during commands where stderr is redirected to /dev/null).  If option
# -v is set, we dup2 sterr to 9 also so commands (and some of their
# results if redirected to &9) are printed also.
exec 9>/dev/null                # kill diagnostic ouput (will be >&2 if -v)
exec 3>&2                       # an always open error channel

#
########### Start of functions
#

# Emit an error message and abort
fatal(){
    # Syntax: fatal <string ...>
    # Output error message, then abort
    echo >&3
    echo >&3 "$pname: $*"
    kill $$
    exit 1
}

# Execute a command, displaying the command if -v:
cmd(){
    # Syntax: cmd <command> <args...>
    # Execute <command>, echo command line if -v
    echo >>"$workspace/log_file" "$*"
    "$@"
}

# Execute a required command, displaying the command if -v, abort on
# error:
rqd(){
    # Syntax: cmd <command> <args...>
    # Execute <command>, echo commandline if -v, abort on error
    cmd "$@" || fatal "$* failed."
}

checkNUL(){
    # Syntax: checkNUL file offset
    # Returns true (0) if byte there is 0x0.
    [ "$(rqd 2>>"$workspace/log_file" "$workspace/dd" if="$1" skip=$2 bs=1 count=1)" = $'\0' ]
}

gunzipWithTrailer(){
    # Syntax gunzipWithTrailer <file> <gzip name, sans .gz> <padding> <trailer>
    #
    # <file>: the input file
    # <gzip name, sans .gz>, <padding>, <trailer>:
    #   The output files.  For the gzipped part, both the
    #   compressed and the uncompressed output is generated, so we have
    #   4 output files.
    local file="$1"
    local gz_result="$2.gz"
    local result="$2"
    local padding="$3"
    local trailer="$4"
    local tmpfile="/tmp/gunzipWithTrailer.$$.gz"
	local original_size=$("$workspace/stat" -c %s "$unpacked/$file") 2>>"$workspace/log_file"
	echo "Original size is $original_size" >> "$workspace/log_file"
    local d=$(( (original_size+1) / 2))
    local direction fini at_min=0
    local results_at_min=()
    local size=$d
    local at_min=
	rm -rf /tmp/test_file
    echo "Separating gzipped part from trailer in "$unpacked/$file"" >> "$workspace/log_file"
    echo -n "Trying size: $size"	>> "$workspace/log_file"
    while :; do
        rqd "$workspace/dd" if="$unpacked/$file" of="$tmpfile" bs=$size count=1 2>>"$workspace/log_file"
        cmd "$workspace/gzip" >/tmp/test_file 2>>"$workspace/log_file" -d -c "$tmpfile"
        res=$?
		echo "result for gunzip is $res" >>"$workspace/log_file"
        if [ "$d" -eq 1 ]; then
            : $((at_min++))
            results_at_min[$size]=1
            [ "$at_min" -gt 3 ] && break
        fi
        d=$(((d+1)/2))
        case $res in
                # 1: too small
            1)  echo "In case 1" >> "$workspace/log_file"
				size=$((size+d)); direction="↑";;
                # 2: trailing garbage
            2) 	echo "In case 2" >> "$workspace/log_file"
				size=$((size-d)); direction="↓";;
                # OK
            0) 	echo "Breaking" >> "$workspace/log_file"
				break;;
            *) 	echo "In case *" >> "$workspace/log_file"
				fatal "gunzip returned $res while checking "$unpacked/$file"";;
        esac
        echo -n "  $size" >> "$workspace/log_file"
    done
    if [ "$at_min" -gt 3 ]; then
        echo -e "\ngunzip result is oscillating between 'too small' and 'too large' at size: ${!results_at_min[*]}"	>> "$workspace/log_file"
        echo -n "Trying lower nearby values:  "	>> "$workspace/log_file"
        fini=
        for ((d=1; d < 30; d++)); do
            : $((size--))
            echo -n "  $size" >> "$workspace/log_file"
            rqd "$workspace/dd" if="$unpacked/$file" of="$tmpfile" bs=$size count=1 2>/dev/null
            if cmd "$workspace/gzip" >/dev/null 2>&1 -d -c  "$tmpfile"; then
                echo -n " - OK"	>> "$workspace/log_file"
                fini=1
                break
            fi
        done
        [ -z "$fini" ] && fatal 'oscillating gunzip result, giving up.'
    fi
    # We've found the end of the gzipped part.  This is not the real
    # end since gzip allows for some trailing padding to be appended
    # before it barfs.  First, go back until we find a non-null
    # character:
    echo -ne "\npadding check (may take some time): " >> "$workspace/log_file"
    real_end=$((size-1))
    while checkNUL "$unpacked/$file" $real_end; do
        : $((real_end--))
    done
	echo "Found real end at $real_end" >> "$workspace/log_file"
    # Second, try if gunzip still succeeds.  If not, add trailing
    # null(s) until it succeeds:
    while :; do
        rqd "$workspace/dd" if="$unpacked/$file" of="$tmpfile" bs=$real_end count=1 2>>"$workspace/log_file"
        "$workspace/gzip" >/tmp/test_file2 2>>"$workspace/log_file" -d -c "$tmpfile"
        case $? in
            # 1: too small
            1) 	echo "In case 1" >> "$workspace/log_file"
				: $((real_end++));;
            *) 	echo "Case other $?" >> "$workspace/log_file"
				break;;
        esac
    done
	echo "Done with add trailing null(s) until it succeeds" >> "$workspace/log_file"
    real_next_start=$size
    # Now, skip NULs forward until we reach a non-null byte.  This is
    # considered as being the start of the next part.
    while checkNUL "$unpacked/$file" $real_next_start; do
        : $((real_next_start++))
    done
    echo $((real_next_start - real_end))	>> "$workspace/log_file"
    echo >> "$workspace/log_file"
    rm "$tmpfile"
    # Using the numbers we got so far, create the output files which
    # reflect the parts we've found so far:
    rqd "$workspace/dd" if="$unpacked/$file" of="$unpacked/$gz_result" bs=$real_end count=1
    rqd "$workspace/dd" if="$unpacked/$file" of="$unpacked/$padding" skip=$real_end bs=1 count=$((real_next_start - real_end))
    rqd "$workspace/dd" if="$unpacked/$file" of="$unpacked/$trailer" bs=$real_next_start skip=1
    rqd "$workspace/gzip" -c -d "$unpacked/$gz_result" > "$unpacked/$result"
}

unpack()(
    [ -d "$unpacked" ] && echo "\
Warning: there is already an unpacking directory.  If you have files added on
your own there, the  repacking result may not reflect the result of the
current unpacking process."
    rqd mkdir -p "$unpacked"
    rqd cd "$unpacked"
    sizes="$unpacked/sizes"
    echo "# Unpacking sizes" > "$sizes"
    log_file="$unpacked/log_file"
    #piggy_start=$1
    if [ -z "$piggy_start" ]; then
		fatal "Can't find a gzip header in file '$zImage'" >> "$workspace/log_file"
        fatal "Can't find a gzip header in file '$zImage'"
	else
		echo "start is $piggy_start" >> "$sizes"
    fi

    rqd "$workspace/dd" if="$zImage" bs="$piggy_start" count=1 of="$unpacked/$decompression_code"
    rqd "$workspace/dd" if="$zImage" bs="$piggy_start" skip=1 of="$piggy_gz_piggy_trailer"

    gunzipWithTrailer  "$piggy_gz_piggy_trailer" \
        "$piggy" "$padding_piggy" "$piggy_trailer"

    echo
    echo "Success."
)

#### start of main program
while getopts xv12345sgrpuhtz-: argv; do
    case $argv in
        p|z|1|2|3|4|5|t|r|g) eval opt_$argv=1;;
        u)  opt_u=1
			workspace=$2
		    zImage="$2/$3"
			piggy_start=$4
			unpacked="${zImage}_unpacked"
			packing="${zImage}_packing";;
        -) if [ "$OPTARG" = "version" ]; then
              echo "$pname $version"
              exit 0
           else
              echo "Wrong Usage, use -u to unpack"
           fi;;
        h|-) echo "Wrong Usage, use -u to unpack";;
        *) fatal "Illegal option";;
    esac
done
if [ -n "$opt_u" ]; then
    [ -f "$zImage" ] || fatal "file '$zImage': not found"
    unpack
fi
if [ -n "$opt_p" ]; then
	work_dir=$2
	tgt_file=$3
	cmd_dir=$4
	rqd cd "$work_dir"
	#remove all links before proceeding with zip processing
	"$cmd_dir/find" . -type l  -exec rm {} \;
	"$cmd_dir/find" . -exec touch -t 200011111111.11 {} \;
	"$cmd_dir/find" . -exec chmod 0755 {} \;
	"$cmd_dir/zip" -ryX "$tgt_file" *
fi
if [ -z "$opt_u$opt_p" ]; then
    echo >&2 "$pname: Need -u or -p option to work"
    echo >&2 "$pname: Type '$pname --help' for usage info."
    exit 1
fi

exit
