/*
 * Web IDE - Command Line Interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.exec;

import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.URL;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tizen.common.config.Preference;
import org.tizen.common.config.provider.EnvironmentProvider;
import org.tizen.common.config.provider.SystemPropertiesProvider;
import org.tizen.common.core.command.ExecutionContext;
import org.tizen.common.core.command.Executor;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.core.command.policy.PolicyRegistry;
import org.tizen.common.file.FileHandler;
import org.tizen.common.file.FileHandler.Attribute;
import org.tizen.common.file.StandardFileHandler;
import org.tizen.common.util.CollectionUtil;
import org.tizen.common.util.IOUtil;
import org.tizen.common.util.MapUtil;
import org.tizen.common.util.OSChecker;
import org.tizen.common.util.PropertyUtil;
import org.tizen.common.util.StringUtil;

/**
 * <p>
 * AbstractLauncher.
 * 
 * abstract class for simple command.
 * 
 * provide {@link ExecutionContext} for command line
 * </p>
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
abstract public class
AbstractLauncher
{
	
	/**
	 * <p>
	 * Option for help text
	 * </p>
	 */
	protected static final String OPT_HELP = "help";
	
	/**
	 * <p>
	 * Description for help option
	 * 
	 * This is printed out in usage
	 * </p>
	 * 
	 * @see #OPT_HELP
	 */
	protected static final String DESC_HELP = "Print out this page";
	
	/**
	 * <p>
	 * Option for log level
	 * 
	 * This option is shared in shell
	 * </p>
	 */
	protected static final String OPT_LOG = "log";
	
	/**
	 * <p>
	 * Description for log level option
	 * 
	 * This is printed out in usage
	 * </p>
	 * 
	 * @see #OPT_LOG
	 */
	protected static final String DESC_LOG = "Specify log category (error | warn | info | debug | trace)";
	
	/**
	 * <p>
	 * Option for no output
	 * </p>
	 */
	protected static final String OPT_QUIET = "quiet";
	
	/**
	 * <p>
	 * Description for no output option
	 * 
	 * This is printed out in usage
	 * </p>
	 * 
	 * @see #OPT_QUIET
	 */
	protected static final String DESC_QUIET = "Execute silently";
	
	/**
	 * <p>
	 * Property key for command line parameter parser
	 * </p>
	 */
	protected static final String PROP_PARSER = "tizen.cli.parser";
	
	
	/**
	 * <p>
	 * suffix for help properties
	 * </p>
	 */
	protected static final String SUFFIX_CONFIG = ".properties";
	
	/**
	 * <p>
	 * logger for this object
	 * </p>
	 */
	protected final Logger logger = LoggerFactory.getLogger( getClass() );
	
	/**
	 * <p>
	 * object for usage help
	 * </p>
	 */
	protected Help help;
	
	/**
	 * <p>
	 * optional flag to be specified by user
	 * 
	 * flag is formatted in POSIX style in default
	 * </p>
	 */
	protected CommandLine cmd;
	
	/**
	 * {@link Executor} for Command Line Interface
	 */
	protected Executor executor; 

	/**
	 * A command user inputs
	 */
	protected String inputCmd = "";
	
	/**
	 * Convert <code>path</code> to canonical path
	 * 
	 * @param path relative or abstract path
	 * 
	 * @return canonical path
	 * 
	 * @throws IOException If file system deny request
	 */
	protected
	String
	convertPath(
		final String  path
	)
	throws IOException
	{
	    String target = path;
	    if ((!OSChecker.isWindows())
	            && target.startsWith("~"+File.separator)) {
	        target = path.replaceFirst("~", System.getProperty("user.home"));
	    }
		final FileHandler fileHandler = getFileHandler();
		return (String) fileHandler.get( target, Attribute.PATH );
	}

	/**
	 * return execution command for usage
	 * 
	 * @return execution command
	 */
	protected String getSyntax()
	{
		return getHelp().getSyntax();
	}
	
	/**
	 * {@link Options} for usage
	 * 
	 * @return defined {@link Options}
	 */
	@SuppressWarnings("static-access")
	protected
	Options
	getOptions()
	{
		final Options options = new Options();
		
		options.addOption( OptionBuilder.withLongOpt( OPT_HELP ).withDescription( DESC_HELP ).create( OPT_HELP.substring( 0, 1 ) ) );
		options.addOption( OptionBuilder.hasArg().withLongOpt( OPT_LOG ).withDescription( DESC_LOG ).create( OPT_LOG.substring( 0, 1 ) ) );
		options.addOption( OptionBuilder.withLongOpt( OPT_QUIET ).withDescription( DESC_QUIET ).create( OPT_QUIET.substring( 0, 1 ) ) );

		return options;
	}

	/**
	 * {@link Options} for usage. A command does not use short option name and argument.
	 * 
	 * @return defined {@line Options}
	 */
	protected
	Options
	getCommands()
	{
		final Options options = new Options();
		return options;
	}
	
	
	/**
	 * <p>
	 * Create and return {@link Help} instance
	 * </p>
	 * 
	 * @return {@link Help}
	 */
	protected
	Help
	createHelp() {
		return new Help( getOptions() );
	}

	/**
	 * <p>
	 * Return {@link Help} in this cli instance
	 * </p>
	 * @return {@link Help}
	 */
	synchronized protected
	Help
	getHelp()
	{
		if ( null == help )
		{
			this.help = createHelp();
			final ClassLoader cl = getClass().getClassLoader();
			final String resourceName = getClass().getName().replace( '.', '/' ) + SUFFIX_CONFIG;
			
			logger.trace( "Resource name :{}", resourceName );
			try
			{
				final Enumeration<URL> iter = cl.getResources( resourceName );
				while ( iter.hasMoreElements() )
				{
					final URL url = iter.nextElement();
					logger.trace( "URL :{}", url );
					final Properties properties = PropertyUtil.loadProperties( url.openStream() );
					logger.debug( "Properties :{}", properties );
					
					for ( final Object key : properties.keySet() )
					{
						final String value = properties.getProperty( (String) key );
						byte[] contents = IOUtil.getBytes( cl.getResourceAsStream( value ), true );
						this.help.addHelpDetail( (String) key, new String( contents, Charset.forName( "utf8" ) ) );
					}
				}
			}
			catch ( final IOException e )
			{
				logger.warn( "Can't load help", e );
			}
		}
		
		return this.help;
	}
	
	/**
	 * Print out sample usage
	 * 
	 * @return sample usage string
	 */
	protected
	String
	printHelpDetail( String option )
	{
		final String detailStr = getHelp().getHelpDetail( option );
		getPrompter().error( detailStr );
		return detailStr;
	}
	
	/**
	 * Print out usage
	 * 
	 * @return usage string
	 */
	protected
	String
	printHelp()
	{
		final String helpStr = getHelp().getHelp();
		getPrompter().error( helpStr );
		return helpStr;
	}

	/**
	 * Print out invalid usage
	 * 
	 * @return invalid usage string
	 */
	protected
	String
	printError( String msg )
	{
		String errorStr = "E: Invalid usage";
		if ( !StringUtil.isEmpty( msg ) ) {
			errorStr = MessageFormat.format( "E: Invalid usage [{0}]", msg );
		}
		getPrompter().error( errorStr );
		return errorStr;
	}

	/**
	 * Entry method for execution
	 * 
	 * @param args command line argument
	 * 
	 * @throws Exception If unhandled exception occured
	 */
	public void
	run(
		final String... args
	)
	throws Exception
	{
		Preference.register( "OS", new EnvironmentProvider() );
		Preference.register( "JVM", new SystemPropertiesProvider() );
		executor = new CommandLineExecutor(
			new ConsolePrompter( System.out, new InputStreamReader( System.in ) )
		);
		if ( logger.isTraceEnabled() )
		{
			logger.trace( "OS Environment :\n{}", MapUtil.toString( System.getenv() ) );
			logger.trace( "System properties :\n{}", MapUtil.toString( System.getProperties() ) );
			logger.trace( "Args :{}", CollectionUtil.concatenate( Arrays.asList( args ), "," ) );
		}
		
		try
		{
			Options cmds = this.getCommands();
			
			if(args.length == 0 && isPrintHelp()) {
				printHelp();
				return;
			}
			if(cmds.getOptions().size() > 0 && args.length > 0 && cmds.hasOption(args[0])) {
				this.inputCmd = args[0];
				String[] newArgs = Arrays.copyOfRange(args, 1, args.length);
				this.cmd = parse( newArgs);
			}
			else {
				this.inputCmd = "";
				cmd = parse( args );
			}
			
			if ( cmd.hasOption( OPT_QUIET ) )
			{
				executor = new CommandLineExecutor(
					new ConsolePrompter( new PrintStream( new OutputStream() {
						public void write(int b) throws IOException {
						}
					} ), new InputStreamReader( System.in ) )
				);
				
			}
			
			if ( cmd.hasOption( OPT_HELP ) )
			{
				printHelp();
				
				logger.debug( "Help argument :{}", cmd.getArgList() );
				
				for ( String keyword : cmd.getArgs() )
				{
					printHelpDetail( keyword );
				}
				return ;
			}

			execute( cmd );
		}
		catch ( final ParseException e )
		{
			
			logger.error( "Parsing Error: ", e );
			logger.trace( "Invalid usage." );
			
			// args include "-h, --help" then working 'help'
			boolean bPrintHelp = false;
			for ( String arg : args ) {
				if ( ( "-" + OPT_HELP.substring( 0, 1 ) ).equals( arg ) || ( "--" + OPT_HELP ).equals( arg ) ) {
					printHelp();
					bPrintHelp = true;
					break;
				}
			}

			if ( !bPrintHelp ) {
				printError( e.getMessage() );
			}

			if ( null != cmd ) {
				logger.debug( "Help argument :{}", cmd.getArgList() );
				
				for ( String keyword : cmd.getArgs() )
				{
					printHelpDetail( keyword );
				}
				
			}
		}
		catch ( final Throwable e )
		{
			handleException( e );
		}
	}
	
	   /**
	 * Gets a command that user inputs.
	 *
	 * @return a command user inputs
	 */
	protected String getInputCmd() {
		return this.inputCmd;
	}

	/**
	 * Handle uncaught exception
	 * 
	 * @param e uncaught exception
	 */
	protected
	void
	handleException(
		final Throwable e
	)
	{
		getPrompter().error( e.getMessage() );
		logger.error( "Command stop because of exception", e );

		Throwable iter = e;
		while ( null != iter )
		{
			final String localMessage = iter.getLocalizedMessage();
			if ( null != localMessage )
			{
				logger.error( localMessage );
				exit( 1 );
			}

			final String message = iter.getMessage();
			if ( null != message )
			{
				logger.error( message );
				exit( 1 );
			}
			iter = iter.getCause();
		}
	}

	/**
	 * Parse arguments <code>args</code> with {@link CommandLineParser}
	 * 
	 * @param args arguments to parse
	 * 
	 * @return {@link CommandLine} to be parsed
	 * 
	 * @throws InstantiationException If custom parser is invalid
	 * @throws IllegalAccessException If custom parser is invalid
	 * @throws ParseException If arguements is invalid
	 */
	protected
	CommandLine
	parse(
		final String... args
	)
	throws InstantiationException, IllegalAccessException, ParseException
	{
		final CommandLineParser parser = getParser();
		
		final Options options = getOptions();

		final CommandLine cmdLine = parser.parse( options, args );
		
		return cmdLine;
	}
	
	/**
	 * Return default parser {@link Class}
	 * 
	 * @return {@link Class} for default parser
	 */
	protected
	static
	Class<?>
	getDefaultParserClass()
	{
		return PosixParser.class;
	}
	
	/**
	 * Return parser along user selection.
	 * 
	 * The selection is specified JVM system properties
	 * 
	 * Support two built-in parser
	 * <ul>
	 * <li>POSIX style - tar -zxvf foo.tar.gz</li>
	 * <li>GNU style - du --human-readable --max-depth=1</li>
	 * </ul>
	 * 
	 * if you want use custom parser, specify class name
	 * 
	 * @return {@link CommandLineParser} to parse
	 * 
	 * @throws InstantiationException If custom parser can't be instantiated
	 * @throws IllegalAccessException If custom parser's constructor is NOT accessible
	 * 
	 * @see #PROP_PARSER
	 * @see CommandLineParser
	 */
	protected
	CommandLineParser
	getParser()
	throws InstantiationException, IllegalAccessException {
		final String parserName = System.getProperty( PROP_PARSER );
		logger.trace( "Parser name :{}", parserName );
		Class<?> parserClazz = getDefaultParserClass();
		try {
			if ( null != parserName )
			{
				parserClazz = Class.forName( parserName );
			}
		}
		catch ( final ClassNotFoundException e )
		{
			if ( "posix".equalsIgnoreCase( parserName ) )
			{	// tar -zxvf foo.tar.gz
				parserClazz = PosixParser.class;
			}
			else if ( "gnu".equalsIgnoreCase( parserName ) )
			{
				parserClazz = GnuParser.class;
			}
			else
			{
				logger.warn( "Invalid argument parser :" + parserName );
			}
		}
		
		logger.trace( "Parser class :{}", parserClazz );
		
		return (CommandLineParser) parserClazz.newInstance();
	}

	/**
	 * Return managed command line arguments
	 * 
	 * @return managed arguments
	 * 
	 * @see CommandLine
	 */
	protected
	CommandLine
	getCommandLine()
	{
		return this.cmd;
	}
	
	/**
	 * Return defined executor for command
	 * 
	 * @return {@link ExecutionContext} to use
	 */
	protected
	Executor
	getExecutor()
	{
		if ( null == executor )
		{
			this.executor = new Executor();
		}
	
		return executor;
	}
	
	/**
	 * Return {@link ExecutionContext} in cli
	 * 
	 * @return {@link ExecutionContext}
	 */
	protected
	ExecutionContext
	getExecutionContext()
	{
		return getExecutor().getContext();
	}
	
	/**
	 * Return {@link PolicyRegistry} in cli
	 * 
	 * @return {@link PolicyRegistry}
	 */
	protected
	PolicyRegistry
	getPolicy()
	{
		return getExecutionContext().getPolicyRegistry();
	}
	
	/**
	 * Return {@link FileHandler} in cli
	 * 
	 * @return {@link FileHandler}
	 * 
	 * @see #getExecutionContext()
	 * @see StandardFileHandler
	 */
	protected
	FileHandler
	getFileHandler()
	{
		return getExecutionContext().getFileHandler();
	}
	
	/**
	 * Return {@link Prompter} in cli
	 * 
	 * @return {@link Prompter}
	 * 
	 * @see #getExecutionContext()
	 */
	protected
	Prompter
	getPrompter()
	{
		return getExecutionContext().getPrompter();
	}

	/**
	 * Returns whether print help or not if argument is empty.
	 * 
	 * @return returns </code>true</code> if it prints help. Else, <code>false</code>.
	 */
	protected
	boolean
	isPrintHelp() {
		return false;
	}
	
	/**
	 * Gets the uses description of the CLI. 
	 * 
	 * @return returns usage description string.
	 */
	protected String getUsageDescription() {
		return StringUtil.EMPTY_STRING;
	}
	
	/**
	 * Terminates the currently running command line interface.<p>
	 * The argument serves as a status code<br>
	 * by convention, a nonzero status code indicates abnormal termination.<br>
	 * This method calls the System.exit method.
	 * 
	 * @param status exit status.
	 */
	protected
	void
	exit(int status)
	{
		System.exit( status );
	}

	/**
	 * Execute command
	 * 
	 * Implement execution of command
	 * 
	 * @param cmdLine arguments
	 * 
	 * @throws Exception If Unhandled exception occured
	 */
	abstract protected void execute( CommandLine cmdLine ) throws Exception;


}
