/*
* Native IDE - Command Line Interface
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* Ho Namkoong <ho.namkoong@samsung.com>
* BonYong Lee <bonyong.lee@samsung.com>
* Kangho Kim <kh5325.kim@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/

package org.tizen.cli.exec.gen;

import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

import org.tizen.common.core.application.InstallPathConfig;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.StringUtil;
import org.tizen.nativecommon.Activator;
import org.tizen.nativecommon.exception.TizenProcessFailureException;
import org.tizen.nativecommon.templateengine.TizenTemplateEngine;
import org.tizen.nativecommon.templateengine.build.AbstractProjectGenerator;
import org.tizen.nativecommon.templateengine.build.model.BuildProperty;
import org.tizen.nativecommon.templateengine.build.model.DevPackageModel;
import org.tizen.nativecommon.templateengine.build.model.RootstrapModel;
import org.tizen.nativecommon.templateengine.build.model.TargetModel;
import org.tizen.nativecommon.templateengine.build.model.ToolchainModel;
import org.tizen.nativecommon.templateengine.model.TizenTemplate;
import org.tizen.nativecommon.templateengine.process.TizenProcessRunner;
import org.tizen.nativecommon.templateengine.util.TemplateUtil;

public class CLIProjectGenerator extends AbstractProjectGenerator {

    private static final String PACKAGE_PREFIX_TEMPLATE_PROCESS = Activator.PLUGIN_ID + ".templateengine.process";
    
    private static final String MACRO_BINARY = "CLI_BINARY";
    private static final String MACRO_PRJTYPE = "projectType";
    private static final String MACRO_PLATFORM_VER = "platformVer";
    private static final String MACRO_TOOLCHAIN = "tc";
    
    private static final String WHITE_SPACE = " ";
    private static final String WHITE_SPACE_IN_CLI = "\\ ";
    
    public static final String DIR_MAKEFILE = InstallPathConfig.getSDKPath() + File.separatorChar + "tools" + File.separatorChar + "ide" + File.separatorChar + "template" + File.separatorChar +"makefile";
    private static final String DIR_ROOTSTRAP = "$(ROOTSTRAP)";
    private static final String DIR_SDK = "$(SDKPATH)";
    private static final String DIR_DEFAULT_TC_CF = "$(TC_COMPILER_MISC)";
    private static final String DIR_DEFAULT_RS_CF = "$(RS_COMPILER_MISC)";
    private static final String DIR_DEFAULT_TC_LF = "$(TC_LINKER_MISC)";
    private static final String DIR_DEFAULT_RS_LF = "$(RS_LINKER_MISC)";
    
    public CLIProjectGenerator() {
        super();
    }

    @Override
    protected String getSDKPath() {
        return DIR_SDK;
    }
    
    @Override
    protected String getDefaultCfTc(ToolchainModel target) {
        return DIR_DEFAULT_TC_CF;
    }
    
    @Override
    protected String getDefaultCfRs(RootstrapModel rootstrap) {
        return DIR_DEFAULT_RS_CF;
    }
    
    @Override
    protected String getDefaultLfTc(ToolchainModel toolchain) {
        return DIR_DEFAULT_TC_LF;
    }
    
    @Override
    protected String getDefaultLfRs(RootstrapModel rootstrap) {
        return DIR_DEFAULT_RS_LF;
    }
    
    @Override
    protected String getRootstrapPath(String rootstrapId) {
        return DIR_ROOTSTRAP;
    }
    
    @Override
    protected String getQuotedString(String msg) {
        return msg.replace(WHITE_SPACE, WHITE_SPACE_IN_CLI);
    }
    
    
    @Override
    protected String getManifestAppId() {
        return "$(APPID)";
    }
    @Override
    public void pressDefaultPostValueStore(Map<String, String> valueStore, TizenTemplate template) throws TizenProcessFailureException {
        
        //MACRO_SRCS, MACRO_OBJ, MACRO_DEP is not used in makefile anymore.
//        pressValueStoreForSrcObjDep(valueStore, template);
        pressValueStoreForBuildData(valueStore, template);
        
        String projectName = template.getProjectName();
        String projectType = (String) template.getProperty(TizenTemplate.PROPERTY_KEY_PROJECTTYPE);
        if(projectType.equals(TizenTemplateEngine.PROJECTTYPE_APP)) {
            valueStore.put(MACRO_BINARY, projectName + ".exe");
        }
        else if(projectType.equals(TizenTemplateEngine.PROJECTTYPE_SHARED)) {
            valueStore.put(MACRO_BINARY, projectName + ".so");
        }
        else if(projectType.equals(TizenTemplateEngine.PROJECTTYPE_STATIC)) {
            valueStore.put(MACRO_BINARY, projectName + ".a");
        }
        else {
            String msg = MessageFormat.format(Messages.getString("CLIProjectGenerator.EXCEPTION_UNEXPECTED_PROJECTTYPE"), projectType); //$NON-NLS-1$
            throw new TizenProcessFailureException(msg);
        }
        
        super.pressDefaultPostValueStore(valueStore, template);
    }

    private void pressValueStoreForBuildData(Map<String, String> valueStore, TizenTemplate template) {
        valueStore.put(MACRO_PLATFORM_VER, (String) template.getProperty(MACRO_PLATFORM_VER));
        valueStore.put(MACRO_PRJTYPE, (String) template.getProperty(MACRO_PRJTYPE));
        valueStore.put(MACRO_TOOLCHAIN, (String) template.getProperty(MACRO_TOOLCHAIN));
    }

    @Override
    public void pressPostValueStoreFromDevPkg(Map<String, String> valueStore, TizenTemplate template) {
        BuildProperty emulBuildProperty = (BuildProperty) template.getProperty(TizenTemplate.PROPERTY_KEY_EMUL_BUILD_PROPERTY);
        
        pressValueStoreForFlag(valueStore, TizenTemplateEngine.MACRO_EMULINC, emulBuildProperty.getIncludePathList(), "-I");
        pressValueStoreForFlag(valueStore, TizenTemplateEngine.MACRO_EMULCF, emulBuildProperty.getcFlagList(), StringUtil.EMPTY_STRING);
        pressValueStoreForFlag(valueStore, TizenTemplateEngine.MACRO_EMULLF, emulBuildProperty.getlFlagList(), StringUtil.EMPTY_STRING);
    }
    
    private static void pressValueStoreForFlag(Map<String, String> valueStore, String macro, List<String> flagList, String pre) {
        StringBuffer emulBuffer = new StringBuffer();
        
        for(String emulInc: flagList) {
            emulBuffer.append(pre + emulInc + " ");
        }
        
        valueStore.put(macro, emulBuffer.toString());
    }

    @Override
    public void createBuildFiles(Map<String, String> postValueStore, TizenTemplate template) throws TizenProcessFailureException {
        String projectType = (String) template.getProperty(TizenTemplate.PROPERTY_KEY_PROJECTTYPE);
        
        String buildDataFile = FileUtil.appendPath(DIR_MAKEFILE, "build_data");
        
        String srcMakeFile = "";
        if(projectType.equals(TizenTemplateEngine.PROJECTTYPE_APP)) {
            srcMakeFile = FileUtil.appendPath(DIR_MAKEFILE, "makefile_app_template");
        }
        else if(projectType.equals(TizenTemplateEngine.PROJECTTYPE_SHARED)) {
            srcMakeFile = FileUtil.appendPath(DIR_MAKEFILE, "makefile_shared_template");
        }
        else if(projectType.equals(TizenTemplateEngine.PROJECTTYPE_STATIC)) {
            srcMakeFile = FileUtil.appendPath(DIR_MAKEFILE, "makefile_static_template");
        }
        
        String buildDir = FileUtil.appendPath(template.getProjectPath(), "CommandLineBuild");
        new File(buildDir, "src").mkdirs();
        
        try {
            TemplateUtil.createReplaceableFile(srcMakeFile, FileUtil.appendPath(buildDir, "makefile"), postValueStore);
            TemplateUtil.createReplaceableFile(buildDataFile, FileUtil.appendPath(buildDir, "build_data"), postValueStore);
        } catch (IOException e) {
            throw new TizenProcessFailureException(Messages.getString("CLIProjectGenerator.EXCEPTION_CREATING_MAKEFILE"), e);
        }
    }

    @Override
    public TizenProcessRunner getTemplateProcess(String templateId, TizenTemplate template) throws TizenProcessFailureException {
        String processResolvedName = PACKAGE_PREFIX_TEMPLATE_PROCESS + "." + FileUtil.getFileExtension(templateId);
        ClassLoader cLoader = ClassLoader.getSystemClassLoader();
        
        TizenProcessRunner runner = null;
        
        try {
            Class<?> processClass = cLoader.loadClass(processResolvedName);
//            Constructor<?> tizenProcessConstructor = processClass.getConstructor(new Class[] {TizenTemplate.class});
            runner = (TizenProcessRunner) processClass.newInstance();
        } catch (Exception e) {
            throw new TizenProcessFailureException(MessageFormat.format(Messages.getString("CLIProjectGenerator.EXCEPTION_CREATING_CLASS"), processResolvedName), e); //$NON-NLS-1$
        }
        
        runner.setTemplate(template);
        return runner;
    }
    
    @Override
    public void doRemainingJob(List<DevPackageModel> emulDevPackageModelList, TizenTemplate template, TargetModel target) {
        ToolchainModel toolChain = target.getToolchain();
        RootstrapModel rootstrap = target.getRootstrap();
        
        String platformVer = rootstrap.getVersion();
        String toolChainName = toolChain.getName();
        String projectType = toolChain.getProjectType();
        projectType = FileUtil.getFileExtension(projectType);
        
        template.setProperty(MACRO_PLATFORM_VER, platformVer);
        template.setProperty(MACRO_PRJTYPE, projectType);
        template.setProperty(MACRO_TOOLCHAIN, toolChainName);
    }
}
