/*
 * usb_host.h
 * Copyright (c) 2016 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __TIZEN_USB_HOST_H__
#define __TIZEN_USB_HOST_H__

#include <stdint.h>
#include <tizen.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief An enumeration representing USB endpoint direction.
 * @details Used to control whether an endpoint writes or reads data.
 * @since_tizen 3.0
 */
typedef enum {
	USB_HOST_DIRECTION_IN,	/**< IN direction */
	USB_HOST_DIRECTION_OUT,	/**< OUT direction */
} usb_host_endpoint_direction_e;

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief An enumeration representing USB endpoint transfer type.
 * @details Used to control how a data transfer request is going to operate.
 * @since_tizen 3.0
 */
typedef enum {
	USB_HOST_TRANSFER_TYPE_CONTROL,		/**< Control transfer */
	USB_HOST_TRANSFER_TYPE_ISOCHRONOUS,	/**< Isochronous transfer */
	USB_HOST_TRANSFER_TYPE_BULK,		/**< Bulk transfer */
	USB_HOST_TRANSFER_TYPE_INTERRUPT,	/**< Interrupt transfer */
} usb_host_transfer_type_e;

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Enumeration of isochronous endpoint's synchronization type.
 * @details Used to control how a data transfer request is going to be synchronized.
 * @since_tizen 3.0
 */
typedef enum {
	USB_HOST_ISO_SYNC_TYPE_NONE,	/**< No synchronization */
	USB_HOST_ISO_SYNC_TYPE_ASYNC,	/**< Asynchronous */
	USB_HOST_ISO_SYNC_TYPE_ADAPTIVE,/**< Adaptive */
	USB_HOST_ISO_SYNC_TYPE_SYNC,	/**< Synchronous */
} usb_host_iso_sync_type_e;

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief An enumeration representing USB endpoint usage type.
 * @details Used to control what sort of tasks an endpoint is up to.
 * @since_tizen 3.0
 */
typedef enum {
	USB_HOST_USAGE_TYPE_DATA,	/**< Data endpoint */
	USB_HOST_USAGE_TYPE_FEEDBACK,	/**< Feedback endpoint */
	USB_HOST_USAGE_TYPE_IMPLICIT,	/**< Implicit feedback Data endpoint */
} usb_host_usage_type_e;

/**
 * @ingroup CAPI_USB_HOST_MODULE
 * @brief Enumeration of error codes reported by USB host API.
 * @details Most USB Host functions will return those errors to denote what happened.
 * @since_tizen 3.0
 */
typedef enum {
	USB_HOST_ERROR_NONE = TIZEN_ERROR_NONE, /**< Successful */
	USB_HOST_ERROR_IO_ERROR = TIZEN_ERROR_IO_ERROR, /**< I/O error */
	USB_HOST_ERROR_INVALID_PARAMETER = TIZEN_ERROR_INVALID_PARAMETER, /**< Invalid parameter */
	USB_HOST_ERROR_PERMISSION_DENIED = TIZEN_ERROR_PERMISSION_DENIED, /**< Permission denied */
	USB_HOST_ERROR_NO_SUCH_DEVICE = TIZEN_ERROR_NO_SUCH_DEVICE, /**< No device */
	USB_HOST_ERROR_NOT_FOUND = TIZEN_ERROR_USB_HOST | 0x01, /**< Entity not found */
	USB_HOST_ERROR_RESOURCE_BUSY = TIZEN_ERROR_RESOURCE_BUSY, /**< Resource busy */
	USB_HOST_ERROR_TIMED_OUT = TIZEN_ERROR_TIMED_OUT, /**< Operation timed out */
	USB_HOST_ERROR_OVERFLOW = TIZEN_ERROR_USB_HOST | 0x02, /**< Overflow */
	USB_HOST_ERROR_DEVICE_NOT_OPENED = TIZEN_ERROR_USB_HOST | 0x03, /**< Device is not opened */
	USB_HOST_ERROR_BROKEN_PIPE = TIZEN_ERROR_BROKEN_PIPE, /**< Pipe error */
	USB_HOST_ERROR_INTERRUPTED_SYS_CALL = TIZEN_ERROR_INTERRUPTED_SYS_CALL, /**< System call interrupted */
	USB_HOST_ERROR_OUT_OF_MEMORY = TIZEN_ERROR_OUT_OF_MEMORY, /**< Insufficient memory */
	USB_HOST_ERROR_NOT_SUPPORTED = TIZEN_ERROR_NOT_SUPPORTED, /**< Operation not supported */
	USB_HOST_ERROR_UNKNOWN = TIZEN_ERROR_UNKNOWN, /**< Other error */
} usb_host_error_e;

/**
 * @ingroup CAPI_USB_HOST_HOTPLUG_MODULE
 * @brief An enumeration representing USB hotplug event types.
 * @details Says whether a hotplug event was about attachment or detachment.
 * @since_tizen 4.0
 */
typedef enum {
	USB_HOST_HOTPLUG_EVENT_ATTACH,	/**< Device was connected */
	USB_HOST_HOTPLUG_EVENT_DETACH,	/**< Device was disconnected */
	USB_HOST_HOTPLUG_EVENT_ANY,	/**< Any event */
} usb_host_hotplug_event_e;

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief An enumeration representing USB async transfer status codes.
 * @details Used to give the current status of a USB transfer handle.
 * @since_tizen 5.0
 */
typedef enum {
	USB_HOST_TRANSFER_COMPLETED, /**< Transfer completed without error*/
	USB_HOST_TRANSFER_ERROR,     /**< Transfer failed */
	USB_HOST_TRANSFER_TIMED_OUT, /**< Transfer timed out */
	USB_HOST_TRANSFER_CANCELLED, /**< Transfer cancelled */
	USB_HOST_TRANSFER_STALL,     /**< Halt condition detected */
	USB_HOST_TRANSFER_NO_DEVICE, /**< Device was disconnected */
	USB_HOST_TRANSFER_OVERFLOW,  /**< Device sent more data than requested */
	USB_HOST_TRANSFER_UNKNOWN,   /**< Unknown status */
} usb_host_transfer_status_e;

/**
 * @ingroup CAPI_USB_HOST_MODULE
 * @brief An opaque handle representing the USB host context.
 * @details This structure represents usb_host session.
 * Using own session allows to use this API independently.
 * For example calling usb_host_destroy() will not destroy resources
 * that are being used by another user of the library.
 *
 * To create session there is need to call usb_host_create(),
 * to destroy call usb_host_destroy(). Each session created by usb_host_create()
 * has to be destroyed using usb_host_destroy()
 * @since_tizen 3.0
 */
typedef struct usb_host_context_s *usb_host_context_h;

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief An opaque handle representing a USB device on the bus.
 * @details This structure represents USB device found on USB bus.
 *
 * This can be obtained by usb_host_get_device_list() or usb_host_device_open_with_vid_pid().
 * Some basic operations can be performed on closed device obtained by usb_host_get_device_list().
 * To perform any I/O operations the device must be opened by calling usb_host_device_open()
 * or usb_host_device_open_with_vid_pid().
 *
 * @since_tizen 3.0
 */
typedef struct usb_host_device_s *usb_host_device_h;

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief An opaque handle representing a USB device configuration.
 * @details This type represents USB device configuration. Device can have multiple configurations,
 * a configuration can have multiple interfaces. This handle can be obtained by
 * usb_host_device_get_config().
 * @since_tizen 3.0
 */
typedef struct usb_host_config_s *usb_host_config_h;

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief An opaque handle representing a USB configuration interface.
 * @details This type represents USB interface. An interface is a part of configuration and
 * can have multiple endpoints. This handle can be obtained by usb_host_config_get_interface().
 * @since_tizen 3.0
 */
typedef struct usb_host_interface_s *usb_host_interface_h;

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief An opaque handle representing a USB interface endpoint.
 * @details This type represents USB endpoint. This handle can be obtained by
 * usb_host_interface_get_endpoint().
 * @since_tizen 3.0
 */
typedef struct usb_host_endpoint_s *usb_host_endpoint_h;

/**
 * @ingroup CAPI_USB_HOST_HOTPLUG_MODULE
 * @brief An opaque handle representing a USB hotplug callback manager.
 * @details This handle is used for managing registered hotplug callbacks.
 * @since_tizen 4.0
 */
typedef struct usb_host_hotplug_s *usb_host_hotplug_h;

/**
 * @ingroup CAPI_USB_HOST_HOTPLUG_MODULE
 * @brief An type representing a USB hotplug callback handler.
 * @details The device handle should be unreffed with usb_host_unref_device()
 * when no longer needed.
 * @since_tizen 4.0
 * @param[in] dev Device which was connected/disconnected
 * @param[in] user_data User data pointer passed on callback registration
 */
typedef void (*usb_host_hotplug_cb)(usb_host_device_h dev, void *user_data);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief An opaque handle representing an asynchronous USB transfer.
 * @details This type represents an asynchronous USB transfer.
 * @since_tizen 5.0
 */
typedef struct usb_host_transfer_s *usb_host_transfer_h;

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief An type representing a USB transfer finish callback.
 * @details Use to be notified for when a USB transfer is done.
 * @since_tizen 5.0
 * @param[in] transfer Transfer for which this callback was set
 * @param[in] user_data User data pointer passed on callback registration
 */
typedef void (*usb_host_transferred_cb)(usb_host_transfer_h transfer, void *user_data);

/* Library initialization and cleanup */

/**
 * @ingroup CAPI_USB_HOST_MODULE
 * @brief Initializes a USB host context, providing a handle to it
 * @details This function must be called before any other function from this module.
 * @since_tizen 3.0
 * @remarks @a ctx should be destroyed by calling usb_host_destroy() when no longer needed.
 * @param[out] ctx Context pointer
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NONE Successful
 * @post usb_host_destroy() should be called to free resources allocated for ctx by this function.
 *
 * @code
#include <usb_host.h>

int main(void)
{
	int ret;
	usb_host_context_h usb_ctx;

	ret = usb_host_create(&usb_ctx);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Use the context here

	ret = usb_host_destroy(usb_ctx);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_create(usb_host_context_h *ctx);

/**
 * @ingroup CAPI_USB_HOST_MODULE
 * @brief Deinitializes a USB host context, invalidating a handle
 * @details This function must be called after closing all devices
 * and before application close. It has to be called to clean
 * the memory used by library.
 * @since_tizen 3.0
 * @param[in] ctx Context to deinitialize
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Success
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre Context must be initialized by usb_host_create().
 *
 * @code
#include <usb_host.h>

int main(void)
{
	int ret;
	usb_host_context_h ctx;

	ret = usb_host_create(&ctx);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Use the context here

	ret = usb_host_destroy(ctx);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_destroy(usb_host_context_h ctx);

/* Device handling and enumeration */

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Returns a list of USB devices attached to the system.
 * @details This function returns list of USB devices attached to system.
 * To free obtained device list usb_host_free_device_list() should be used, this
 * function can also unref devices. Do not unref device and then open it.
 *
 * All devices have reference counter. Functions usb_host_ref_device() and
 * usb_host_unref_device() are used to ref or unref device. When ref counter
 * reaches 0 device will be freed.
 * Devices reached by calling usb_host_get_device_list() have a reference count of
 * 1, and usb_host_free_device_list() can optionally decrease the reference count
 * on all devices in the list. usb_host_device_open() adds another reference which is
 * later destroyed by usb_host_device_close().
 *
 * @since_tizen 3.0
 * @param[in] ctx Context handle
 * @param[out] devs An array of devices
 * @param[out] length Number of devices
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Operation not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @post @a devs must be freed with usb_host_free_device_list() when no longer needed.
 *
 * @code
#include <usb_host.h>

void iterate_devices(usb_host_context_h ctx)
{
	int ret, length;
	usb_host_device_h *devs;

	ret = usb_host_get_device_list(ctx, &devs, &length);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (int i = 0; i < length; ++i)
		handle_device(devs[i]);

	ret = usb_host_free_device_list(devs, true);
	// Or if handle_device unreferences the devices:
	// ret = usb_host_free_device_list(devs, false);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_get_device_list(usb_host_context_h ctx, usb_host_device_h **devs, int *length);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Frees devices list and optionally unrefs its contents
 * @details This function needs to be called to free device list. This
 * function can also unref devices if unref_devices is set to non-zero value.
 * Do not unref device and then open it.
 * @since_tizen 3.0
 * @param[in] devs List of devices
 * @param[in] unref_devices Set to true to unreference devices, set to false to not unref
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre usb_host_get_device_list() must be called before using this function.
 *
 * @code
#include <usb_host.h>

void iterate_devices(usb_host_context_h ctx)
{
	int ret, length;
	usb_host_device_h *devs;

	ret = usb_host_get_device_list(ctx, &devs, &length);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (int i = 0; i < length; ++i)
		handle_device(devs[i]);

	ret = usb_host_free_device_list(devs, true);
	// Or if handle_device unreferences the devices:
	// ret = usb_host_free_device_list(devs, false);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_free_device_list(usb_host_device_h *devs, bool unref_devices);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Increments the reference counter of given USB device
 * @details Increment ref count of device. If ref count reaches zero,
 * device will be destroyed.
 * @since_tizen 3.0
 * @param[in] dev Device to reference
 * @return 0 on success, error code otherwise
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

usb_host_device_h pick_device(usb_host_device_h *device_list, int list_len)
{
	int ret;
	usb_host_device_h out = NULL;

	for (int i = 0; ++i; i < list_len) {
		if (interesting(device_list[i])) {
			// Reference the device that we will use
			ret = usb_host_ref_device(device_list[i]);
			if (ret != USB_HOST_ERROR_NONE)
				handle_error(ret);

			out = device_list[i];
			break;
		}
	}

	// Free the devices and the list
	ret = usb_host_free_device_list(devs, true);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// The device we referenced is the only device that remains valid
	return out;
}
 * @endcode
 */
int usb_host_ref_device(usb_host_device_h dev);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Decrements the reference counter of given USB device
 * @details Decrements ref count of device. If ref count reaches zero,
 * device will be destroyed.
 * @since_tizen 3.0
 * @param[in] dev Device to unreference
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

void free_device_list_manually(usb_host_device_h *device_list, int list_len)
{
	int ret;

	for (int i = 0; ++i; i < list_len) {
		ret = usb_host_unref_device(device_list[i]);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);
	}

	// Now we only need to free the list and not the devices, hence false
	ret = usb_host_free_device_list(devs, false);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_unref_device(usb_host_device_h dev);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Opens a device obtained from the list of devices to allow operations.
 * @details This function opens a device, which allows performing operations on it
 * (including transfer operations and strings introspection).
 * @since_tizen 3.0
 * @remarks An application having platform privilege level can use this api without user confirmation
 * by declaring %http://tizen.org/privilege/usb.host, which has been added since 6.5.
 * @param[in] dev Device to open
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Memory allocation failure
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE There is no device connected
 * @retval #USB_HOST_ERROR_PERMISSION_DENIED No proper permission to access device
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Operation not supported
 * @see usb_host_is_device_opened()
 *
 * @code
#include <usb_host.h>

void work_on_device(usb_host_device_h dev)
{
	int ret;

	ret = usb_host_device_open(dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Use the device here

	ret = usb_host_device_close(dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_device_open(usb_host_device_h dev);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Closes an opened but not destroyed USB device handle.
 * @details Function should be called before usb_host_destroy().
 * @since_tizen 3.0
 * @param[in] dev Device that should be closed
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_DEVICE_NOT_OPENED If device is not opened
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

void work_on_device(usb_host_device_h dev)
{
	int ret;

	ret = usb_host_device_open(dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Use the device here

	ret = usb_host_device_close(dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_device_close(usb_host_device_h dev);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Opens device with valid, known idVendor and idProduct.
 * @details This function can be used to open device with known idVendor and
 * idProduct. If two or more devices have same vendor and product id only
 * first will be opened.
 * @since_tizen 3.0
 * @remarks An application having platform privilege level can use this api without user confirmation
 * by declaring %http://tizen.org/privilege/usb.host, which has been added since 6.5.
 * @param[in] ctx Context
 * @param[in] vendor_id idVendor of connected device
 * @param[in] product_id idProduct of connected device
 * @param[out] device_handle Opened device handle
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Insufficient memory
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE No device
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

#define VENDOR_ID 0x04e8
#define PRODUCT_ID 0x685e

void work_on_device(usb_host_context_h ctx)
{
	int ret;
	usb_host_device_h dev;

	ret = usb_host_device_open_with_vid_pid(ctx, VENDOR_ID, PRODUCT_ID, &dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Use the device here

	ret = usb_host_device_close(dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// It is still required to unreference the device,
	// as usb_host_device_close only closes the connection,
	// it does not deallocate the object.
	ret = usb_host_unref_device(dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_device_open_with_vid_pid(usb_host_context_h ctx,
				  int vendor_id, int product_id, usb_host_device_h *device_handle);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the number of the bus the given device is connected to.
 * @details Gets device bus number. This is number of the bus
 * that device is connected to.
 * @since_tizen 3.0
 * @param[in] dev Device handle
 * @param[out] bus_number Device bus number
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_bus_number(usb_host_device_h dev)
{
	int ret, bus_number;

	ret = usb_host_device_get_bus_number(dev, &bus_number);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Bus number: %d\n", bus_number);
}
 * @endcode
 */
int usb_host_device_get_bus_number(usb_host_device_h dev, int *bus_number);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the address on the bus the given device is connected to.
 * @details Gets device address. This is address of device on the bus
 * that device is connected to.
 * @since_tizen 3.0
 * @param[in] dev Device
 * @param[out] device_address Device address
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_address(usb_host_device_h dev)
{
	int ret, device_address;

	ret = usb_host_device_get_address(dev, &device_address);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Device address: %d\n", device_address);
}
 * @endcode
 */
int usb_host_device_get_address(usb_host_device_h dev, int *device_address);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets list of port numbers available from a device.
 * @details Gets list of all port numbers from a device.
 * @since_tizen 3.0
 * @param[in] dev Device
 * @param[out] port_numbers Array to be filled with port numbers
 * @param[in] port_numbers_len Max length of array
 * @param[out] ports_count Number of all ports obtained from device
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Insufficient memory
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

#define MAX_PORT_NUMBER 128

void print_port_numbers(usb_host_device_h dev)
{
	int ret, port_numbers[MAX_PORT_NUMBER], ports_count;

	ret = usb_host_device_get_port_numbers(dev, port_numbers, MAX_PORT_NUMBER, &ports_count);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (int i = 0; i < ports_count; ++i)
		printf("Available port: %d\n", port_numbers[i]);
}
 * @endcode
 */
int usb_host_device_get_port_numbers(usb_host_device_h dev, int *port_numbers, int port_numbers_len, int *ports_count);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets one of the USB configurations for an USB device.
 * @details Gets one of the USB configurations by its index.
 * @since_tizen 3.0
 * @remarks @a config must be freed with usb_host_config_destroy().
 * @param[in] dev Device
 * @param[in] config_index index of configuration to retrieve (counting from 0)
 * @param[out] config Output location for USB configuration
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_FOUND The configuration does not exist
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @post Returned configuration should be destroyed by usb_host_config_destroy()
 * when no longer needed.
 *
 * @code
#include <usb_host.h>

void select_configuration(usb_host_device_h dev)
{
	int ret, num_configurations;
	usb_host_config_h config;

	ret = usb_host_device_get_num_configurations(dev, &num_configurations);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (int i = 0; i < num_configurations; ++i) {
		ret = usb_host_device_get_config(dev, i, &config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);

		if (interesting(config)) {
			ret = usb_host_set_config(config);
			if (ret != USB_HOST_ERROR_NONE)
				handle_error(ret);
		}

		ret = usb_host_config_destroy(config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);
	}
}
 * @endcode
 */
int usb_host_device_get_config(usb_host_device_h dev, int config_index, usb_host_config_h *config);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets a handle to the configuration currently active on the device.
 * @details Gets handle to the currently active configuration.
 * This function will return 0 value in config parameter :if device is unconfigured.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] config Handle to active configuration
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE the dev has been disconnected
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_DEVICE_NOT_OPENED The device was not opened
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @post Obtained configuration should be destroyed by usb_host_config_destroy()
 * when no longer needed.
 *
 * @code
#include <usb_host.h>

void inspect_configuration(usb_host_device_h dev)
{
	int ret;
	usb_host_config_h config;

	ret = usb_host_get_active_config(config);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Read the configuration here

	ret = usb_host_config_destroy(config);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_get_active_config(usb_host_device_h dev, usb_host_config_h *config);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Sets a configuration as the currently active on a device.
 * @details Makes one of the configurations currently active on the device.
 * @since_tizen 3.0
 * @param[in] configuration Handle to configuration to be activated
 * @remarks Initially, the device might be configured with any configuration.
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_FOUND Requested configuration does not exist
 * @retval #USB_HOST_ERROR_RESOURCE_BUSY Interfaces are currently claimed
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE The device has been disconnected
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_DEVICE_NOT_OPENED The device was not opened
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>

void select_configuration(usb_host_device_h dev)
{
	int ret, num_configurations;
	usb_host_config_h config;

	ret = usb_host_device_get_num_configurations(dev, &num_configurations);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (int i = 0; i < num_configurations; ++i) {
		ret = usb_host_device_get_config(dev, i, &config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);

		if (interesting(config)) {
			ret = usb_host_set_config(config);
			if (ret != USB_HOST_ERROR_NONE)
				handle_error(ret);
		}

		ret = usb_host_config_destroy(config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);
	}
}
 * @endcode
 */
int usb_host_set_config(usb_host_config_h configuration);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Returns a USB device back to the unconfigured state.
 * @details Make the device not configured with any configuration.
 * @since_tizen 4.0
 * @param[in] dev Device to be unconfigured
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_RESOURCE_BUSY Interfaces are currently claimed
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE Device has been disconnected
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

void unconfigure_device(usb_host_device_h dev)
{
	int ret;

	ret = usb_host_device_unconfigure(dev);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_device_unconfigure(usb_host_device_h dev);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets USB specification release number supported by given device.
 * @details Gets binary-coded decimal USB specification release number.
 * This value is equal to bcdUSB field of device descriptor. See USB specification
 * for more info.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] bcd_usb Bcd release number of USB
 * @remarks Each hexadecimal digit represents a single decimal digit, and there are
 * two major version digits and two minor version digits. For instance, 0x1234 represents
 * version 12.34.
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_usb_spec(usb_host_device_h dev)
{
	int ret, bcd_usb;

	ret = usb_host_device_get_bcd_usb(dev, &bcd_usb);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("USB specification version: %d%d.%d%d\n",
		(bcd_usb << 12) & 0xf,
		(bcd_usb << 8) & 0xf,
		(bcd_usb << 4) & 0xf,
		(bcd_usb << 0) & 0xf,
	);
}
 * @endcode
 */
int usb_host_device_get_bcd_usb(usb_host_device_h dev, int *bcd_usb);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the numerical device class of given USB device.
 * @details Returns the numerical device class, which represents in high level
 * the functionality of the device.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] device_class Device class
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_device_class(usb_host_device_h dev)
{
	int ret, device_class;

	ret = usb_host_device_get_class(dev, &device_class);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Device class: %d\n", device_class);
}
 * @endcode
 */
int usb_host_device_get_class(usb_host_device_h dev, int *device_class);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the numerical subclass of given USB device.
 * @details Returns the numerical device subclass, which refines the class code
 * returned by usb_host_device_get_class().
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] subclass Device subclass
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_device_subclass(usb_host_device_h dev)
{
	int ret, subclass;

	ret = usb_host_device_get_sub_class(dev, &subclass);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Device subclass: %d\n", subclass);
}
 * @endcode
 */
int usb_host_device_get_sub_class(usb_host_device_h dev, int *subclass);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the protocol being used by the given USB device.
 * @details Returns the ID of the protocol, which refines the subclass code
 * returned by usb_host_device_get_sub_class().
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] protocol Device protocol
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_device_protocol_id(usb_host_device_h dev)
{
	int ret, protocol;

	ret = usb_host_device_get_protocol(dev, &protocol);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Device protocol ID: %d\n", protocol);
}
 * @endcode
 */
int usb_host_device_get_protocol(usb_host_device_h dev, int *protocol);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the maximum packet size for endpoint 0 of this device.
 * @details Returns the maximum packet size for endpoint 0 on the given USB device.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] max_packet_size Maximum size of single packet, in bytes
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_device_max_packet_size_0(usb_host_device_h dev)
{
	int ret, max_packet_size;

	ret = usb_host_device_get_max_packet_size_0(dev, &max_packet_size);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Maximum packet size for endpoint 0: %d bytes\n", max_packet_size);
}
 * @endcode
 */
int usb_host_device_get_max_packet_size_0(usb_host_device_h dev, int *max_packet_size);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the numerical vendor identifier of the given USB device.
 * @details Returns the numerical vendor identifier, representing the vendor who produced the device.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] vendor_id Vendor id of @a dev
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_device_vendor_id(usb_host_device_h dev)
{
	int ret, vendor_id;

	ret = usb_host_device_get_id_vendor(dev, &vendor_id);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Vendor ID: %d\n", vendor_id);
}
 * @endcode
 */
int usb_host_device_get_id_vendor(usb_host_device_h dev, int *vendor_id);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the numerical product identifier of the given USB device.
 * @details Returns the numerical product identifier, representing a specific product of a vendor.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] product_id Product id of @a dev
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_device_product_id(usb_host_device_h dev)
{
	int ret, product_id;

	ret = usb_host_device_get_id_product(dev, &vendor_id);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Product ID: %d\n", product_id);
}
 * @endcode
 */
int usb_host_device_get_id_product(usb_host_device_h dev, int *product_id);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets device release number for the given USB device.
 * @details Gets device release number. The number will be returned as a binary-coded decimal.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] device_bcd Device release number
 * @remarks Each hexadecimal digit corresponds to a single decimal digit, so you should expect
 * only digits between 0x0 and 0x9 inclusively. The meaning of the value depends on the device.
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_device_product_id(usb_host_device_h dev)
{
	int ret, device_bcd;

	ret = usb_host_device_get_id_product(dev, &device_bcd);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	int release_number = 0;
	int mult = 1;

	while (device_bcd > 0) {
		release_number += mult * (device_bcd & 0xf);
		device_bcd >>= 4;
		mult *= 10;
	}

	printf("Release number: %d\n", release_number);
}
 * @endcode
 */
int usb_host_device_get_bcd_device(usb_host_device_h dev, int *device_bcd);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the number of configurations for given USB device.
 * @details Returns the number of configurations supported by the device.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] num_configurations Number of configurations for given device
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

void select_configuration(usb_host_device_h dev)
{
	int ret, num_configurations;
	usb_host_config_h config;

	ret = usb_host_device_get_num_configurations(dev, &num_configurations);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (int i = 0; i < num_configurations; ++i) {
		ret = usb_host_device_get_config(dev, i, &config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);

		if (interesting(config)) {
			ret = usb_host_set_config(config);
			if (ret != USB_HOST_ERROR_NONE)
				handle_error(ret);
		}

		ret = usb_host_config_destroy(config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);
	}
}
 * @endcode
 */
int usb_host_device_get_num_configurations(usb_host_device_h dev, int *num_configurations);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Checks whether given USB device is currently opened.
 * @details Returns information about if the device is opened at the moment.
 * @since_tizen 3.0
 * @param[in] dev A device
 * @param[out] is_opened True if device is opened, false otherwise
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void check_if_device_opened(usb_host_device_h dev)
{
	int ret;
	bool is_opened;

	ret = usb_host_is_device_opened(dev, &is_opened);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Device is %s\n", is_opened ? "opened" : "closed");
}
 * @endcode
 */
int usb_host_is_device_opened(usb_host_device_h dev, bool *is_opened);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the manufacturer string for the connected device.
 * @details Gets string describing an open device's manufacturer, in ASCII.
 * @since_tizen 3.0
 * @param[in] dev A handle to opened device
 * @param[in, out] length Pointer to a variable containing the current size of the buffer,
 * which will be modified to actual usage
 * @param[out] data Buffer to store string
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_OVERFLOW There was no space in buffer
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre dev must point to device opened by usb_host_device_open() or usb_host_device_open_with_vid_pid().
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

#define BUFFER_SIZE 128

void print_manufacturer_str(usb_host_device_h dev)
{
	int ret, length = BUFFER_SIZE - 1;
	unsigned char data[BUFFER_SIZE];

	ret = usb_host_device_get_manufacturer_str(dev, &length, data);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	data[length] = '\0';

	printf("Manufacturer: %s\n", data);
}
 * @endcode
 */
int usb_host_device_get_manufacturer_str(usb_host_device_h dev, int *length, unsigned char *data);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the product string for the connected device.
 * @details Gets the product string of an open USB device, in ASCII.
 * @since_tizen 3.0
 * @param[in] dev A handle to opened device
 * @param[in, out] length Pointer to a variable containing the current size of the buffer,
 * which will be modified to actual usage
 * @param[out] data Buffer to store string
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_OVERFLOW There was no space in buffer
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre dev must point to device opened by usb_host_device_open() or usb_host_device_open_with_vid_pid().
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

#define BUFFER_SIZE 128

void print_product_str(usb_host_device_h dev)
{
	int ret, length = BUFFER_SIZE - 1;
	unsigned char data[BUFFER_SIZE];

	ret = usb_host_device_get_product_str(dev, &length, data);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	data[length] = '\0';

	printf("Product: %s\n", data);
}
 * @endcode
 */
int usb_host_device_get_product_str(usb_host_device_h dev, int *length, unsigned char *data);

/**
 * @ingroup CAPI_USB_HOST_DEV_MODULE
 * @brief Gets the serial number string for the connected device.
 * @details Gets the serial number of an open USB device, in ASCII.
 * @since_tizen 3.0
 * @param[in] dev A handle to opened device
 * @param[in, out] length Pointer to a variable containing the current size of the buffer,
 * which will be modified to actual usage
 * @param[out] data Buffer to store string
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_OVERFLOW There was no space in buffer
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre dev must point to device opened by usb_host_device_open() or usb_host_device_open_with_vid_pid().
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

#define BUFFER_SIZE 128

void print_serial_number_str(usb_host_device_h dev)
{
	int ret, length = BUFFER_SIZE - 1;
	unsigned char data[BUFFER_SIZE];

	ret = usb_host_device_get_serial_number_str(dev, &length, data);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	data[length] = '\0';

	printf("Serial number: %s\n", data);
}
 * @endcode
 */
int usb_host_device_get_serial_number_str(usb_host_device_h dev, int *length, unsigned char *data);

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief Gets number of interfaces for given configuration.
 * @details Gets the number of interfaces supported by the provided configuration.
 * @since_tizen 3.0
 * @param[in] config A configuration
 * @param[out] num_interfaces Number of interfaces
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_num_interfaces(usb_host_interface_h interface)
{
	int ret, num_interfaces;

	ret = usb_host_interface_get_num_interfaces(inteface, &num_interfaces);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Number of interfaces: %d\n", num_interfaces);
}
 * @endcode
 */
int usb_host_config_get_num_interfaces(usb_host_config_h config, int *num_interfaces);

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief Checks if the device is self-powered in given configuration.
 * @details Checks if the device in provided configuration is self-powered.
 * @since_tizen 3.0
 * @param[in] config A configuration
 * @param[out] self_powered True if device is self-powered in given configuration,
 * false otherwise
 * @return 0 on success, negative error code otherwise
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_self_powered(usb_host_config_h config)
{
	int ret;
	bool self_powered;

	ret = usb_host_config_is_self_powered(dev, &self_powered);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Configuration is %sself-powered\n", self_powered ? "" : "not ");
}
 * @endcode
 */
int usb_host_config_is_self_powered(usb_host_config_h config, bool *self_powered);

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief Checks if the device supports remote wakeup in given configuration.
 * @details Checks if the device in provided configuration supports remote wakeup.
 * @since_tizen 3.0
 * @param[in] config A configuration
 * @param[out] remote_wakeup True if device supports remote wakeup in given configuration,
 * false otherwise
 * @return 0 on success, negative error code otherwise
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_remote_wakeup(usb_host_config_h config)
{
	int ret;
	bool remote_wakeup;

	ret = usb_host_config_support_remote_wakeup(dev, &remote_wakeup);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Configuration %s remote wakeup\n", self_powered ? "supports" : "does not support");
}
 * @endcode
 */
int usb_host_config_support_remote_wakeup(usb_host_config_h config, bool *remote_wakeup);

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief Gets maximum power in given USB configuration.
 * @details Returns the maximum power drawn by the device in the provided USB configuration, in mA.
 * @since_tizen 3.0
 * @param[in] config A configuration
 * @param[out] max_power Maximum power, in mA
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_remote_wakeup(usb_host_config_h config)
{
	int ret, max_power;

	ret = usb_host_config_get_max_power(dev, &max_power);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Max power: %d mA\n", max_power);
}
 * @endcode
 */
int usb_host_config_get_max_power(usb_host_config_h config, int *max_power);

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief Gets a string describing a USB configuration into given buffer.
 * @details Puts a string representation of the USB configuration into the given buffer.
 * @since_tizen 3.0
 * @param[in] config A configuration
 * @param[in, out] length Pointer to a variable containing the current size of the buffer,
 * which will be modified to actual usage
 * @param[out] data Buffer to store string
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_OVERFLOW There was no space in buffer
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre config must be configuration of device opened by usb_host_device_open() or
 * usb_host_device_open_with_vid_pid()
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

#define BUFFER_SIZE 128

void print_config_str(usb_host_config_h config)
{
	int ret, length = BUFFER_SIZE - 1;
	unsigned char data[BUFFER_SIZE];

	ret = usb_host_device_get_config_str(dev, &length, data);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	data[length] = '\0';

	printf("Configuration description: %s\n", data);
}
 * @endcode
 */
int usb_host_device_get_config_str(usb_host_config_h config, int *length, unsigned char *data);

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief Gets a USB interface from given USB configuration.
 * @details Gets a USB interface from configuration by its index.
 * @since_tizen 3.0
 * @param[in] config Configuration handle
 * @param[in] interface_index index of interface to retrieve (counting from 0)
 * @param[out] interface Interface handle
 * @remarks There is no need to destroy the @a interface handle. It is no longer valid
 * when config is destroyed.
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_FOUND Configuration does not exist
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

#define INTERFACE_INDEX 3

usb_host_interface_h get_interface(usb_host_config_h config)
{
	int ret;
	usb_host_interface_h interface;

	ret = usb_host_config_get_interface(config, INTERFACE_INDEX, &interface);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	return ret;
}
 * @endcode
 */
int usb_host_config_get_interface(usb_host_config_h config, int interface_index, usb_host_interface_h *interface);

/**
 * @ingroup CAPI_USB_HOST_CONFIG_MODULE
 * @brief Frees a configuration struct and its associated data.
 * @details Frees configuration obtained from usb_host_device_get_config().
 * @since_tizen 3.0
 * @param[in] config Configuration to free
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre config must be obtained by usb_host_device_get_config().
 *
 * @code
#include <usb_host.h>

void select_configuration(usb_host_device_h dev)
{
	int ret, num_configurations;
	usb_host_config_h config;

	ret = usb_host_device_get_num_configurations(dev, &num_configurations);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (int i = 0; i < num_configurations; ++i) {
		ret = usb_host_device_get_config(dev, i, &config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);

		if (interesting(config)) {
			ret = usb_host_set_config(config);
			if (ret != USB_HOST_ERROR_NONE)
				handle_error(ret);
		}

		ret = usb_host_config_destroy(config);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);
	}
}
 * @endcode
 */
int usb_host_config_destroy(usb_host_config_h config);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Claims a USB interface on a device for further I/O.
 * @details Claims interface on a device.
 * To perform I/O operations on interface user has to claim it.
 * Remember to call usb_host_release_interface() when communication
 * with the device is finished.
 * @since_tizen 3.0
 * @param[in] interface The bInterfaceNumber of interface to claim
 * @param[in] force Set to true to auto detach kernel driver, set to false to not detach it
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_FOUND Requested interface does not exist
 * @retval #USB_HOST_ERROR_RESOURCE_BUSY Another program or driver has claimed the
 * interface
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE Device has been disconnected
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_DEVICE_NOT_OPENED The device was not opened
 *
 * @code
#include <usb_host.h>

void use_interface(usb_host_interface_h interface)
{
	int ret;

	ret = usb_host_claim_interface(interface, true);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Use the interface here

	ret = usb_host_release_interface(interface, true);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
  */
int usb_host_claim_interface(usb_host_interface_h interface, bool force);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Releases a previously claimed interface on a device.
 * @details Releases interface previously claimed by usb_host_claim_interface().
 * This is a blocking function.
 * @since_tizen 3.0
 * @param[in] interface The bInterfaceNumber of interface to release
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_FOUND Interface was not claimed
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE Device has been disconnected
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_DEVICE_NOT_OPENED The device was not opened
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>

void use_interface(usb_host_interface_h interface)
{
	int ret;

	ret = usb_host_claim_interface(interface, true);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Use the interface here

	ret = usb_host_release_interface(interface, true);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_release_interface(usb_host_interface_h interface);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Gets the "number" of the given USB interface handle.
 * @details Gets the index of the interface, as passed to usb_host_config_get_interface().
 * @since_tizen 3.0
 * @param[in] interface An interface
 * @param[out] number Number of given interface
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_interface_index(usb_host_interface_h interface)
{
	int ret, number;

	ret = usb_host_interface_get_number(dev, &number);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Interface index: %d\n", number);
}
 * @endcode
 */
int usb_host_interface_get_number(usb_host_interface_h interface, int *number);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Gets the number of endpoints in given USB interface.
 * @details Gets the number of endpoints supported by the provided interface.
 * @since_tizen 3.0
 * @param[in] interface An interface
 * @param[out] num_endpoints Number of endpoints in @a interface
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_num_endpoints(usb_host_interface_h interface)
{
	int ret, num_endpoints;

	ret = usb_host_interface_get_num_endpoints(inteface, &num_endpoints);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Number of endpoints: %d\n", num_endpoints);
}
 * @endcode
 */
int usb_host_interface_get_num_endpoints(usb_host_interface_h interface, int *num_endpoints);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Gets an endpoint via index from given USB interface.
 * @details Returns the correct endpoint of the provided interface by its index.
 * @since_tizen 3.0
 * @param[in] interface Interface handle
 * @param[in] ep_index index of endpoint to retrieve (counting from 0)
 * @param[out] ep Endpoint handle
 * @remarks @a ep handle is no longer valid when config will be destroyed. There is no need to destroy it,
 * it is done automatically when the configuration is destroyed.
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

#define ENDPOINT_INDEX 3

usb_host_endpoint_h get_endpoint(usb_host_interface_h interface)
{
	int ret;
	usb_host_endpoint_h ep;

	ret = usb_host_interface_get_endpoint(interface, ENDPOINT_INDEX, &ep);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	return ret;
}
 * @endcode
 */
int usb_host_interface_get_endpoint(usb_host_interface_h interface, int ep_index,
		usb_host_endpoint_h *ep);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Sets an alternative setting for given USB interface.
 * @details Selects an alternative setting represented by the provided index.
 * @since_tizen 3.0
 * @param[in] interface Interface handle
 * @param[in] altsetting Index of new alternative setting for given interface
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

void set_0_altsetting(usb_host_interface_h interface)
{
	int ret = usb_host_interface_set_altsetting(interface, 0);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_interface_set_altsetting(usb_host_interface_h interface, int altsetting);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Gets current alternative setting from an interface.
 * @details Retrieves the currently selected alternative setting.
 * @since_tizen 4.0
 * @param[in] interface Interface handle
 * @param[out] altsetting Index of alternative setting set for the given interface
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_current_altsetting(usb_host_interface_h interface)
{
	int ret, altsetting;

	ret = usb_host_interface_get_altsetting(interface, &altsetting);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Current alternative setting index: %d\n");
}
 * @endcode
 */
int usb_host_interface_get_altsetting(usb_host_interface_h interface, int *altsetting);

/**
 * @ingroup CAPI_USB_HOST_INTERFACE_MODULE
 * @brief Gets a string describing the given USB interface into a buffer.
 * @details Returns a string representation of the given USB interface.
 * @since_tizen 3.0
 * @param[in] interface An interface
 * @param[in, out] length Pointer to a variable containing the current size of the buffer,
 * which will be modified to actual usage
 * @param[out] data Buffer to store string
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_OVERFLOW There was no space in buffer
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre device which interface is part of must be opened by usb_host_device_open() or
 * usb_host_device_open_with_vid_pid()
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

#define BUFFER_SIZE 128

void print_interface_str(usb_host_interface_h interface)
{
	int ret, length = BUFFER_SIZE - 1;
	unsigned char data[BUFFER_SIZE];

	ret = usb_host_interface_get_str(dev, &length, data);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	data[length] = '\0';

	printf("Interface: %s\n", data);
}
 * @endcode
 */
int usb_host_interface_get_str(usb_host_interface_h interface, int *length,
		unsigned char *data);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets the "number" of the given USB endpoint handle.
 * @details Gets the index of the endpoint handle, as passed to usb_host_interface_get_endpoint().
 * @since_tizen 3.0
 * @param[in] ep An endpoint
 * @param[out] number Number of given endpoint
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_endpoint_index(usb_host_endpoint_h ep)
{
	int ret, number;

	ret = usb_host_endpoint_get_number(ep, &number);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Endpoint index: %d\n", number);
}
 * @endcode
 */
int usb_host_endpoint_get_number(usb_host_endpoint_h ep, int *number);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets the direction of the given USB endpoint handle.
 * @details Returns the direction of the provided USB endpoint handle.
 * @since_tizen 3.0
 * @param[in] ep An endpoint
 * @param[out] direction Direction of endpoint (a value from enum #usb_host_endpoint_direction_e)
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_endpoint_direction(usb_host_endpoint_h ep)
{
	int ret;
	usb_host_endpoint_direction_e direction;

	ret = usb_host_endpoint_get_direction(ep, &direction);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	switch (direction) {
	case USB_HOST_DIRECTION_IN:
		printf("Input endpoint\n");
		break;
	case USB_HOST_DIRECTION_OUT:
		printf("Output endpoint\n");
		break;
	}
}
 * @endcode
 */
int usb_host_endpoint_get_direction(usb_host_endpoint_h ep, usb_host_endpoint_direction_e *direction);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets transfer type of the given USB endpoint handle.
 * @details Returns the transfer type of the provided USB endpoint handle.
 * @since_tizen 3.0
 * @param[in] ep An endpoint
 * @param[out] transfer_type Transfer type (a value from enum #usb_host_transfer_type_e)
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_endpoint_transfer_type(usb_host_endpoint_h ep)
{
	int ret;
	usb_host_transfer_type_e transport_type;

	ret = usb_host_endpoint_get_transfer_type(ep, &transport_type);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	switch (transfer_type) {
	case USB_HOST_TRANSFER_TYPE_CONTROL:
		printf("Control endpoint\n");
		break;
	case USB_HOST_TRANSFER_TYPE_ISOCHRONOUS:
		printf("Isochronous endpoint\n");
		break;
	case USB_HOST_TRANSFER_TYPE_BULK:
		printf("Bulk endpoint\n");
		break;
	case USB_HOST_TRANSFER_TYPE_INTERRUPT:
		printf("Interrupt endpoint\n");
		break;
	}
}
 * @endcode
 */
int usb_host_endpoint_get_transfer_type(usb_host_endpoint_h ep, usb_host_transfer_type_e *transfer_type);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets synchronization type of given USB endpoint handle.
 * @details Returns the synchronization type of provided USB endpoint handle.
 * @since_tizen 3.0
 * @param[in] ep An endpoint
 * @param[out] synch_type Synch type (a value from enum #usb_host_iso_sync_type_e)
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_endpoint_synch_type(usb_host_endpoint_h ep)
{
	int ret;
	usb_host_iso_sync_type_e synch_type;

	ret = usb_host_endpoint_get_synch_type(ep, &synch_type);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	switch (synch_type) {
	case USB_HOST_ISO_SYNC_TYPE_NONE:
		printf("No synchronization\n");
		break;
	case USB_HOST_ISO_SYNC_TYPE_ASYNC:
		printf("Asynchronous synchronization\n");
		break;
	case USB_HOST_ISO_SYNC_TYPE_ADAPTIVE:
		printf("Adaptive synchronization\n");
		break;
	case USB_HOST_ISO_SYNC_TYPE_SYNC:
		printf("Synchronous synchronization\n");
		break;
	}
}
 * @endcode
 */
int usb_host_endpoint_get_synch_type(usb_host_endpoint_h ep, usb_host_iso_sync_type_e *synch_type);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets the usage type of the given USB endpoint handle.
 * @details Returns the usage type of the provided USB endpoint handle.
 * @since_tizen 3.0
 * @param[in] ep An endpoint
 * @param[out] usage_type Usage type (a value from enum #usb_host_usage_type_e)
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_endpoint_usage_type(usb_host_endpoint_h ep)
{
	int ret;
	usb_host_usage_type_e usage_type;

	ret = usb_host_endpoint_get_usage_type(ep, &usage_type);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	switch (usage_type) {
	case USB_HOST_USAGE_TYPE_DATA:
		printf("Data endpoint\n");
		break;
	case USB_HOST_USAGE_TYPE_FEEDBACK:
		printf("Feedback endpoint\n");
		break;
	case USB_HOST_USAGE_TYPE_IMPLICIT:
		printf("Implicit feedback data endpoint\n");
		break;
	}
}
 * @endcode
 */
int usb_host_endpoint_get_usage_type(usb_host_endpoint_h ep, usb_host_usage_type_e *usage_type);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets max packet size from given USB endpoint handle.
 * @details Gets max packet size from given USB endpoint handle.
 * @since_tizen 3.0
 * @param[in] ep An endpoint
 * @param[out] max_packet_size Max packet size, in bytes
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_endpoint_max_packet_size(usb_host_endpoint_h ep)
{
	int ret, max_packet_size;

	ret = usb_host_endpoint_get_max_packet_size(dev, &max_packet_size);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Maximum packet size: %d bytes\n", max_packet_size);
}
 * @endcode
 */
int usb_host_endpoint_get_max_packet_size(usb_host_endpoint_h ep, int *max_packet_size);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets interval for polling endpoint for data transfers.
 * @details Returns the interval for polling endpoint for data transfers.
 * @since_tizen 3.0
 * @param[in] ep An endpoint
 * @param[out] interval Interval for polling, in frame counts (refer to USB protocol specification)
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_endpoint_interval(usb_host_endpoint_h ep)
{
	int ret, interval;

	ret = usb_host_endpoint_get_interval(dev, &interval);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Polling interval: %d frames\n", interval);
}
 * @endcode
 */
int usb_host_endpoint_get_interval(usb_host_endpoint_h ep, int *interval);

/* sync IO */

/**
 * @ingroup CAPI_USB_HOST_SYNCIO_MODULE
 * @brief Performs unidirectional USB control transfer on given device.
 * @details For more explanation about the values please refer to USB protocol specification
 * @since_tizen 3.0
 * @remarks The wValue, wIndex and wLength fields values should be given in host-endian
 * byte order.
 * @param[in] dev Device handle
 * @param[in] bm_request_type bmRequestType type field for the setup packet
 * @param[in] b_request bRequest field for the setup packet
 * @param[in] w_value wValue field for the setup packet
 * @param[in] w_index wIndex field for the setup packet
 * @param[in] data Suitably-sized data buffer for either input or output
 * (depending on direction bits within bmRequestType)
 * @param[in] w_length wLength field for the setup packet. The data buffer should
 * be at least this size
 * @param[in] timeout Timeout (in milliseconds) that this function should wait
 * before giving up due to no response being received. For an unlimited
 * @param[out] transferred Number of transferred bytes
 * timeout, 0 value should be used.
 * @return  0 on success, negative error code otherwise
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_TIMED_OUT Transfer timed out
 * @retval #USB_HOST_ERROR_BROKEN_PIPE Control request was not supported by the device
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE The device has been disconnected
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_DEVICE_NOT_OPENED The device was not opened
 * @retval #USB_HOST_ERROR_OVERFLOW Device offered more data
 * @pre dev must point to device opened by usb_host_device_open() or usb_host_device_open_with_vid_pid()
 *
 * @code
#include <usb_host.h>

#define REQUEST_TYPE_FOR_WRITING 0x40
#define REQUEST_TYPE_FOR_READING 0xC0
#define REQUEST 0x04
#define VALUE 0x20
#define INDEX 5

#define BUFFER_SIZE 16

void perform_sync_operations(usb_host_device_h dev)
{
	int ret;
	unsigned char data[BUFFER_SIZE];
	int transferred_write, transferred_read;

	// Read some data synchronously, with a 1 s timeout
	ret = usb_host_control_transfer(dev, REQUEST_TYPE_FOR_READING, REQUEST, VALUE,
		INDEX, data, BUFFER_SIZE, 1000, &transferred_read);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Write back the same data, with also 1 s timeout
	ret = usb_host_control_transfer(dev, REQUEST_TYPE_FOR_WRITING, REQUEST, VALUE,
		INDEX, data, transferred_read, 1000, &transferred_write);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_control_transfer(usb_host_device_h dev,
	uint8_t bm_request_type, uint8_t b_request, uint16_t w_value, uint16_t w_index,
	unsigned char *data, uint16_t w_length, unsigned int timeout, int *transferred);

/**
 * @ingroup CAPI_USB_HOST_SYNCIO_MODULE
 * @brief Performs unidirectional transfer on given endpoint.
 * @details Performs a USB transfer on given endpoint. Direction of transfer is
 * determined by the endpoint.
 * @since_tizen 3.0
 * @param[in] ep Endpoint handle
 * @param[in] data Suitably-sized data buffer for either input or output
 * (depending on endpoint)
 * @param[in] length For writes, the number of bytes from data to be sent, for
 * reads the maximum number of bytes to receive into the data buffer
 * @param[out] transferred number of bytes actually transferred
 * @param[in] timeout Timeout (in milliseconds) that this function should wait
 * before giving up due to no response being received (for an unlimited
 * timeout 0 value should be used)
 * @return 0 on success (and populates @a transferred), negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_TIMED_OUT Transfer timed out
 * @retval #USB_HOST_ERROR_BROKEN_PIPE Endpoint halted
 * @retval #USB_HOST_ERROR_OVERFLOW Device offered more data
 * @retval #USB_HOST_ERROR_NO_SUCH_DEVICE Device has been disconnected
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_DEVICE_NOT_OPENED The device was not opened
 * @pre ep must be a valid endpoint received from usb_host_interface_get_endpoint().
 * @pre ep must be an endpoint of device opened by usb_host_device_open() or
 * usb_host_device_open_with_vid_pid().
 *
 * @code
#include <usb_host.h>

#define REQUEST_TYPE_FOR_WRITING 0x40
#define REQUEST_TYPE_FOR_READING 0xC0
#define REQUEST 0x04
#define VALUE 0x20
#define INDEX 5

#define BUFFER_SIZE 16

void perform_sync_operations(usb_host_endpoint_h write_endpoint, usb_host_endpoint_h read_endpoint)
{
	int ret;
	unsigned char data[BUFFER_SIZE];
	int transferred_write, transferred_read;

	// Read some data synchronously, with a 1 s timeout
	ret = usb_host_transfer(read_endpoint, data, BUFFER_SIZE, &transferred_read, 1000);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Write back the same data, with also 1 s timeout
	ret = usb_host_transfer(write_endpoint, data, transferred_read, &transferred_write, 1000);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer(usb_host_endpoint_h ep, unsigned char *data, int length,
		int *transferred, unsigned int timeout);

/**
 * @ingroup CAPI_USB_HOST_HOTPLUG_MODULE
 * @brief Sets a callback function to be invoked when a device is connected or disconnected.
 * @details After this call, a callback function will be invoked whenever a device
 * is connected or disconnected.
 * @since_tizen 4.0
 * @param[in] ctx Context handle
 * @param[in] cb The callback function to be registered
 * @param[in] event Event that will trigger registered callback
 * @param[in] user_data The user data to be passed to the callback function
 * @param[out] handle Handle of the registered callback
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory
 *
 * @code
#include <usb_host.h>

void hotplug_callback(usb_host_device_h dev, void *user_data)
{
	struct our_data_type *callback_data = user_data;

	process_new_device(data);
}

usb_host_hotplug_h set_callback(usb_host_context_h ctx)
{
	struct our_data_type *callback_data = create_callback_data();
	int ret;
	usb_host_hotplug_h handle;

	ret = usb_host_set_hotplug_cb(ctx, hotplug_callback, USB_HOST_HOTPLUG_EVENT_ATTACH, callback_data, &handle);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	return handle;
}
 * @endcode
 */
int usb_host_set_hotplug_cb(usb_host_context_h ctx, usb_host_hotplug_cb cb, usb_host_hotplug_event_e event, void *user_data, usb_host_hotplug_h *handle);

/**
 * @ingroup CAPI_USB_HOST_HOTPLUG_MODULE
 * @brief Unsets the (dis)connection hotplug callback function.
 * @details After this call, the callback will not be connected anymore.
 * @since_tizen 4.0
 * @param[in] handle Handle of the callback to be unregistered
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 *
 * @code
#include <usb_host.h>

void unset_callback(usb_host_hotplug_h handle, struct our_data_type *callback_data)
{
	int ret = usb_host_unset_hotplug_cb(handle);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Make sure the callback is not running in parallel!
	destroy_callback_data(callback_data);
}
 * @endcode
 */
int usb_host_unset_hotplug_cb(usb_host_hotplug_h handle);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Prepares an asynchronous USB transfer and provides a handle to it.
 * @details This function prepares transfer handle for asynchronous communication.
 * Transfer handle can be used for multiple transfers after this initialization.
 * @since_tizen 5.0
 * @param[in] ep Endpoint handle
 * @param[in] callback Callback to be called when transfer is finished
 * @param[in] data Suitably-sized data buffer, similar to synchronized transfer
 * @param[in] length For writes, the number of bytes from data to be sent; for
 * reads, the maximum number of bytes to receive into the data buffer
 * @param[in] user_data Pointer to data which will be passed to callback function later on
 * @param[in] timeout Timeout (in milliseconds) that transfer should wait before giving up
 * due to no response being received (for an unlimited timeout use value of 0)
 * @param[out] transfer Transfer handle
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory
 * @pre @a ep must be a valid endpoint received from usb_host_interface_get_endpoint().
 * @post @a transfer should be destroyed by calling usb_host_transfer_destroy() when it's no longer needed.
 *
 * @code
#include <usb_host.h>

static unsigned char data_to_send[] = { 0x1, 0x2, 0x3, 0x4 };

#define READ_BUFFER_SIZE 64

void write_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	// Handle exit
	usb_host_transfer_destroy(transfer);
}

void read_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	unsigned char *data = user_data;
	// Use the data, handle exit
	free(data);
	usb_host_transfer_destroy(transfer);
}

void create_transfers(usb_host_endpoint_h write_endpoint, usb_host_endpoint_h read_endpoint)
{
	int ret;
	usb_host_transfer_h write_transfer, read_transfer;

	unsigned char *read_buffer = calloc(READ_BUFFER_SIZE, 1);
	if (!read_buffer)
		handle_oom();

	ret = usb_host_create_transfer(write_endpoint, write_completion_cb,
		data_to_send, sizeof(data_to_send) / sizeof(data_to_send[0]),
		NULL, 0, &write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	ret = usb_host_create_transfer(read_endpoint, read_completion_cb,
		read_buffer, READ_BUFFER_SIZE, NULL, 0, &write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Initialize transfers as you need

	// Finally:
	ret = usb_host_transfer_submit(write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
	ret = usb_host_transfer_submit(read_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_create_transfer(usb_host_endpoint_h ep, usb_host_transferred_cb callback,
	unsigned char *data, int length, void *user_data, unsigned int timeout, usb_host_transfer_h *transfer);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Prepares an asynchronous USB isochronous transfer and provides a handle to it.
 * @details This function prepares transfer handle for asynchronous communication.
 * Usage is similar to usb_host_create_transfer(), except this function is intended for
 * isochronous endpoints. Transfer handle can be used for multiple transfers after this initialization.
 * Note however, that this function needs to allocate memory for @a num_iso_packets isochronous packets and
 * it will be the limit for number of them in this transfer.
 * @since_tizen 5.0
 * @param[in] ep Endpoint handle
 * @param[in] callback Callback to be called when transfer is finished
 * @param[in] data Suitably-sized data buffer, similar to synchronized transfer
 * @param[in] length For writes, the number of bytes from data to be sent; for
 * reads, the maximum number of bytes to receive into the data buffer
 * @param[in] num_iso_packets Number of isochronous packets
 * @param[in] user_data Pointer to data which will be passed to callback function later on
 * @param[in] timeout Timeout (in milliseconds) that transfer should wait before giving up
 * due to no response being received (for an unlimited timeout use value of 0)
 * @param[out] transfer Transfer handle
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory
 * @pre @a ep must be a valid endpoint received from usb_host_interface_get_endpoint().
 * @post @a transfer should be destroyed by calling usb_host_transfer_destroy() when it's no longer needed.
 *
 * @code
#include <usb_host.h>

static unsigned char data_to_send[] = { 0x1, 0x2, 0x3, 0x4 };

#define READ_BUFFER_SIZE 64
#define NUM_ISO 5

void write_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	// Handle exit
	usb_host_transfer_destroy(transfer);
}

void read_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	unsigned char *data = user_data;
	// Use the data, handle exit
	free(data);
	usb_host_transfer_destroy(transfer);
}

void create_transfers(usb_host_endpoint_h write_endpoint, usb_host_endpoint_h read_endpoint)
{
	int ret;
	usb_host_transfer_h write_transfer, read_transfer;

	unsigned char *read_buffer = calloc(READ_BUFFER_SIZE, 1);
	if (!read_buffer)
		handle_oom();

	ret = usb_host_create_isochronous_transfer(write_endpoint, write_completion_cb,
		data_to_send, sizeof(data_to_send) / sizeof(data_to_send[0]),
		NUM_ISO, NULL, 0, &write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	ret = usb_host_create_isochronous_transfer(read_endpoint, read_completion_cb,
		read_buffer, READ_BUFFER_SIZE, NUM_ISO, NULL, 0, &write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Initialize transfers as you need

	// Finally:
	ret = usb_host_transfer_submit(write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
	ret = usb_host_transfer_submit(read_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_create_isochronous_transfer(usb_host_endpoint_h ep, usb_host_transferred_cb callback,
	unsigned char *data, int length, unsigned int num_iso_packets, void *user_data, unsigned int timeout, usb_host_transfer_h *transfer);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Prepares an asynchronous USB control transfer and provides a handle to it.
 * @details This function prepares control transfer handle. Transfer handle can
 * be used for multiple transfers after this initialization. Note, that first
 * 8 bytes of data buffer are interpreted as control setup packet. You may use
 * usb_host_control_transfer_set_* functions to set the setup packet fields.
 * @since_tizen 5.0
 * @param[in] dev Device handle
 * @param[in] callback Callback to be called when transfer is finished
 * @param[in] data Suitably-sized data buffer
 * @param[in] length For writes, the number of bytes from data to be sent, for
 * reads, the maximum number of bytes to receive into the data buffer
 * @param[in] user_data Pointer to data which will be passed to callback function later on
 * @param[in] timeout Timeout (in milliseconds) that transfer should wait before giving up
 * due to no response being received (for an unlimited timeout use value of 0)
 * @param[out] transfer Transfer handle
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory
 * @pre @a dev must be an opened device handle
 * @post @a transfer should be destroyed by calling usb_host_transfer_destroy() when it's no longer needed.
 *
 * @code
#include <usb_host.h>

static unsigned char data_to_send[] = { 0x1, 0x2, 0x3, 0x4 };

#define READ_BUFFER_SIZE 64

void write_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	// Handle exit
	usb_host_transfer_destroy(transfer);
}

void read_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	unsigned char *data = user_data;
	// Use the data, handle exit
	free(data);
	usb_host_transfer_destroy(transfer);
}

void create_transfers(usb_host_device_h dev)
{
	int ret;
	usb_host_transfer_h write_transfer, read_transfer;

	unsigned char *read_buffer = calloc(READ_BUFFER_SIZE, 1);
	if (!read_buffer)
		handle_oom();

	ret = usb_host_create_control_transfer(dev, write_completion_cb,
		data_to_send, sizeof(data_to_send) / sizeof(data_to_send[0]),
		NULL, 0, &write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	ret = usb_host_create_control_transfer(dev, read_completion_cb,
		read_buffer, READ_BUFFER_SIZE, NULL, 0, &write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	// Initialize transfers as you need, also setting the correct
	// transfer types for each transfer

	// Finally:
	ret = usb_host_transfer_submit(write_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
	ret = usb_host_transfer_submit(read_transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_create_control_transfer(usb_host_device_h dev, usb_host_transferred_cb callback,
	unsigned char *data, int length, void *user_data, unsigned int timeout, usb_host_transfer_h *transfer);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets an endpoint for asynchronous transfer for a transfer handle.
 * @details This function changes the endpoint on which given transfer is performed. Next submissions will be
 * performed on this endpoint.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] ep An endpoint handle
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_transfer() or usb_host_create_isochronous_transfer().
 *
 * @code
#include <usb_host.h>

void change_endpoint(usb_host_transfer_h transfer, usb_host_endpoint_h ep)
{
	int ret = usb_host_transfer_set_ep(transfer, ep);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer_set_ep(usb_host_transfer_h transfer, usb_host_endpoint_h ep);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets a callback for asynchronous transfer for a transfer handle.
 * @details This function changes the callback to be called on transfer completion.
 * @since_tizen 5.0
 * @remarks The user_data provided to the callback does not change.
 * @param[in] transfer A transfer handle
 * @param[in] callback A callback function
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_transfer() or usb_host_create_isochronous_transfer().
 *
 * @code
#include <usb_host.h>

void transfer_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	// Handle exit
	usb_host_transfer_destroy(transfer);
}

void change_callback(usb_host_transfer_h transfer)
{
	int ret = usb_host_transfer_set_callback(transfer, transfer_completion_cb);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer_set_callback(usb_host_transfer_h transfer, usb_host_transferred_cb callback);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets data buffer for asynchronous transfer for a transfer handle.
 * @details This function changes the data buffer used for this transfer.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] data A data buffer
 * @param[in] length Length of data buffer
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_transfer() or usb_host_create_isochronous_transfer().
 *
 * @code
#include <usb_host.h>

#define BUFFER_SIZE 4096

void set_data_buffer(usb_host_transfer_h transfer)
{
	unsigned char *data = calloc(BUFFER_SIZE, 1);
	if (!data)
		handle_oom();

	int ret = usb_host_transfer_set_data(transfer, data, BUFFER_SIZE);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer_set_data(usb_host_transfer_h transfer, unsigned char *data, int length);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets timeout for asynchronous transfer for a transfer handle.
 * @details This function changes the timeout after which transfer will be stopped due to
 * no response being received.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] timeout A timeout in milliseconds
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_transfer() or usb_host_create_isochronous_transfer().
 *
 * @code
#include <usb_host.h>

void set_1_second_timeout(usb_host_transfer_h transfer)
{
	int ret = usb_host_transfer_set_timeout(transfer, 1000);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer_set_timeout(usb_host_transfer_h transfer, unsigned int timeout);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets number of isochronous packet for isochronous transfer.
 * @details This function changes the number of isochronous packets in transfer.
 * This parameter affects only isochronous transfers (i.e. transfers on isochronous endpoints).
 * Use usb_host_endpoint_get_transfer_type() for checking types of your endpoints.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] num_iso_packets Number of isochronous packets in this transfer
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_OUT_OF_MEMORY Out of memory (too many packets)
 * @pre @a transfer should be created by usb_host_create_transfer() or usb_host_create_isochronous_transfer().
 *
 * @code
#include <usb_host.h>

void set_4_packets(usb_host_transfer_h transfer)
{
	int ret = usb_host_transfer_set_num_iso_packets(transfer, 4);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer_set_num_iso_packets(usb_host_transfer_h transfer, unsigned int num_iso_packets);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets request type for control transfer setup packet.
 * @details Switches the request type, which will be used for the control transfer setup packet.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] bm_request_type bmRequestType type field for the setup packet
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_control_transfer().
 *
 * @code
#include <usb_host.h>

#define REQUEST_TYPE_FOR_WRITING 0x40
#define REQUEST_TYPE_FOR_READING 0xC0

void set_request_types(usb_host_transfer_h write_transfer, usb_host_transfer_h read_transfer)
{
	int ret = usb_host_control_transfer_set_request_type(write_transfer, REQUEST_TYPE_FOR_WRITING);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	int ret = usb_host_control_transfer_set_request_type(read_transfer, REQUEST_TYPE_FOR_READING);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_control_transfer_set_request_type(usb_host_transfer_h transfer, uint8_t bm_request_type);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets request field for control transfer setup packet.
 * @details Switches the request code, which will be used for the control transfer setup packet.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] b_request Request field for the setup packet
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_control_transfer().
 *
 * @code
#include <usb_host.h>

#define REQUEST 0x04

void set_request(usb_host_transfer_h transfer)
{
	int ret = usb_host_control_transfer_set_request(transfer, REQUEST);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_control_transfer_set_request(usb_host_transfer_h transfer, uint8_t b_request);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets w_value field for control transfer setup packet.
 * @details Sets the value of the w_value parameter of the control transfer setup packet.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] w_value wValue field for the setup packet
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_control_transfer().
 *
 * @code
#include <usb_host.h>

#define VALUE 0x20

void set_value(usb_host_transfer_h transfer)
{
	int ret = usb_host_control_transfer_set_value(transfer, VALUE);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_control_transfer_set_value(usb_host_transfer_h transfer, uint16_t w_value);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets w_index field for control transfer setup packet.
 * @details Sets the value of the w_index parameter of the control transfer setup packet.
 * @since_tizen 5.0
 * @param[in] transfer A transfer handle
 * @param[in] w_index wIndex field for the setup packet
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be created by usb_host_create_control_transfer().
 *
 * @code
#include <usb_host.h>

#define INDEX 5

void set_index(usb_host_transfer_h transfer)
{
	int ret = usb_host_control_transfer_set_index(transfer, INDEX);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_control_transfer_set_index(usb_host_transfer_h transfer, uint16_t w_index);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets the status of a transfer from a transfer handle.
 * @details Gets the transfer status, specified as a value of the #usb_host_transfer_status_e type.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle
 * @param[out] status Status of this transfer
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_transfer_status(usb_host_transfer_h transfer)
{
	int ret, status;

	ret = usb_host_transfer_get_status(transfer, &status);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	switch (status) {
	case USB_HOST_TRANSFER_COMPLETED:
		printf("Transfer completed successfully\n");
	default:
		printf("Transfer not completed successfully\n");
	}
}
 * @endcode
 */
int usb_host_transfer_get_status(usb_host_transfer_h transfer, int *status);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets the data from the asynchronous USB transfer.
 * @details Borrows a handle to the transfer data from a transfer handle.
 * @since_tizen 5.0
 * @remarks @a data is part of the transfer object and should not be released
 * separately. It should not be accessed after @a transfer is destroyed.
 * @param[in] transfer Transfer handle
 * @param[out] data Data buffer of this transfer
 * @param[out] actual_length Actual length of transferred data
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_transfer_data(usb_host_transfer_h transfer)
{
	int ret;
	char *data;
	unsigned actual_length;

	ret = usb_host_transfer_get_data(transfer, &data, &actual_length);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (unsigned i = 0; i < actual_length; ++i)
		printf("%02x", data[i]);
	printf("\n");
}
 * @endcode
 */
int usb_host_transfer_get_data(usb_host_transfer_h transfer, unsigned char **data, unsigned int *actual_length);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets the *control* data from the asynchronous USB transfer.
 * @details Borrows a handle to the *control* transfer data from a transfer handle.
 * @since_tizen 5.0
 * @remarks @a data is part of the transfer object and should not be released
 * separately. It should not be accessed after @a transfer is destroyed.
 * @param[in] transfer Control transfer handle
 * @param[out] data Data buffer of this transfer
 * @param[out] actual_length Actual length of transferred data
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_control_transfer_data(usb_host_transfer_h transfer)
{
	int ret;
	char *data;
	unsigned actual_length;

	ret = usb_host_control_transfer_get_data(transfer, &data, &actual_length);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (unsigned i = 0; i < actual_length; ++i)
		printf("%02x", data[i]);
	printf("\n");
}
 * @endcode
 */
int usb_host_control_transfer_get_data(usb_host_transfer_h transfer, unsigned char **data, unsigned int *actual_length);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets nominal length of data buffer from a transfer handle.
 * @details This functions gets length that was set for data buffer, not the actual transferred data length.
 * For length of transferred data see usb_host_transfer_get_data().
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle
 * @param[out] length Length of data buffer for this transfer
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_buffer_length(usb_host_transfer_h transfer)
{
	int ret;
	unsigned length;

	ret = usb_host_transfer_get_length(transfer, &length);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Buffer length: %d bytes\n", length);
}
 * @endcode
 */
int usb_host_transfer_get_length(usb_host_transfer_h transfer, unsigned int *length);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets number of isochronous packets for this transfer.
 * @details Returns the number of isochronous packets reserved for this transfer.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle
 * @param[out] num_iso_packets Number of isochronous packets
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_num_iso_packets(usb_host_transfer_h transfer)
{
	int ret;
	unsigned num_iso_packets;

	ret = usb_host_transfer_get_num_iso_packets(transfer, &num_iso_packets);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	printf("Isochronous packets capacity: %d packets\n", num_iso_packets);
}
 * @endcode
 */
int usb_host_transfer_get_num_iso_packets(usb_host_transfer_h transfer, unsigned int *num_iso_packets);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Sets an isochronous individual packet length for a transfer handle.
 * @details Sets the length of an packet within an isochronous transfer.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle
 * @param[in] packet_number Number of isochronous packet
 * @param[in] length Length of the packet handle
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_FOUND Packet of given number not found
 * @retval #USB_HOST_ERROR_OVERFLOW Not enough space for this packet in data buffer
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void set_iso_packets_lengths(usb_host_transfer_h transfer)
{
	int ret, num_iso_packets;

	ret = usb_host_transfer_get_num_iso_packets(transfer, &num_iso_packets);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (unsigned i = 0; i < num_iso_packets; ++i) {
		// Set the 1st packet length to 1, 2nd packet length to 2 and so on.
		ret = usb_host_transfer_set_iso_packet_length(transfer, i, i + 1);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);
	}
}
 * @endcode
 */
int usb_host_transfer_set_iso_packet_length(usb_host_transfer_h transfer, unsigned int packet_number, int length);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets an isochronous packet status from a transfer handle.
 * @details Gets the status of the packet, specified as a value of the #usb_host_transfer_status_e type.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle
 * @param[in] packet_number Number of isochronous packet
 * @param[out] status Status of selected packet
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_FOUND Packet of given number not found
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_iso_packets_lengths(usb_host_transfer_h transfer)
{
	int ret, status;
	unsigned num_iso_packets;

	ret = usb_host_transfer_get_num_iso_packets(transfer, &num_iso_packets);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (unsigned i = 0; i < num_iso_packets; ++i) {
		ret = usb_host_transfer_get_iso_packet_status(transfer, i, &status);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);

		switch (status) {
		case USB_HOST_TRANSFER_COMPLETED:
			printf("Packet #%d completed successfully\n", i);
		default:
			printf("Transfer #%d not completed successfully\n", i);
		}
	}
}
 * @endcode
 */
int usb_host_transfer_get_iso_packet_status(usb_host_transfer_h transfer, unsigned int packet_number, int *status);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets data buffer of isochronous packet from a transfer handle.
 * @details Gets the buffer underlying the isochronous packet.
 * @since_tizen 5.0
 * @remarks @a data is part of the transfer object and should not be released
 * separately. It should not be accessed after @a transfer is destroyed.
 * @param[in] transfer Transfer handle
 * @param[in] packet_number Number of isochronous packet
 * @param[out] data Data buffer for this packet
 * @param[out] actual_length Length of transferred data
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_FOUND Packet of given number not found
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_packets_data(usb_host_transfer_h transfer)
{
	int ret;
	char *data;
	unsigned actual_length;

	ret = usb_host_transfer_get_num_iso_packets(transfer, &num_iso_packets);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (unsigned i = 0; i < num_iso_packets; ++i) {
		printf("Packet #%d data: ", i + 1);

		ret = usb_host_transfer_get_data(transfer, &data, &actual_length);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);

		for (unsigned i = 0; i < actual_length; ++i)
			printf("%02x", data[i]);
		printf("\n");
	}
}
 * @endcode
 */
int usb_host_transfer_get_iso_packet_data(usb_host_transfer_h transfer, unsigned int packet_number, unsigned char **data, int *actual_length);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Gets isochronous packet data buffer length from a transfer handle.
 * @details Gets the length of the buffer underlying the isochronous packet.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle
 * @param[in] packet_number Number of isochronous packet
 * @param[out] length Length of data buffer
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @retval #USB_HOST_ERROR_NOT_FOUND Packet of given number not found
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_iso_packets_lengths(usb_host_transfer_h transfer)
{
	int ret, length;
	unsigned num_iso_packets;

	ret = usb_host_transfer_get_num_iso_packets(transfer, &num_iso_packets);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	for (unsigned i = 0; i < num_iso_packets; ++i) {
		ret = usb_host_transfer_get_iso_packet_length(transfer, i, &length);
		if (ret != USB_HOST_ERROR_NONE)
			handle_error(ret);

		printf("Packet #%d size: %d bytes\n", i + 1, length);
	}
}
 * @endcode
 */
int usb_host_transfer_get_iso_packet_length(usb_host_transfer_h transfer, unsigned int packet_number, int *length);

/**
 * @ingroup CAPI_USB_HOST_ENDPOINT_MODULE
 * @brief Gets type of a transfer from a given transfer handle.
 * @details Gets the type of the transfer provided by its handle.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle
 * @param[out] transfer_type Transfer type (a value from enum #usb_host_transfer_type_e)
 * @return 0 on success, otherwise a negative error value
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>
#include <stdio.h>

void print_transfer_type(usb_host_transfer_h transfer)
{
	int ret;
	usb_host_transfer_type_e transport_type;

	ret = usb_host_transfer_get_type(transfer, &transport_type);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);

	switch (transfer_type) {
	case USB_HOST_TRANSFER_TYPE_CONTROL:
		printf("Control transfer\n");
		break;
	case USB_HOST_TRANSFER_TYPE_ISOCHRONOUS:
		printf("Isochronous transfer\n");
		break;
	case USB_HOST_TRANSFER_TYPE_BULK:
		printf("Bulk transfer\n");
		break;
	case USB_HOST_TRANSFER_TYPE_INTERRUPT:
		printf("Interrupt transfer\n");
		break;
	}
}
 * @endcode
 */
int usb_host_transfer_get_type(usb_host_transfer_h transfer, usb_host_transfer_type_e *transfer_type);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Destroys a finished or cancelled asynchronous transfer structure.
 * @details When no longer needed, transfer should be destroyed by this function.
 * It frees memory allocated for the transfer. You cannot destroy unfinished transfer,
 * wait for its completion or cancel it.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle to be destroyed
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

void transfer_completion_cb(usb_host_transfer_h transfer, void *user_data);
{
	// The transfer completion callback is the earliest possible place to call this.
	// Make sure all other copies are not used anymore!
	usb_host_transfer_destroy(transfer);
}
 * @endcode
 */
int usb_host_transfer_destroy(usb_host_transfer_h transfer);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Submits an initialized, asynchronous USB transfer.
 * @details Calling this will start the actual process of the transfer.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle to be submitted
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 * @pre @a transfer should be initialized by one of initialization functions:
 * usb_host_create_transfer(), usb_host_create_isochronous_transfer() or usb_host_create_control_transfer().
 *
 * @code
#include <usb_host.h>

void submit(usb_host_transfer_h transfer)
{
	int ret = usb_host_transfer_submit(transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer_submit(usb_host_transfer_h transfer);

/**
 * @ingroup CAPI_USB_HOST_ASYNC_MODULE
 * @brief Attempts to cancel an ongoing asynchronous USB transfer.
 * @details After calling this function the transfer will be cancelled, if only
 * it was not finished already. The transfer callback will be called with #USB_HOST_TRANSFER_CANCELLED status.
 * @since_tizen 5.0
 * @param[in] transfer Transfer handle to be cancelled
 * @return 0 on success, negative error code on error
 * @retval #USB_HOST_ERROR_NONE Successful
 * @retval #USB_HOST_ERROR_NOT_SUPPORTED Not supported
 * @retval #USB_HOST_ERROR_INVALID_PARAMETER Invalid parameter was passed
 *
 * @code
#include <usb_host.h>

void cancel_transfer(usb_host_transfer_h transfer)
{
	int ret = usb_host_transfer_cancel(transfer);
	if (ret != USB_HOST_ERROR_NONE)
		handle_error(ret);
}
 * @endcode
 */
int usb_host_transfer_cancel(usb_host_transfer_h transfer);

#ifdef __cplusplus
}
#endif

#endif /* __TIZEN_USB_HOST_H__ */
