/*
 * Copyright (c) 2018 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#ifndef __TIZEN_CAPI_NETWORK_USER_AWARENESS_H__
#define __TIZEN_CAPI_NETWORK_USER_AWARENESS_H__

#include <stdbool.h>
#include <glib.h>
#include <tizen.h>

#include <user-awareness-type.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @file user-awareness.h
 */

/**
 * @ingroup CAPI_NETWORK_FRAMEWORK
 * @addtogroup CAPI_NETWORK_UA_MODULE
 * @{
 */

/**
 * @brief Initializes the user awareness framework.
 * @since_tizen 6.5
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_ALREADY_DONE Already initialized
 *
 * @see ua_deinitialize()
 */
int ua_initialize(void);

/**
 * @brief De-initializes the user awareness framework.
 * @since_tizen 6.5
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 * @retval #UA_ERROR_ALREADY_DONE Already initialized
 *
 * @see ua_initialize()
 */
int ua_deinitialize(void);

/**
 * @brief Retrieves handles of all added services.
 * @since_tizen 6.5
 *
 * @param[in] foreach_cb Callback function to be invoked with each service handle.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_cb()
 */
int ua_foreach_service(
		ua_service_cb foreach_cb,
		void *user_data);

/**
 * @brief Gets default service handle.
 * @since_tizen 6.5
 *
 * @remarks The @a service handle is managed by the platform and will be released
 *             when ua_deinitialize() is called.
 * @param[out] service The service handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_get_name()
 * @see ua_service_create()
 */
int ua_get_default_service(
		ua_service_h *service);

/**
 * @brief Gets service handle by service name for added service.
 * @since_tizen 6.5
 *
 * @remarks The @a service handle should be destroyed by using ua_service_destroy().
 *
 * @param[in] name The service name information
 * @param[out] service The service handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter or if service with such name does not exist.
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_create()
 */
int ua_get_service_by_name(
			const char *name,
			ua_service_h *service);

/**
 * @brief Creates the ua monitor's handle.
 * @since_tizen 6.5
 *
 * @remarks The @a monitor handle should be destroyed by using ua_monitor_destroy().
 *
 * @param[out] monitor The monitor handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_destroy()
 */
int ua_monitor_create(
		ua_monitor_h *monitor);

/**
 * @brief Destroys the ua monitor's handle.
 * @since_tizen 6.5
 *
 * @param[in] monitor The monitor handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_create()
 */
int ua_monitor_destroy(
		ua_monitor_h monitor);

/**
 * @brief Adds sensors to UA monitor for which monitoring is to be done.
 * Availability of sensors to be added can be checked using ua_monitor_is_sensor_available().
 * @since_tizen 6.5
 * @remarks If you want to add the BLE sensor,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to add the Wi-Fi sensor,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] monitor The monitor handle
 * @param[in] sensor The sensor to be added to monitor
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter or sensor not available to add
 * @retval #UA_ERROR_NOT_PERMITTED Operation not permitted
 * @retval #UA_ERROR_ALREADY_DONE Already done
 * @retval #UA_ERROR_NOT_READY Resource not ready
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_remove_sensor()
 * @see ua_monitor_foreach_sensor()
 */
int ua_monitor_add_sensor(
		ua_monitor_h monitor,
		ua_sensor_type_e sensor);

/**
 * @brief Removes sensors from UA monitor.
 * @since_tizen 6.5
 * @remarks If you want to remove the BLE sensor,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to remove the Wi-Fi sensor,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] monitor The monitor handle
 * @param[in] sensor Sensor to be removed from monitor
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter or sensor not available to remove
 * @retval #UA_ERROR_NOT_PERMITTED Operation not permitted
 * @retval #UA_ERROR_ALREADY_DONE Already done
 * @retval #UA_ERROR_NOT_READY Resource not ready
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_add_sensor()
 * @see ua_monitor_foreach_sensor()
 */
int ua_monitor_remove_sensor(
		ua_monitor_h monitor,
		ua_sensor_type_e sensor);

/**
 * @brief Retrieves all sensors present in UA monitor.
 * @since_tizen 6.5
 *
 * @param[in] monitor The monitor handle
 * @param[in] foreach_cb Callback function to be invoked foreach sensor present in monitor handle.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_add_sensor()
 * @see ua_monitor_remove_sensor()
 */
int ua_monitor_foreach_sensor(
		ua_monitor_h monitor,
		ua_monitor_sensor_cb foreach_cb,
		void *user_data);

/**
 * @brief Checks if sensor is available or not.
 * @since_tizen 6.5
 *
 * @param[in] sensor The sensor for which status to be checked.
 * @param[out] available TRUE if sensor is available FALSE otherwise.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_monitor_is_sensor_available(
		ua_sensor_type_e sensor,
		bool *available);

/**
 * @brief Sets sensor status changed callback. Callback to be invoked
 * when sensor status changes from 'PRESENCE' to 'ABSENCE' or vice-versa.
 * @since_tizen 6.5
 *
 * @remarks Sensor status change will be reported only for environmental
 * sensors (ie. the sensors for which user/device information is not available,
 * For example. Light and Motion Sensors)
 *
 * @param[in] monitor The monitor handle
 * @param[in] callback Sensor status changed callback
 * @param[in] user_data The user data to be passed in sensor state changed callback.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_monitor_set_sensor_status_changed_cb(
		ua_monitor_h monitor,
		ua_sensor_status_changed_cb callback,
		void *user_data);

/**
 * @brief Unsets sensor status changed callback.
 * @since_tizen 6.5
 *
 * @param[in] monitor The monitor handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_set_sensor_status_changed_cb()
 */
int ua_monitor_unset_sensor_status_changed_cb(
		ua_monitor_h monitor);

/**
 * @brief Starts scan of registered devices.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to start BLE scan,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to start Wi-Fi scan,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @remarks Only one scan can run at one time.
 *
 * @param[in] monitor The monitor handle
 * @param[in] scan_time_multiplier A value which we will multiply by #UA_SCAN_TIME_MULTIPLIER.
 *            This is maximum time (in seconds) for which scan can run, but scan can complete earlier also.
 * @param[in] callback It will be called for each scanned device one by one, and on scan
 *            completion too.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOW_IN_PROGRESS Operation now in progress
 * @retval #UA_ERROR_NO_DATA No sensor available
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see UA_SCAN_TIME_MULTIPLIER
 * @see ua_scan_completed_cb()
 * @see ua_monitor_stop_scan()
 */
int ua_monitor_start_scan(
		ua_monitor_h monitor,
		int scan_time_multiplier,
		ua_scan_completed_cb callback,
		void *user_data);

/**
 * @brief Stops ua_monitor_start_scan().
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to stop BLE scan,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to stop Wi-Fi scan,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] monitor UA monitor's handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_IN_PROGRESS Operation not in progress
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_start_scan()
 */
int ua_monitor_stop_scan(ua_monitor_h monitor);

/**
 * @brief Starts user presence detection.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to start BLE presence detection,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to start Wi-Fi presence detection,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @remarks This function is not specific for any particular user's presence detection.
 * Therefore, the callback will be invoked when any user is detected by sensors.
 *
 * @param[in] monitor The monitor handle
 * @param[in] service The service handle. If service is NULL then monitor will start for default service.
 * @param[in] mode  User detection mode. \n
 *  #UA_DETECT_MODE_ALL_SENSORS : Detection callback will be invoked only after a user is
 *  detected by all sensors added to monitor. \n
 *  #UA_DETECT_MODE_ANY_SENSOR : Detection callback will be invoked as soon as a user is
 *  detected by any one of the sensors. \n
 * @param[in] callback Callback to be invoked after device presence detection.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOW_IN_PROGRESS Operation now in progress
 * @retval #UA_ERROR_NO_DATA No sensor available
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_presence_detected_cb()
 * @see ua_monitor_stop_presence_detection()
 */
int ua_monitor_start_presence_detection(
		ua_monitor_h monitor,
		ua_service_h service,
		ua_detection_mode_e mode,
		ua_presence_detected_cb callback,
		void *user_data);

/**
 * @brief Stops user presence detection.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to stop BLE presence detection,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to stop Wi-Fi presence detection,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] monitor UA monitor's handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_IN_PROGRESS Operation not in progress
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_start_presence_detection()
 */
int ua_monitor_stop_presence_detection(
		ua_monitor_h monitor);

/**
 * @brief Starts user absence detection.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to start BLE absence detection,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to start Wi-Fi absence detection,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @remarks This function is not specific for any particular user's absence detection.
 * Therefore, the callback will be invoked only when no user is detected by sensors.
 *
 * @param[in] monitor The monitor handle
 * @param[in] service The service handle. If service is NULL then monitor will start for default service.
 * @param[in] mode User detection mode. \n
 *  #UA_DETECT_MODE_ALL_SENSORS : Detection callback will be invoked only after no user is
 *  detected by all sensors added to monitor. \n
 *  #UA_DETECT_MODE_ANY_SENSOR : Detection callback will be invoked as soon as no user is
 *  detected by any one of the sensors. \n
 * @param[in] callback  Callback to be invoked after absence detection.
 * @param[in] user_data  The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOW_IN_PROGRESS Operation now in progress
 * @retval #UA_ERROR_NO_DATA No sensor available
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_absence_detected_cb()
 * @see ua_monitor_stop_absence_detection()
 */
int ua_monitor_start_absence_detection(
		ua_monitor_h monitor,
		ua_service_h service,
		ua_detection_mode_e mode,
		ua_absence_detected_cb callback,
		void *user_data);

/**
 * @brief Stops user absence detection.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to stop BLE absence detection,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to stop Wi-Fi absence detection,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] monitor The monitor handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_IN_PROGRESS Operation not in progress
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_start_absence_detection()
 */
int ua_monitor_stop_absence_detection(
		ua_monitor_h monitor);

/**
 * @brief Sets the location callback period in milli-seconds.
 * If location period is not set, it will be automatically
 * set to #UA_LOCATION_PERIOD_DEFAULT.
 * @since_tizen 6.5
 *
 * @param[in] monitor The monitor handle
 * @param[in] location_period location period (0 to UINT_MAX)
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 */
int ua_monitor_set_location_period(
	ua_monitor_h monitor, unsigned int location_period);

/**
 * @brief Starts user location detection.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to start BLE location detection,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * If you want to start Wi-Fi location detection,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] monitor The monitor handle
 * @param[in] service The service handle. If service is NULL then monitor will start for default service.
 * @param[in] callback Callback to be invoked after device location detection.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOW_IN_PROGRESS Operation now in progress
 * @retval #UA_ERROR_NO_DATA No sensor available
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_location_detected_cb()
 * @see ua_monitor_set_location_period()
 * @see ua_monitor_stop_location_detection()
 */
int ua_monitor_start_location_detection(
		ua_monitor_h monitor,
		ua_service_h service,
		ua_location_detected_cb callback,
		void *user_data);

/**
 * @brief Stops user location detection.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 * @remarks If you want to stop BLE location detection,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * If you want to stop Wi-Fi location detection,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] monitor UA monitor's handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_IN_PROGRESS Operation not in progress
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_monitor_start_location_detection()
 */
int ua_monitor_stop_location_detection(
			ua_monitor_h monitor);

/**
 * @brief Creates the user handle.
 * @since_tizen 6.5
 *
 * @remarks The @a user handle should be destroyed by using ua_user_destroy().
 *
 * @param[in] account Account information. Maximum Account string
 * length allowed is 254.
 * @param[out] user The user handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_user_destroy()
 */
int ua_user_create(const char *account,
		ua_user_h *user);

/**
 * @brief Destroys the user handle.
 * @since_tizen 6.5
 *
 * @remarks User context will removed and all allocated memory (if any) will be released.
 *
 * @param[in] user The user handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_user_create()
 */
int ua_user_destroy(
		ua_user_h user);

/**
 * @brief Adds the user. Added user is stored in the database
 * and is also available after the application restarts.
 * @since_tizen 6.5
 *
 * @param[in] user The user handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 * @retval #UA_ERROR_ALREADY_REGISTERED Already Registered
 *
 * @see ua_user_remove()
 */
int ua_user_add(ua_user_h user);

/**
 * @brief Removes the user.
 * @since_tizen 6.5
 *
 * @param[in] user The user handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_user_add()
 */
int ua_user_remove(ua_user_h user);

/**
 * @brief Gets account info for user handle.
 * @since_tizen 6.5
 *
 * @remarks You must release @a account using g_free().
 * @param[in] user The user handle
 * @param[out] account  Account information
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_user_create()
 */
int ua_user_get_account(ua_user_h user, char **account);

/**
 * @brief Adds device for a user. This information is stored in database
 * and is also available after the application restarts.
 * @since_tizen 6.5
 * @remarks If you want to add the BLE device,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to add the Wi-Fi device,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] user The user handle.
 * @param[in] device The device handle.
 * @param[in] callback Callback to be invoked after adding the user done.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_user_remove_device()
 * @see ua_user_device_added_cb()
 */
int ua_user_add_device(
		ua_user_h user,
		ua_device_h device,
		ua_user_device_added_cb callback,
		void *user_data);

/**
 * @brief Removes the device handle for a user handle.
 * @since_tizen 6.5
 * @remarks If you want to remove the BLE device,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to remove the Wi-Fi device,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] user The user handle.
 * @param[in] device The device handle.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_RESOURCE_BUSY Remove done. but another service uses this device.
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_user_add_device()
 */
int ua_user_remove_device(
		ua_user_h user,
		ua_device_h device);

/**
 * @brief Retrieves the device handle of all the registered devices for a specific user.
 * @since_tizen 6.5
 *
 * @param[in] user The user handle.
 * @param[in] foreach_cb Callback function to be invoked with user device handle.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_registered_device_cb()
 */
int ua_user_foreach_devices(
		ua_user_h user,
		ua_registered_device_cb foreach_cb,
		void *user_data);

/**
 * @brief Creates the service handle.
 * @since_tizen 6.5
 *
 * @remarks The @a service handle should be destroyed by using ua_service_destroy().
 *
 * @param[in] name The service name information
 * @param[out] service The service handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_destroy()
 */
int ua_service_create(
	const char *name,
	ua_service_h *service);

/**
 * @brief Destroys the service handle.
 * @since_tizen 6.5
 *
 * @remarks Service context will be removed and all allocated memory (if any) will be released.
 *
 * @param[in] service The service handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_create()
 */
int ua_service_destroy(
		ua_service_h service);

/**
 * @brief Adds the service. Added service is stored in database
 * and is also available after the application restarts.
 * @since_tizen 6.5
 *
 * @param[in] service The service handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_remove()
 */
int ua_service_add(
		ua_service_h service);

/**
 * @brief Updates the properties for the added service.
 * To update a property of a service its respective setter functions
 * should be called and then followed by the call of ua_service_update().
 * @since_tizen 6.5
 *
 * @param[in] service The service handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_create()
 * @see ua_service_remove()
 */
int ua_service_update(
		ua_service_h service);

/**
 * @brief Removes the service.
 * @since_tizen 6.5
 *
 * @param[in] service The service handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_add()
 */
int ua_service_remove(
		ua_service_h service);

/**
 * @brief Gets name info for service handle.
 * @since_tizen 6.5
 *
 * @remarks You must release @a name using g_free().
 *
 * @param[in] service The service handle
 * @param[out] name The service name information
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_create()
 */
int ua_service_get_name(
		ua_service_h service,
		char **name);

/**
 * @brief Sets detection threshold for service handle.
 * If the service has already been added, this call should be followed by
 * the call of ua_service_update() to reflect the modifications.
 * @since_tizen 6.5
 *
 * @param[in] service The service handle
 * @param[in] presence_threshold The service presence threshold information (in lux)
 * @param[in] absence_threshold The service absence threshold information (in lux)
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_service_set_detection_threshold(
		ua_service_h service,
		unsigned int presence_threshold,
		unsigned int absence_threshold);

/**
 * @brief Gets detection threshold for service handle.
 * @since_tizen 6.5
 *
 * @param[in] service The service handle
 * @param[out] presence_threshold The service presence threshold information (in lux)
 * @param[out] absence_threshold The service absence threshold information (in lux)
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_set_detection_threshold()
 */
int ua_service_get_detection_threshold(
		ua_service_h service,
		unsigned int *presence_threshold,
		unsigned int *absence_threshold);

/**
 * @brief Sets name info for service handle.
 * @since_tizen 6.5
 *
 * @remarks The @a user should not be released.
 * @remarks The @a user is available until user is
 * removed and released by application using ua_user_remove()
 * and ua_user_destroy()
 *
 * @param[in] service The service handle
 * @param[in] account The user account information
 * @param[out] user The user handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_service_get_user_by_account(ua_service_h service,
		const char* account, ua_user_h *user);

/**
 * @brief Adds a user for a specific service.
 * @since_tizen 6.5
 *
 * @param[in] service The service handle.
 * @param[in] user The user handle.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_remove_user()
 */
int ua_service_add_user(
		ua_service_h service,
		ua_user_h user);

/**
 * @brief Removes the user for the specific service handle.
 * @since_tizen 6.5
 *
 * @param[in] service The service handle.
 * @param[in] user The user handle.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_add_user()
 */
int ua_service_remove_user(
		ua_service_h service,
		ua_user_h user);

/**
 * @brief Triggers callback for every user of a specific service.
 * @since_tizen 6.5
 *
 * @param[in] service The service handle.
 * @param[in] foreach_cb Callback function to be invoked with user handle.
 * @param[in] user_data The user data to be passed when callback is called.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_service_user_cb()
 */
int ua_service_foreach_users(
		ua_service_h service,
		ua_service_user_cb foreach_cb,
		void *user_data);

/**
 * @brief Creates the device handle.
 * @since_tizen 6.5
 *
 * @remarks The @a device handle should be destroyed using ua_user_destroy().
 *
 * @param[in] mac_type The MAC address type of the device
 * @param[in] mac_address The device's MAC address.
 * @param[in] device_id The device ID.
 *
 * @param[out] device The device handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_device_destroy()
 */
int ua_device_create(ua_mac_type_e mac_type,
			const char* mac_address,
			const char *device_id,
			ua_device_h *device);

/**
 * @brief Destroys the device handle.
 * @since_tizen 6.5
 *
 * @remarks Device context will be removed and all allocated memory (if any) will be released.
 *
 * @param[in] device The device handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_device_create()
 */
int ua_device_destroy(
		ua_device_h device);

/**
 * @brief Gets device type info from the device.
 * @since_tizen 6.5
 *
 * @param[in] device The device handle
 * @param[out] mac_type The MAC address type of the device
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int ua_device_get_mac_type(
		ua_device_h device,
		ua_mac_type_e *mac_type);

/**
 * @brief Sets device's operating system info.
 * If the device has already been added, this call should be followed by
 * the call of ua_device_update() to reflect the modifications.
 * @since_tizen 6.5
 *
 * @param[in] device The device handle
 * @param[in] os The device's operating system.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_device_get_os_info()
 */
int ua_device_set_os_info(
		ua_device_h device,
		ua_os_type_e os);

/**
 * @brief Gets device's operating system info.
 * @since_tizen 6.5
 *
 * @param[in] device The device handle
 * @param[out] os_info Device's operating system info.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see ua_device_set_os_info()
 */
int ua_device_get_os_info(
		ua_device_h device,
		ua_os_type_e *os_info);

/**
 * @brief Gets device's MAC address.
 * @since_tizen 6.5
 *
 * @remarks You must release @a mac_address using g_free().
 *
 * @param[in] device The device handle
 * @param[out] mac_address The device's MAC address.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int ua_device_get_mac_address(
		ua_device_h device,
		char **mac_address);

/**
 * @brief Gets device ID of device, which was set while creating the device handle.
 * @since_tizen 6.5
 *
 * @remarks You must release @a device_id using g_free().
 * @param[in] device The device handle
 * @param[out] device_id The device ID.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int ua_device_get_device_id(
		ua_device_h device,
		char **device_id);

/**
 * @brief Sets device's Wi-Fi BSSID.
 * If the device has already been added, this call should be followed by
 * the call of ua_device_update() to reflect the modifications.
 * @since_tizen 6.5
 *
 * @param[in] device The device handle
 * @param[in] bssid  The device's Wi-Fi BSSID.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_device_get_wifi_bssid()
 */
int ua_device_set_wifi_bssid(
		ua_device_h device,
		const char *bssid);

/**
 * @brief Gets device's Wi-Fi BSSID.
 * @since_tizen 6.5
 *
 * @remarks You must release @a bssid using g_free().
 * @param[in] device The device handle
 * @param[out] bssid The device's Wi-Fi BSSID.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see ua_device_set_wifi_bssid()
 */
int ua_device_get_wifi_bssid(
		ua_device_h device,
		char **bssid);
/**
 * @brief Sets device's Wi-Fi IPv4 address.
 * If the device has already been added, this call should be followed by
 * the call of ua_device_update() to reflect the modifications.
 * @since_tizen 6.5
 *
 * @param[in] device The device handle
 * @param[in] ipv4_address The device's wifi IPv4 address.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_device_get_wifi_ipv4_address()
 */
int ua_device_set_wifi_ipv4_address(
		ua_device_h device,
		const char *ipv4_address);

/**
 * @brief Gets device's Wi-Fi IPv4 address.
 * @since_tizen 6.5
 *
 * @remarks You must release @a ipv4_address using g_free().
 *
 * @param[in] device The device handle
 * @param[out] ipv4_address The device's Wi-Fi IPv4 address.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see ua_device_set_wifi_ipv4_address()
 */
int ua_device_get_wifi_ipv4_address(
		ua_device_h device,
		char **ipv4_address);

/**
 * @brief Gets last presence time for device handle.
 * @since_tizen 6.5
 *
 * @param[in] device The device handle
 * @param[out] last_seen The last presence monotonic timestamp for UA device (in msec)
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int ua_device_get_last_presence(
		ua_device_h device,
		unsigned long long *last_seen);

/**
 * @brief Gets whether pairing is required for the user device.
 * @since_tizen 6.5
 *
 * @param[in] device The device handle
 * @param[out] pairing_required TRUE if pairing is required, FALSE otherwise.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_device_create()
 */
int ua_device_get_pairing_required(
		ua_device_h device,
		bool *pairing_required);

/**
 * @brief Updates the properties for the added device.
 * To update a property of a device its respective setter functions
 * should be called and then followed by the call of ua_device_update().
 * @since_tizen 6.5
 * @remarks If you want to update BLE device,
 * you should add privilege %http://tizen.org/privilege/bluetooth.
 * if you want to update Wi-Fi device,
 * you should add privilege %http://tizen.org/privilege/network.get.
 *
 * @param[in] device The device handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OPERATION_FAILED Operation failed
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 * @see ua_device_set_os_info()
 * @see ua_device_set_wifi_ipv4_address()
 */
int ua_device_update(
		ua_device_h device);

/**
 * @brief Gets device handle by MAC address.
 * @since_tizen 6.5
 *
 * @remarks The @a device handle should not be released.
 * @remarks The @a device handle is available until device is
 * removed and released by application using ua_user_remove_device()
 * and ua_device_destroy()
 *
 * @param[in] mac The device's MAC address
 * @param[out] device The device handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_device_get_by_mac_address(
		const char *mac,
		ua_device_h *device);

/**
 * @brief Gets device handle by device ID.
 * @since_tizen 6.5
 *
 * @remarks The @a device handle should not be released.
 * @remarks The @a device handle is available until device is
 * removed and released by application using ua_user_remove_device()
 * and ua_device_destroy()
 *
 * @param[in] device_id The device ID
 * @param[in] mac_type The device's MAC type
 * @param[out] device The device handle.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_device_get_by_device_id(
		const char *device_id,
		ua_mac_type_e mac_type,
		ua_device_h *device);

/**
 * @brief Gets sensor's status report.
 * @since_tizen 6.5
 *
 * @param[in] sensor The sensor handle
 * @param[out] report The sensor's status report.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_sensor_get_status(
		ua_sensor_h sensor,
		ua_sensor_report_e *report);

/**
 * @brief Gets sensor's timestamp.
 * @since_tizen 6.5
 *
 * @param[in] sensor The sensor handle
 * @param[out] timestamp The sensor's timestamp.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_sensor_get_timestamp(
		ua_sensor_h sensor,
		unsigned long long *timestamp);

/**
 * @brief Gets sensor's type.
 * @since_tizen 6.5
 *
 * @param[in] sensor The sensor handle
 * @param[out] type The sensor's type.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_NOT_SUPPORTED Not Supported
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 *
 */
int ua_sensor_get_type(
		ua_sensor_h sensor,
		ua_sensor_type_e *type);

/**
 * @brief Gets the distance for the location handle.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 *
 * @param[in] location The location handle for location information.
 * @param[out] distance distance of the detected device and user.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 */
int ua_location_get_distance(ua_location_h location, int *distance);

/**
 * @brief Gets the position for the location handle.
 * @since_tizen 6.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/location
 *
 * @param[in] location The location handle for location information.
 * @param[out] x x-coordinate of the location.
 * @param[out] y y-coordinate of the location.
 * @param[out] z z-coordinate of the location.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #UA_ERROR_NONE Successful
 * @retval #UA_ERROR_PERMISSION_DENIED Permission Denied
 * @retval #UA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UA_ERROR_NOT_INITIALIZED Not initialized yet
 */
int ua_location_get_position(ua_location_h location,
					int *x, int *y, int *z);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif
#endif /* __TIZEN_CAPI_NETWORK_USER_AWARENESS_H__ */
