/*
 * Copyright (c) 2012-2013 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __TIZEN_WIFI_MANAGER_H__
#define __TIZEN_WIFI_MANAGER_H__

#include <tizen.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @file wifi-manager.h
 */

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_MODULE
* @{
*/

/**
 * @brief The Wi-Fi manager handle.
 * @since_tizen 3.0
*/
typedef void *wifi_manager_h;

/**
 * @brief Enumeration for the Wi-Fi error type.
 * @since_tizen 3.0
 */
typedef enum {
	/**
	 * Successful
	 */
	WIFI_MANAGER_ERROR_NONE = TIZEN_ERROR_NONE,

	/**
	 * Invalid parameter
	 */
	WIFI_MANAGER_ERROR_INVALID_PARAMETER = TIZEN_ERROR_INVALID_PARAMETER,

	/**
	 * Out of memory error
	 */
	WIFI_MANAGER_ERROR_OUT_OF_MEMORY = TIZEN_ERROR_OUT_OF_MEMORY,

	/**
	 * Invalid operation
	 */
	WIFI_MANAGER_ERROR_INVALID_OPERATION = TIZEN_ERROR_INVALID_OPERATION,

	/**
	 * Address family not supported
	 */
	WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED = TIZEN_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED,

	/**
	 * Operation failed
	 */
	WIFI_MANAGER_ERROR_OPERATION_FAILED = TIZEN_ERROR_WIFI_MANAGER|0x01,

	/**
	 * There is no connected AP
	 */
	WIFI_MANAGER_ERROR_NO_CONNECTION = TIZEN_ERROR_WIFI_MANAGER|0x02,

	/**
	 * Now in progress
	 */
	WIFI_MANAGER_ERROR_NOW_IN_PROGRESS = TIZEN_ERROR_NOW_IN_PROGRESS,

	/**
	 * Already exists
	 */
	WIFI_MANAGER_ERROR_ALREADY_EXISTS = TIZEN_ERROR_WIFI_MANAGER|0x03,

	/**
	 * Operation is aborted
	 */
	WIFI_MANAGER_ERROR_OPERATION_ABORTED = TIZEN_ERROR_WIFI_MANAGER|0x04,

	/**
	 * DHCP failed
	 */
	WIFI_MANAGER_ERROR_DHCP_FAILED = TIZEN_ERROR_WIFI_MANAGER|0x05,

	/**
	 * Invalid key
	 */
	WIFI_MANAGER_ERROR_INVALID_KEY = TIZEN_ERROR_WIFI_MANAGER|0x06,

	/**
	 * No reply
	 */
	WIFI_MANAGER_ERROR_NO_REPLY = TIZEN_ERROR_WIFI_MANAGER|0x07,

	/**
	 * Restricted by security system policy
	 */
	WIFI_MANAGER_ERROR_SECURITY_RESTRICTED = TIZEN_ERROR_WIFI_MANAGER|0x08,

	/**
	 * Already initialized
	 */
	WIFI_MANAGER_ERROR_ALREADY_INITIALIZED = TIZEN_ERROR_WIFI_MANAGER|0x09,

	/**
	 * Out of range
	 */
	WIFI_MANAGER_ERROR_OUT_OF_RANGE = TIZEN_ERROR_WIFI_MANAGER|0x0A,

	/**
	 * Connect failed
	 */
	WIFI_MANAGER_ERROR_CONNECT_FAILED = TIZEN_ERROR_WIFI_MANAGER|0x0B,

	/**
	 * Login failed
	 */
	WIFI_MANAGER_ERROR_LOGIN_FAILED = TIZEN_ERROR_WIFI_MANAGER|0x0C,

	/**
	 * Authentication failed
	 */
	WIFI_MANAGER_ERROR_AUTHENTICATION_FAILED = TIZEN_ERROR_WIFI_MANAGER|0x0D,

	/**
	 * PIN missing
	 */
	WIFI_MANAGER_ERROR_PIN_MISSING = TIZEN_ERROR_WIFI_MANAGER|0x0E,

	/**
	 * WPS Overlap (Since 4.0)
	 */
	WIFI_MANAGER_ERROR_WPS_OVERLAP = TIZEN_ERROR_WIFI_MANAGER|0x0F,

	/**
	 * WPS Timeout (Since 4.0)
	 */
	WIFI_MANAGER_ERROR_WPS_TIMEOUT = TIZEN_ERROR_WIFI_MANAGER|0x10,

	/**
	 * WPS WEP Prohibited (Since 4.0)
	 */
	WIFI_MANAGER_ERROR_WPS_WEP_PROHIBITED = TIZEN_ERROR_WIFI_MANAGER|0x11,

	/**
	 * Permission Denied
	 */
	WIFI_MANAGER_ERROR_PERMISSION_DENIED = TIZEN_ERROR_PERMISSION_DENIED,

	/**
	 * Connection Offline (Since 5.0)
	 */
	WIFI_MANAGER_ERROR_OFFLINE = TIZEN_ERROR_WIFI_MANAGER|0x12,

	/**
	 * Invalid Gateway (Since 5.0)
	 */
	WIFI_MANAGER_ERROR_INVALID_GATEWAY = TIZEN_ERROR_WIFI_MANAGER|0x13,

	/**
	 * Not Supported
	 */
	WIFI_MANAGER_ERROR_NOT_SUPPORTED = TIZEN_ERROR_NOT_SUPPORTED,

	/**
	 * Not initialized (Since 5.0)
	 */
	WIFI_MANAGER_ERROR_NOT_INITIALIZED = TIZEN_ERROR_WIFI_MANAGER|0x14,

	/**
	 * Association failed (Since 7.0)
	 */
	WIFI_MANAGER_ERROR_ASSOCIATION_FAILED = TIZEN_ERROR_WIFI_MANAGER|0x15,

	/**
	 * The wireless device is unavailable (Since 8.0)
	 */
	WIFI_MANAGER_ERROR_NO_CARRIER = TIZEN_ERROR_WIFI_MANAGER|0x16,
} wifi_manager_error_e;


/**
* @}
*/

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_MONITOR_MODULE
* @{
*/

/**
 * @brief Enumeration for the state of the Wi-Fi device.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_DEVICE_STATE_DEACTIVATED = 0,    /**< Wi-Fi is Deactivated */
	WIFI_MANAGER_DEVICE_STATE_ACTIVATED = 1,      /**< Wi-Fi is activated */
} wifi_manager_device_state_e;

/**
 * @brief Enumeration for the state of the Wi-Fi connection.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_CONNECTION_STATE_FAILURE = -1,       /**< Connection failed state */
	WIFI_MANAGER_CONNECTION_STATE_DISCONNECTED = 0,   /**< Disconnected state */
	WIFI_MANAGER_CONNECTION_STATE_ASSOCIATION = 1,    /**< Association state */
	WIFI_MANAGER_CONNECTION_STATE_CONFIGURATION = 2,  /**< Configuration state */
	WIFI_MANAGER_CONNECTION_STATE_CONNECTED = 3,      /**< Connected state */
} wifi_manager_connection_state_e;

/**
 * @brief Enumeration for the state of the static IP conflict.
 * @since_tizen 5.0
 */
typedef enum {
	WIFI_MANAGER_IP_CONFLICT_STATE_UNKNOWN = 0,            /**< Unknown state */
	WIFI_MANAGER_IP_CONFLICT_STATE_CONFLICT_NOT_DETECTED = 1,  /**< Resolved state */
	WIFI_MANAGER_IP_CONFLICT_STATE_CONFLICT_DETECTED = 2,  /**< Conflict state */
} wifi_manager_ip_conflict_state_e;

/**
 * @brief Enumeration for the RSSI level.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_RSSI_LEVEL_0 = 0,      /**< No signal */
	WIFI_MANAGER_RSSI_LEVEL_1 = 1,      /**< Very weak signal ~ -83dBm */
	WIFI_MANAGER_RSSI_LEVEL_2 = 2,      /**< Weak signal -82 ~ -75dBm */
	WIFI_MANAGER_RSSI_LEVEL_3 = 3,      /**< Strong signal -74 ~ -64dBm */
	WIFI_MANAGER_RSSI_LEVEL_4 = 4,      /**< Very strong signal -63dBm ~ */
} wifi_manager_rssi_level_e;

/**
 * @brief Enumeration for the wifi scanning state.
 * @since_tizen 4.0
 */
typedef enum {
	WIFI_MANAGER_SCAN_STATE_NOT_SCANNING = 0,  /**< Scan is not running */
	WIFI_MANAGER_SCAN_STATE_SCANNING = 1,      /**< Scan is in progress */
} wifi_manager_scan_state_e;

/**
 * @brief Enumeration for the TDLS Discovery State.
 * @since_tizen 4.0
 */
typedef enum {
	WIFI_MANAGER_TDLS_DISCOVERY_STATE_ONGOING = 0,    /**< Discovery is in progress */
	WIFI_MANAGER_TDLS_DISCOVERY_STATE_FINISHED = 1,   /**< Discovery is finished */
} wifi_manager_tdls_discovery_state_e;

/**
 * @brief Enumeration for the Wi-Fi Module state.
 * @since_tizen 4.0
 */
typedef enum {
	WIFI_MANAGER_MODULE_STATE_DETACHED = 0,  /**< Wi-Fi Module is detached */
	WIFI_MANAGER_MODULE_STATE_ATTACHED = 1,  /**< Wi-Fi Module is attached */
} wifi_manager_module_state_e;

/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_NETWORK_MODULE
* @{
*/

/**
 * @brief Enumeration for the Net IP configuration type.
 * @since_tizen 3.0
 */
typedef enum {
	/**
	 * Not defined
	 */
	WIFI_MANAGER_IP_CONFIG_TYPE_NONE = 0,

	/**
	 * Manual IP configuration
	 */
	WIFI_MANAGER_IP_CONFIG_TYPE_STATIC  = 1,

	/**
	 * Config IP using DHCP client (IPv4 Only)
	 */
	WIFI_MANAGER_IP_CONFIG_TYPE_DYNAMIC = 2,

	/**
	 * Config IP from Auto IP pool (169.254/16). Later with DHCP client, if available
	 */
	WIFI_MANAGER_IP_CONFIG_TYPE_AUTO = 3,

	/**
	 * Indicates an IP address that can not be modified (IPv4 Only)
	 */
	WIFI_MANAGER_IP_CONFIG_TYPE_FIXED = 4,

} wifi_manager_ip_config_type_e;

/**
 * @brief Enumeration for DNS Configuration type.
 * @since_tizen 4.0
 */
typedef enum {
	WIFI_MANAGER_DNS_CONFIG_TYPE_NONE = 0,     /**< Not defined */
	WIFI_MANAGER_DNS_CONFIG_TYPE_STATIC = 1,   /**< Manual DNS configuration */
	WIFI_MANAGER_DNS_CONFIG_TYPE_DYNAMIC = 2,  /**< Config DNS using DHCP client*/
} wifi_manager_dns_config_type_e;

/**
 * @brief Enumeration for the address type.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_ADDRESS_FAMILY_IPV4 = 0,       /**< IPV4 Address family */
	WIFI_MANAGER_ADDRESS_FAMILY_IPV6 = 1,       /**< IPV6 Address family */
} wifi_manager_address_family_e;

/**
 * @brief Enumeration for the proxy method type.
 * @since_tizen 3.0
 */
typedef enum {
	/**
	 * Direct connection
	 */
	WIFI_MANAGER_PROXY_TYPE_DIRECT = 0,

	/**
	 * Auto configuration(Use PAC file). If URL property is not set,
	 * DHCP/WPAD auto-discover will be tried
	 */
	WIFI_MANAGER_PROXY_TYPE_AUTO = 1,

	/**
	 * Manual configuration
	 */
	WIFI_MANAGER_PROXY_TYPE_MANUAL = 2

} wifi_manager_proxy_type_e;

/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_SECURITY_MODULE
* @{
*/

/**
 * @brief Enumeration for Wi-Fi security type.
 * @details The following security modes are used in infrastructure and ad-hoc mode.
 * For now all EAP security mechanisms are provided only in infrastructure mode.
 *
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_SECURITY_TYPE_NONE = 0,        /**< Security disabled */
	WIFI_MANAGER_SECURITY_TYPE_WEP = 1,         /**< WEP */
	WIFI_MANAGER_SECURITY_TYPE_WPA_PSK = 2,     /**< WPA-PSK */
	WIFI_MANAGER_SECURITY_TYPE_WPA2_PSK = 3,    /**< WPA2-PSK */
	WIFI_MANAGER_SECURITY_TYPE_EAP = 4,         /**< EAP */
	WIFI_MANAGER_SECURITY_TYPE_WPA_FT_PSK = 5,  /**< FT-PSK (Since 5.0) */
	WIFI_MANAGER_SECURITY_TYPE_SAE = 6,         /**< SAE (Since 5.5) */
	WIFI_MANAGER_SECURITY_TYPE_OWE = 7,         /**< OWE (Since 5.5) */
	WIFI_MANAGER_SECURITY_TYPE_DPP = 8,         /**< DPP (Since 5.5) */
	WIFI_MANAGER_SECURITY_TYPE_FT_SAE = 9,      /**< FT-SAE (Since 10.0) */
} wifi_manager_security_type_e;

/**
 * @brief Enumeration for Wi-Fi encryption type.
 * @details The following encryption modes are used in infrastructure and ad-hoc mode.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_ENCRYPTION_TYPE_NONE = 0,            /**< Encryption disabled */
	WIFI_MANAGER_ENCRYPTION_TYPE_WEP = 1,             /**< WEP */
	WIFI_MANAGER_ENCRYPTION_TYPE_TKIP = 2,            /**< TKIP */
	WIFI_MANAGER_ENCRYPTION_TYPE_AES = 3,             /**< AES */
	WIFI_MANAGER_ENCRYPTION_TYPE_TKIP_AES_MIXED = 4,  /**< TKIP and AES are both supported */
} wifi_manager_encryption_type_e;

/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_SECURITY_EAP_MODULE
* @{
*/

/**
 * @brief Enumeration for EAP type.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_EAP_TYPE_PEAP = 0,         /**< EAP PEAP type */
	WIFI_MANAGER_EAP_TYPE_TLS = 1,          /**< EAP TLS type */
	WIFI_MANAGER_EAP_TYPE_TTLS = 2,         /**< EAP TTLS type */
	WIFI_MANAGER_EAP_TYPE_SIM = 3,          /**< EAP SIM type */
	WIFI_MANAGER_EAP_TYPE_AKA = 4,          /**< EAP AKA type */
	WIFI_MANAGER_EAP_TYPE_AKA_PRIME = 5,	/**< EAP AKA Prime type */
	WIFI_MANAGER_EAP_TYPE_FAST = 6,		/**< EAP FAST type */
	WIFI_MANAGER_EAP_TYPE_PWD = 7,		/**< EAP PWD type */
} wifi_manager_eap_type_e;

/**
 * @brief Enumeration for EAP phase2 authentication type.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_EAP_AUTH_TYPE_NONE = 0,      /**< EAP phase2 authentication none */
	WIFI_MANAGER_EAP_AUTH_TYPE_PAP = 1,       /**< EAP phase2 authentication PAP */
	WIFI_MANAGER_EAP_AUTH_TYPE_MSCHAP = 2,    /**< EAP phase2 authentication MSCHAP */
	WIFI_MANAGER_EAP_AUTH_TYPE_MSCHAPV2 = 3,  /**< EAP phase2 authentication MSCHAPv2 */
	WIFI_MANAGER_EAP_AUTH_TYPE_GTC = 4,       /**< EAP phase2 authentication GTC */
	WIFI_MANAGER_EAP_AUTH_TYPE_MD5 = 5,       /**< EAP phase2 authentication MD5 */
} wifi_manager_eap_auth_type_e;

/**
 * @brief Enumeration for Wi-Fi disconnect reason, provided by the supplicant.
 * @since_tizen 4.0
 */
typedef enum {
	/**
	 * Locally Generate Disconnect from user side (Since 5.0)
	 */
	WIFI_REASON_LOCAL_GENERATE_FROM_USER = -3,

	/**
	 * Unspecified reason
	 */
	WIFI_REASON_UNSPECIFIED = 1,

	/**
	 * Previous authentication no longer valid
	 */
	WIFI_REASON_PREV_AUTH_NOT_VALID = 2,

	/**
	 * Deauthenticated because sending STA is leaving (or has left) IBSS or ESS
	 */
	WIFI_REASON_DEAUTH_LEAVING = 3,

	/**
	 * Disassociated due to inactivity
	 */
	WIFI_REASON_DISASSOC_DUE_TO_INACTIVITY = 4,

	/**
	 * Disassociated because AP is unable to handle all currently associated STAs
	 */
	WIFI_REASON_DISASSOC_AP_BUSY = 5,

	/**
	 * Class 2 frame received from nonauthenticated STA
	 */
	WIFI_REASON_CLASS2_FRAME_FROM_NONAUTH_STA = 6,

	/**
	 * Class 3 frame received from nonassociated STA
	 */
	WIFI_REASON_CLASS3_FRAME_FROM_NONASSOC_STA = 7,

	/**
	 * Disassociated because sending STA is leaving (or has left) BSS
	 */
	WIFI_REASON_DISASSOC_STA_HAS_LEFT = 8,

	/**
	 * STA requesting (re)association is not authenticated with responding STA
	 */
	WIFI_REASON_STA_REQ_ASSOC_WITHOUT_AUTH = 9,

	/**
	 * Disassociated because the information in the Power Capability
	 * element is unacceptable
	 */
	WIFI_REASON_PWR_CAPABILITY_NOT_VALID = 10,

	/**
	 * Disassociated because the information in the Supported Channels
	 * element is unacceptable
	 */
	WIFI_REASON_SUPPORTED_CHANNEL_NOT_VALID = 11,

	/**
	 * Invalid element i.e., an element defined in this standard for which the
	 * content does not meet the specifications in Clause 8
	 */
	WIFI_REASON_INVALID_IE = 13,

	/**
	 * Message Integrity Code (MIC) failure
	 */
	WIFI_REASON_MICHAEL_MIC_FAILURE = 14,

	/**
	 * 4-Way Handshake timeout
	 */
	WIFI_REASON_4WAY_HANDSHAKE_TIMEOUT = 15,

	/**
	 * Group Key Handshake timeout
	 */
	WIFI_REASON_GROUP_KEY_UPDATE_TIMEOUT = 16,

	/**
	 * Element in 4-Way Handshake different from (Re)Association Request/Probe
	 * Response/Beacon frame
	 */
	WIFI_REASON_IE_IN_4WAY_DIFFERS = 17,

	/**
	 * Invalid group cipher
	 */
	WIFI_REASON_GROUP_CIPHER_NOT_VALID = 18,

	/**
	 * Invalid pairwise cipher
	 */
	WIFI_REASON_PAIRWISE_CIPHER_NOT_VALID = 19,

	/**
	 * Invalid AKMP
	 */
	WIFI_REASON_AKMP_NOT_VALID = 20,

	/**
	 * Unsupported RSNE version
	 */
	WIFI_REASON_UNSUPPORTED_RSN_IE_VERSION = 21,

	/**
	 * Invalid RSNE capabilities
	 */
	WIFI_REASON_INVALID_RSN_IE_CAPAB = 22,

	/**
	 * IEEE 802.1X authentication failed
	 */
	WIFI_REASON_IEEE_802_1X_AUTH_FAILED = 23,

	/**
	 * Cipher suite rejected because of the security policy
	 */
	WIFI_REASON_CIPHER_SUITE_REJECTED = 24,

	/**
	 * TDLS direct-link teardown due to TDLS peer STA unreachable via the
	 * TDLS direct link
	 */
	WIFI_REASON_TDLS_TEARDOWN_UNREACHABLE = 25,

	/**
	 * TDLS direct-link teardown for unspecified reason
	 */
	WIFI_REASON_TDLS_TEARDOWN_UNSPECIFIED = 26,

	/**
	 * Disassociated because excessive number of frames need to be acknowledged,
	 * but are not acknowledged due to AP transmissions and/or poor channel conditions
	 */
	WIFI_REASON_DISASSOC_LOW_ACK = 34,

	/**
	 * SME cancels the mesh peering instance with the reason other than reaching
	 * the maximum number of peer mesh STAs
	 */
	WIFI_REASON_MESH_PEERING_CANCELLED = 52,

	/**
	 * The mesh STA has reached the supported maximum number of peer mesh STAs
	 */
	WIFI_REASON_MESH_MAX_PEERS = 53,

	/**
	 * The received information violates the Mesh Configuration policy
	 * configured in the mesh STA profile
	 */
	WIFI_REASON_MESH_CONFIG_POLICY_VIOLATION = 54,

	/**
	 * The mesh STA has received a Mesh Peering Close message requesting
	 * to close the mesh peering
	 */
	WIFI_REASON_MESH_CLOSE_RCVD = 55,

	/**
	 * The mesh STA has resent dot11MeshMaxRetries Mesh Peering Open messages,
	 * without receiving a Mesh Peering Confirm message
	 */
	WIFI_REASON_MESH_MAX_RETRIES = 56,

	/**
	 * The confirmTimer for the mesh peering instance times out
	 */
	WIFI_REASON_MESH_CONFIRM_TIMEOUT = 57,

	/**
	 * The mesh STA fails to unwrap the GTK or the values in the wrapped
	 * contents do not match
	 */
	WIFI_REASON_MESH_INVALID_GTK = 58,

	/**
	 * The mesh STA receives inconsistent information about the mesh parameters
	 * between Mesh Peering Management frames
	 */
	WIFI_REASON_MESH_INCONSISTENT_PARAMS = 59,

	/**
	 * The mesh STA does not have proxy information for this external destination
	 */
	WIFI_REASON_MESH_INVALID_SECURITY_CAP = 60,

} wifi_manager_disconnect_reason_e;

/**
 * @brief Enumeration for Wi-Fi Association Status code, provided by the supplicant.
 * @details The Wi-Fi Standard Reference : Status codes (IEEE 802.11-2007, 7.3.1.9, Table 7-23).
 * @since_tizen 5.0
 */
typedef enum {
	/**
	 * Successful
	 */
	WLAN_STATUS_SUCCESS = 0,

	/**
	 * Unspecified failure
	 */
	WLAN_STATUS_UNSPECIFIED_FAILURE = 1,

	/**
	 * Responding STA does not support the specified authentication
	 * algorithm
	 */
	WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG = 13,

	/**
	 * Association denied because AP is unable to handle additional
	 * associated STAs
	 */
	WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA = 17,

	/**
	 * Association denied because the requesting STA does not support
	 * HT feature
	 */
	WLAN_STATUS_ASSOC_DENIED_NO_HT = 27,

	/**
	 * Association request rejected temporarily
	 */
	WLAN_STATUS_ASSOC_REJECTED_TEMPORARILY = 30,

	/**
	 * Robust management frame policy violation
	 */
	WLAN_STATUS_ROBUST_MGMT_FRAME_POLICY_VIOLATION = 31,

	/**
	 * Invalid element
	 */
	WLAN_STATUS_INVALID_IE = 40,

	/**
	 * Invalid group cipher
	 */
	WLAN_STATUS_GROUP_CIPHER_NOT_VALID = 41,

	/**
	 * Invalid pairwise cipher
	 */
	WLAN_STATUS_PAIRWISE_CIPHER_NOT_VALID = 42,

	/**
	 * Invalid AKMP
	 */
	WLAN_STATUS_AKMP_NOT_VALID = 43,

	/**
	 * Cipher suite rejected because of security policy
	 */
	WLAN_STATUS_CIPHER_REJECTED_PER_POLICY = 46,

	/**
	 * Association denied because the Listen interval is too large
	 */
	WLAN_STATUS_ASSOC_DENIED_LISTEN_INT_TOO_LARGE = 51,

	/**
	 * Invalid pairwise master key identifier (PKMID)
	 */
	WLAN_STATUS_INVALID_PMKID = 53,

	/**
	 * Invalid MDE
	 */
	WLAN_STATUS_INVALID_MDIE = 54,

	/**
	 * Invalid FTE
	 */
	WLAN_STATUS_INVALID_FTIE = 55,

	/**
	 * Association denied because the requesting STA does not support
	 * VHT feature
	 */
	WLAN_STATUS_ASSOC_DENIED_NO_VHT = 104,
} wifi_manager_assoc_status_code_e;

/**
 * @}
 */

/**
 * @addtogroup CAPI_NETWORK_WIFI_MANAGER_TDLS_MODULE
 * @{
 */

/**
 * @brief Enumeration for the state of the Wi-Fi TDLS.
 * @since_tizen 3.0
 */
typedef enum {
	WIFI_MANAGER_TDLS_STATE_DISCONNECTED = 0,  /**< Wi-Fi TDLS is Disconnected */
	WIFI_MANAGER_TDLS_STATE_CONNECTED = 1,     /**< Wi-Fi TDLS is Connected */
} wifi_manager_tdls_state_e;

/**
 * @}
 */

/**
 * @addtogroup CAPI_NETWORK_WIFI_MANAGER_MANAGEMENT_MODULE
 * @{
 */

/**
 * @brief The Wi-Fi specific ap scan handle.
 * @since_tizen 4.0
 */
typedef void *wifi_manager_specific_scan_h;

/**
 * @}
 */

/**
 * @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_MODULE
 * @{
 */

/**
 * @brief The Wi-Fi access point handle.
 * @since_tizen 3.0
 */
typedef void *wifi_manager_ap_h;

/**
* @}
*/

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_CONFIG_MODULE
* @{
*/

/**
 * @brief The Wi-Fi access point configuration handle.
 * @since_tizen 3.0
 */
typedef void *wifi_manager_config_h;

/**
* @}
*/

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_MANAGEMENT_MODULE
* @{
*/

/**
 * @brief Enumeration for Wi-Fi Frame type.
 * @since_tizen 5.0
 * @see wifi_manager_add_vsie()
 * @see wifi_manager_remove_vsie()
 * @see wifi_manager_get_vsie()
 */
typedef enum {
	/**
	 * WiFi probe request frame
	 */
	WIFI_MANAGER_VSIE_FRAME_PROBE_REQ,

	/**
	 * WiFi association request frame
	 */
	WIFI_MANAGER_VSIE_FRAME_ASSOC_REQ,

	/**
	 * WiFi re-association request frame
	 */
	WIFI_MANAGER_VSIE_FRAME_REASSOC,

	/**
	 * WiFi authentication request frame
	 */
	WIFI_MANAGER_VSIE_FRAME_AUTH_REQ,

	/**
	 * WiFi action frame
	 */
	WIFI_MANAGER_VSIE_FRAME_ACTION,

} wifi_manager_vsie_frames_e;

/**
 * @brief Called for each found access point.
 * @details This callback function is used in conjunction with `wifi_manager_foreach_found_ap()` or `wifi_manager_foreach_found_specific_ap()` to iterate through all discovered access points.
 * @since_tizen 3.0
 * @remarks @a ap is valid only in this function. In order to use @a ap outside this function, you must copy the @a ap with wifi_manager_ap_clone().
 * @param[in]  ap           The access point
 * @param[in]  user_data    The user data passed from the request function
 * @return  @c true to continue with the next iteration of the loop, \n
 *     otherwise @c false to break out of the loop
 * @pre  wifi_manager_foreach_found_ap() will invoke this callback.
 * @pre  wifi_manager_foreach_found_specific_ap() will invoke this callback.
 * @see  wifi_manager_foreach_found_ap()
 * @see  wifi_manager_foreach_found_specific_ap()
 */
typedef bool(*wifi_manager_found_ap_cb)(wifi_manager_ap_h ap, void *user_data);

/**
 * @brief Called when the scan is finished.
 * @details The following error codes can be received: \n
 *              #WIFI_MANAGER_ERROR_NONE                 Successful \n
 *              #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed \n
 * @since_tizen 3.0
 * @param[in] error_code    The error code
 * @param[in] user_data     The user data passed from the callback registration function
 * @see wifi_manager_scan()
 * @see wifi_manager_set_background_scan_cb()
 * @see wifi_manager_unset_background_scan_cb()
 */
typedef void(*wifi_manager_scan_finished_cb)(wifi_manager_error_e error_code, void *user_data);

/**
 * @brief Called when the scanning state is changed.
 * @details This callback function is registered using `wifi_manager_set_scan_state_changed_cb()` and will be called whenever the scanning state of the Wi-Fi manager changes.
 * @since_tizen 4.0
 * @param[in] state         The wifi scanning state
 * @param[in] user_data     The user data passed from the callback registration function
 * @see wifi_manager_set_scan_state_changed_cb()
 * @see wifi_manager_unset_scan_state_changed_cb()
 */
typedef void(*wifi_manager_scan_state_changed_cb)(wifi_manager_scan_state_e state, void *user_data);

/**
 * @brief Called after wifi_manager_activate() or wifi_manager_activate_with_wifi_picker_tested() is completed.
 * @details The following error codes can be received: \n
 *              #WIFI_MANAGER_ERROR_NONE                 Successful \n
 *              #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed \n
 *              #WIFI_MANAGER_ERROR_SECURITY_RESTRICTED  Restricted by security system policy
 * @since_tizen 3.0
 * @param[in] result        The result
 * @param[in] user_data     The user data passed from wifi_manager_activate() and wifi_manager_activate_with_wifi_picker_tested()
 * @pre wifi_manager_activate() or wifi_manager_activate_with_wifi_picker_tested() will invoke this callback function.
 * @see wifi_manager_activate()
 * @see wifi_manager_activate_with_wifi_picker_tested()
 */
typedef void(*wifi_manager_activated_cb)(wifi_manager_error_e result, void *user_data);

/**
 * @brief Called after wifi_manager_deactivate() is completed.
 * @details The following error codes can be received: \n
 *              #WIFI_MANAGER_ERROR_NONE                 Successful \n
 *              #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed \n
 *              #WIFI_MANAGER_ERROR_SECURITY_RESTRICTED  Restricted by security system policy
 * @since_tizen 3.0
 * @param[in] result        The result
 * @param[in] user_data     The user data passed from wifi_manager_deactivate()
 * @pre wifi_manager_deactivate() will invoke this callback function.
 * @see wifi_manager_deactivate()
 */
typedef void(*wifi_manager_deactivated_cb)(wifi_manager_error_e result, void *user_data);

/**
 * @brief Called when the connection process is completed.
 * @details The following functions start the connection process and will
 *          result in the callback being called:
 *     - wifi_manager_connect()
 *     - wifi_manager_connect_by_wps_pbc()
 *     - wifi_manager_connect_by_wps_pin()
 *     - wifi_manager_connect_by_wps_pbc_without_ssid()
 *     - wifi_manager_connect_by_wps_pin_without_ssid()
 *     - wifi_manager_connect_hidden_ap() \n\n
 * The following error codes can be received: \n
 *              #WIFI_MANAGER_ERROR_NONE                     Successful \n
 *              #WIFI_MANAGER_ERROR_OPERATION_FAILED         Operation failed \n
 *              #WIFI_MANAGER_ERROR_OPERATION_ABORTED        Operation aborted \n
 *              #WIFI_MANAGER_ERROR_ALREADY_EXISTS           Connection already exists \n
 *              #WIFI_MANAGER_ERROR_OUT_OF_RANGE             Out of range \n
 *              #WIFI_MANAGER_ERROR_DHCP_FAILED              DHCP failed \n
 *              #WIFI_MANAGER_ERROR_CONNECT_FAILED           Connect failed \n
 *              #WIFI_MANAGER_ERROR_LOGIN_FAILED             Login failed \n
 *              #WIFI_MANAGER_ERROR_AUTHENTICATION_FAILED    Authentication failed \n
 *              #WIFI_MANAGER_ERROR_ASSOCIATION_FAILED       Association failed \n
 *              #WIFI_MANAGER_ERROR_INVALID_KEY              Invalid key \n
 *              #WIFI_MANAGER_ERROR_PIN_MISSING              PIN missing \n
 *              #WIFI_MANAGER_ERROR_WPS_OVERLAP              WPS Overlap \n
 *              #WIFI_MANAGER_ERROR_WPS_TIMEOUT              WPS Timeout \n
 *              #WIFI_MANAGER_ERROR_WPS_WEP_PROHIBITED       WPS WEP Prohibited
 * @since_tizen 3.0
 * @param[in] result        The result
 * @param[in] user_data     The user data passed from the connection function.
 * @pre Any connection function will invoke this callback.
 * @see wifi_manager_connect()
 * @see wifi_manager_connect_by_wps_pbc()
 * @see wifi_manager_connect_by_wps_pin()
 * @see wifi_manager_connect_by_wps_pbc_without_ssid()
 * @see wifi_manager_connect_by_wps_pin_without_ssid()
 * @see wifi_manager_connect_hidden_ap()
 */
typedef void(*wifi_manager_connected_cb)(wifi_manager_error_e result, void *user_data);

/**
 * @brief Called after wifi_manager_disconnect() is completed.
 * @details The following error codes can be received: \n
 *              #WIFI_MANAGER_ERROR_NONE                 Successful \n
 *              #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @since_tizen 3.0
 * @param[in] result        The result
 * @param[in] user_data     The user data passed from wifi_manager_disconnect()
 * @pre wifi_manager_disconnect() will invoke this callback function.
 * @see wifi_manager_disconnect()
 */
typedef void(*wifi_manager_disconnected_cb)(wifi_manager_error_e result, void *user_data);

/**
 * @brief Called after wifi_manager_forget_ap_async() is completed.
 * @details The following error codes can be received: \n
 *              #WIFI_MANAGER_ERROR_NONE                 Successful \n
 *              #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @since_tizen 5.0
 * @param[in] result        The result
 * @param[in] user_data     The user data passed from wifi_manager_forget_ap_async()
 * @pre wifi_manager_forget_ap_async() will invoke this callback function.
 * @see wifi_manager_forget_ap_async()
 */
typedef void(*wifi_manager_forget_ap_finished_cb)(wifi_manager_error_e result, void *user_data);

/**
 * @brief Called when the BSSID scan is finished.
 * @details The following error codes can be received: \n
 *              #WIFI_MANAGER_ERROR_NONE                 Successful \n
 *              #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @since_tizen 4.0
 * @param[in] error_code    The error code
 * @param[in] user_data     The user data passed from the callback registration function
 * @see wifi_manager_scan()
 * @see wifi_manager_bssid_scan()
 * @see wifi_manager_set_background_scan_cb()
 * @see wifi_manager_unset_background_scan_cb()
 */
typedef void(*wifi_manager_bssid_scan_finished_cb)(wifi_manager_error_e error_code, void *user_data);

/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_MONITOR_MODULE
* @{
*/

/**
 * @brief Called when the device state is changed.
 * @details This callback function is registered using `wifi_manager_set_device_state_changed_cb()` and will be called whenever the state of the Wi-Fi device changes.
 * @since_tizen 3.0
 * @param[in] state         The device state
 * @param[in] user_data     The user data passed from the callback registration function
 * @see wifi_manager_set_device_state_changed_cb()
 * @see wifi_manager_unset_device_state_changed_cb()
 */
typedef void(*wifi_manager_device_state_changed_cb)(wifi_manager_device_state_e state, void *user_data);

/**
 * @brief Called when the connection state is changed.
 * @details This callback function is registered using `wifi_manager_set_connection_state_changed_cb()` and will be called whenever the connection state of a Wi-Fi access point changes.
 * @since_tizen 3.0
 * @param[in] state         The connection state
 * @param[in] ap            The access point
 * @param[in] user_data     The user data passed from the callback registration function
 * @see wifi_manager_set_connection_state_changed_cb()
 * @see wifi_manager_unset_connection_state_changed_cb()
 */
typedef void(*wifi_manager_connection_state_changed_cb)(wifi_manager_connection_state_e state, wifi_manager_ap_h ap, void *user_data);

/**
 * @brief Called when the IP conflict state is changed.
 * @details This callback function is registered using `wifi_manager_set_ip_conflict_cb()` and will be called whenever the IP conflict state changes.
 * @since_tizen 5.0
 * @remarks @a mac should not be freed. @a mac is available only in the callback. To use
 *          outside the callback, make a copy.
 * @param[in] mac           The destination MAC address causing conflict
 * @param[in] state         The current state
 * @param[in] user_data     The user data passed from the callback registration function
 * @see wifi_manager_set_ip_conflict_cb()
 * @see wifi_manager_unset_ip_conflict_cb()
 */
typedef void(*wifi_manager_ip_conflict_cb)(char *mac, wifi_manager_ip_conflict_state_e state, void *user_data);

/**
 * @brief Called when the RSSI of connected Wi-Fi is changed.
 * @details This callback function is registered using `wifi_manager_set_rssi_level_changed_cb()` and will be called whenever the RSSI level of the connected Wi-Fi network changes.
 * @since_tizen 3.0
 * @param[in] rssi_level    The level of RSSI
 * @param[in] user_data     The user data passed from the callback registration function
 * @see wifi_manager_set_rssi_level_changed_cb()
 * @see wifi_manager_unset_rssi_level_changed_cb()
 */
typedef void(*wifi_manager_rssi_level_changed_cb)(wifi_manager_rssi_level_e rssi_level, void *user_data);

/**
 * @brief Called when the Wi-Fi Module state is changed.
 * @details This callback function is registered using `wifi_manager_set_module_state_changed_cb()` and will be called whenever the state of the Wi-Fi module changes.
 * @since_tizen 4.0
 * @param[in] wifi_module_state    The Wi-Fi Module state
 * @param[in] user_data            The user data passed from the callback registration function
 * @see wifi_manager_set_module_state_changed_cb()
 * @see wifi_manager_unset_module_state_changed_cb()
 */
typedef void(*wifi_manager_module_state_changed_cb)(wifi_manager_module_state_e wifi_module_state,
		void *user_data);

/**
* @}
*/

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_CONFIG_MODULE
* @{
*/

/**
 * @brief Called for each found access point configuration.
 * @details This callback function is used in conjunction with `wifi_manager_config_foreach_configuration()` to iterate through all saved access point configurations.
 * @since_tizen 3.0
 * @remarks @a config is valid only in this function. In order to use @a config outside this function, you must copy the config with wifi_manager_config_clone().
 *
 * @param[in]  config       The access point configuration handle
 * @param[in]  user_data    The user data passed from the request function
 *
 * @return  @c true to continue with the next iteration of the loop, otherwise @c false to break out of the loop \n
 * @pre  wifi_manager_config_foreach_configuration() will invoke this callback.
 * @see  wifi_manager_config_foreach_configuration()
 */
typedef bool (*wifi_manager_config_list_cb)(const wifi_manager_config_h config, void *user_data);

/**
* @}
*/

/**
 * @addtogroup CAPI_NETWORK_WIFI_MANAGER_TDLS_MODULE
 * @{
 */

/**
 * @brief Called when the Wi-Fi TDLS state is changed.
 * @details This callback function is registered using `wifi_manager_tdls_set_state_changed_cb()` and
 *          will be called whenever the state of a TDLS connection changes.
 * @since_tizen 3.0
 *
 * @param[in] state          The TDLS state
 * @param[in] peer_mac_addr  The MAC address of the TDLS peer
 * @param[in] user_data      The user data passed from the callback registration function
 * @see wifi_manager_tdls_set_state_changed_cb()
 * @see wifi_manager_tdls_unset_state_changed_cb()
 */
typedef void(*wifi_manager_tdls_state_changed_cb)(wifi_manager_tdls_state_e state,
		char *peer_mac_addr, void *user_data);

/**
 * @brief Called when the Wi-Fi TDLS is discovered.
 * @details This callback function is registered using `wifi_manager_tdls_set_discovered_cb()` and
 *          will be called when the TDLS discovery process is completed.
 * @since_tizen 4.0
 * @remarks @a peer_mac_addr is usable only in the callback. To use outside the callback, make a copy.
 * @param[in] state          The TDLS state
 * @param[in] peer_mac_addr  The MAC address of the TDLS peer
 * @param[in] user_data      The user data passed from the callback registration function
 * @see wifi_manager_tdls_set_discovered_cb()
 * @see wifi_manager_tdls_unset_discovered_cb()
 */
typedef void(*wifi_manager_tdls_discovered_cb)(wifi_manager_tdls_discovery_state_e state,
		char *peer_mac_addr, void *user_data);

/**
 * @}
 */

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_MODULE
* @{
*/

/**
 * @brief Initializes the Wi-Fi service and prepares it for use.
 * @details This function initializes the Wi-Fi service,
 *          allocating necessary resources and preparing it for subsequent Wi-Fi management operations such as scanning for networks,
 *          connecting to a network, or retrieving Wi-Fi information.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @remarks You must release @a wifi using wifi_manager_deinitialize().
 * @param[out] wifi        The Wi-Fi handle
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_ALREADY_INITIALIZED  Already initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_initialize(wifi_manager_h *wifi);

/**
 * @brief Terminates the Wi-Fi service and releases all resources associated with it.
 * @details This function deinitializes the Wi-Fi service, releasing all allocated resources and terminating any ongoing operations related to Wi-Fi management.
 * @since_tizen 3.0
 * @param[in] wifi        The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE               Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED    Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER  Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION  Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED   Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED      Not supported
 */
int wifi_manager_deinitialize(wifi_manager_h wifi);

/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_MANAGEMENT_MODULE
* @{
*/

/**
 * @brief Activates Wi-Fi asynchronously.
 * @details This function initiates the process to activate the Wi-Fi service.
 *          When Wi-Fi is activated, the Wi-Fi radio is turned on,
 *          and the device becomes capable of connecting to wireless networks.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi          The Wi-Fi handle
 * @param[in] callback      The callback function to be called \n
 *            This can be @c NULL if you don't want to get the notification.
 * @param[in] user_data     The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOW_IN_PROGRESS      Now in progress
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_SECURITY_RESTRICTED  Restricted by security system policy
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post wifi_manager_activated_cb() will be invoked.
 * @see wifi_manager_activated_cb()
 * @see wifi_manager_deactivate()
 */
int wifi_manager_activate(wifi_manager_h wifi, wifi_manager_activated_cb callback, void *user_data);

/**
 * @brief Activates Wi-Fi asynchronously and displays Wi-Fi picker (popup) when Wi-Fi is not automatically connected.
 * @details This function activates the Wi-Fi service and, if Wi-Fi does not automatically connect to a known network,
 *          it will display a Wi-Fi picker (popup) allowing the user to manually select a network to connect to.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi          The Wi-Fi handle
 * @param[in] callback      The callback function to be called \n
 *            This can be @c NULL if you don't want to get the notification.
 * @param[in] user_data     The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOW_IN_PROGRESS      Now in progress
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_SECURITY_RESTRICTED  Restricted by security system policy
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post wifi_manager_activated_cb() will be invoked.
 * @see wifi_manager_activated_cb()
 * @see wifi_manager_deactivate()
 */
int wifi_manager_activate_with_wifi_picker_tested(wifi_manager_h wifi,
		wifi_manager_activated_cb callback, void *user_data);

/**
 * @brief Deactivates Wi-Fi asynchronously.
 * @details This function initiates the process to deactivate the Wi-Fi service. When Wi-Fi is deactivated,
 *          the Wi-Fi radio is turned off, and the device will no longer be able to connect to wireless networks.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi          The Wi-Fi handle
 * @param[in] callback      The callback function to be called \n
 *            This can be @c NULL if you don't want to get the notification.
 * @param[in] user_data     The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post wifi_manager_deactivated_cb() will be invoked.
 * @see wifi_manager_deactivated_cb()
 * @see wifi_manager_activate()
 */
int wifi_manager_deactivate(wifi_manager_h wifi, wifi_manager_deactivated_cb callback, void *user_data);

/**
 * @brief Checks whether Wi-Fi is activated.
 * @details This function checks the status of the Wi-Fi service to determine if it is currently activated.
 *          If Wi-Fi is activated, it means that the Wi-Fi radio is turned on and ready to connect to wireless networks.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi        The Wi-Fi handle
 * @param[out] activated  @c true if Wi-Fi is activated,
 *             otherwise @c false if Wi-Fi is not activated.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_is_activated(wifi_manager_h wifi, bool *activated);

/**
 * @brief Gets the local MAC address.
 * @details This function queries the Wi-Fi manager to obtain the MAC address of the Wi-Fi device.
 *          The MAC address is a unique identifier assigned to network interfaces for communications on a physical network segment.
 * @since_tizen 3.0
 * @remarks You must release @a mac_address using free().
 * @param[in] wifi          The Wi-Fi handle
 * @param[out] mac_address  The MAC address
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_mac_address(wifi_manager_h wifi, char **mac_address);

/**
 * @brief Gets the name of the network interface.
 * @details This function queries the Wi-Fi manager to obtain the name of the network interface that is currently being used for the Wi-Fi connection.
 *          The network interface name is typically a string identifier like 'wlan0' or 'eth0', which can be used for various networking operations.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @remarks You must release @a name using free().
 * @param[in] wifi        The Wi-Fi handle
 * @param[out] name       The name of network interface
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_network_interface_name(wifi_manager_h wifi, char **name);

/**
 * @brief Starts scan asynchronously.
 * @details This function starts a Wi-Fi scan to discover all available access points (APs) in the vicinity.
 *          The scan results will include information about each detected AP, such as its SSID, signal strength, and security type.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_scan_finished_cb().
 */
int wifi_manager_scan(wifi_manager_h wifi, wifi_manager_scan_finished_cb callback, void *user_data);

/**
 * @brief Gets the Wi-Fi scan state.
 * @details This function queries the Wi-Fi manager to determine the current state of the Wi-Fi scan operation.
 *          The scan state indicates whether a scan is currently in progress or if the results from the last scan are still valid.
 * @since_tizen 4.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] scan_state      The Wi-Fi scan state
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_scan_state(wifi_manager_h wifi, wifi_manager_scan_state_e *scan_state);

/**
 * @brief Starts specific AP scan, asynchronously.
 * @details This function starts a Wi-Fi scan targeting a specific access point (AP) with the given ESSID (Extended Service Set Identifier).
 *          Unlike a regular scan, which searches for all available APs in the vicinity, a specific scan focuses on a single AP with the specified ESSID,
 *          potentially improving scan performance and reducing scan time.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] essid           The essid of specific AP
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_scan_finished_cb().
 */
int wifi_manager_scan_specific_ap(wifi_manager_h wifi,
		const char *essid, wifi_manager_scan_finished_cb callback, void *user_data);

/**
 * @brief Creates a Wi-Fi specific AP scan handle.
 * @details This function creates a new specific AP scan handle, which can be used to configure and initiate specific AP scans.
 *          The handle is required for setting up specific scan parameters such as SSID and channel frequency.
 * @since_tizen 4.0
 * @remarks You must release @a specific_scan using wifi_manager_specific_scan_destroy().
 * @param[in] wifi              The Wi-Fi handle
 * @param[out] specific_scan    The Wi-Fi specific AP scan handle
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre This function needs a wifi_manager_initialize() call before use.
 * @see wifi_manager_specific_scan_destroy()
 */
int wifi_manager_specific_scan_create(wifi_manager_h wifi, wifi_manager_specific_scan_h *specific_scan);

/**
 * @brief Destroys a Wi-Fi specific AP scan handle.
 * @details This function releases the memory allocated for a specific AP scan handle,
 *          which was created using `wifi_manager_specific_scan_create()`.
 *          It is important to destroy the handle when it is no longer needed to avoid memory leaks.
 * @since_tizen 4.0
 * @param[in] wifi              The Wi-Fi handle
 * @param[in] specific_scan     The Wi-Fi specific AP scan handle
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre This function needs a wifi_manager_initialize() call before use.
 * @see wifi_manager_specific_scan_create()
 */
int wifi_manager_specific_scan_destroy(wifi_manager_h wifi, wifi_manager_specific_scan_h specific_scan);

/**
 * @brief Gets the maximum number of SSIDs supported by the Wi-Fi chipset for the scan operation.
 * @details This function queries the Wi-Fi manager to determine the maximum number of SSIDs that the Wi-Fi chipset can support during a single scan operation.
 *          This value is important when configuring specific AP scans, as it limits the number of SSIDs that can be targeted in a single scan request.
 * @since_tizen 5.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi		The Wi-Fi handle
 * @param[out] max_scan_ssids	The maximum number of SSIDs supported by the Wi-Fi \n
 *                              chipset for the scan operation.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_specific_scan_set_ssid()
 */
int wifi_manager_specific_scan_get_max_ssids(wifi_manager_h wifi, int *max_scan_ssids);

/**
 * @brief Sets the SSID of a specific AP scan.
 * @details This function sets the SSID for a specific AP scan, which determines the target access point to be scanned.
 *          By specifying the SSID, the scan can focus on a particular AP rather than scanning for all available APs in the vicinity,
 *          potentially improving scan performance and reducing scan time.
 * @since_tizen 4.0
 * @param[in] specific_scan     The Wi-Fi specific AP scan handle
 * @param[in] essid             The SSID of specific AP scan
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_specific_scan_get_max_ssids()
 */
int wifi_manager_specific_scan_set_ssid(wifi_manager_specific_scan_h specific_scan, const char *essid);

/**
 * @brief Sets the channel frequency of a specific AP scan.
 * @details This function sets the channel frequency for a specific AP scan, which determines the radio frequency band on which the scan will be performed.
 *          By specifying the channel frequency, the scan can be focused on a particular frequency band, potentially improving scan performance and reducing scan time.
 * @since_tizen 4.0
 * @param[in] specific_scan     The Wi-Fi specific AP scan handle
 * @param[in] freq              The channel frequency of specific AP scan
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_specific_scan_set_freq(wifi_manager_specific_scan_h specific_scan, int freq);

/**
 * @brief Starts multi SSID and multi channel specific scan, asynchronously.
 * @details This function starts a Wi-Fi scan targeting multiple specific access points (APs) across multiple channels simultaneously.
 *          Unlike a regular scan, which searches for all available APs in the vicinity, 
 *          a specific scan focuses on a predefined set of APs, which can improve scan efficiency and reduce scan time. 
 *          The multi-SSID and multi-channel specific scan extends this concept further by allowing the scan to target multiple sets of APs across different channels at once.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] specific_scan   The Wi-Fi specific AP scan handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_scan_finished_cb().
 */
int wifi_manager_specific_ap_start_multi_scan(wifi_manager_h wifi,
		wifi_manager_specific_scan_h specific_scan,
		wifi_manager_scan_finished_cb callback, void *user_data);

/**
 * @brief Flushes BSS entries from the cache.
 * @details This function removes all previously discovered BSS entries from the Wi-Fi manager's internal cache.
 *          BSS entries include information about nearby Wi-Fi access points (APs) that have been scanned and stored by the Wi-Fi manager.
 *          By flushing these entries, the Wi-Fi manager effectively forgets about previously discovered APs,
 *          which can be useful in scenarios where the environment has changed significantly or when privacy concerns dictate that old scan results should be discarded.
 * @since_tizen 7.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi           The Wi-Fi handle
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE  Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 */
int wifi_manager_flush_bss(wifi_manager_h wifi);

/**
 * @brief Connects to the hidden AP, asynchronously.
 * @details This function attempts to connect to a hidden Wi-Fi access point (AP) with the specified ESSID, security type, and passphrase.
 *          Hidden APs do not broadcast their presence, so the ESSID must be explicitly provided to the function.
 *          The connection process is performed asynchronously, meaning that the function returns immediately,
 *          and the result of the connection attempt is reported via the provided callback function.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get \n
 *            %http://tizen.org/privilege/network.profile
 * @remarks This function needs all listed privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] essid           The ESSID of the hidden AP
 * @param[in] sec_type        The security type of the hidden AP
 * @param[in] passphrase      The passphrase of the hidden AP
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre This function needs wifi_manager_scan() before use.
 * @post This function invokes wifi_manager_connected_cb().
 */
int wifi_manager_connect_hidden_ap(wifi_manager_h wifi,
		const char *essid, wifi_manager_security_type_e sec_type,
		const char *passphrase, wifi_manager_connected_cb callback, void *user_data);

/**
 * @brief Gets the handle of the connected access point.
 * @details This function retrieves the handle of the access point to which the Wi-Fi interface is currently connected.
 *          If there is no active connection, the function will return an error.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @remarks You must release @a ap using wifi_manager_ap_destroy().
 * @param[in] wifi            The Wi-Fi handle
 * @param[out] ap             The access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NO_CONNECTION        There is no connected AP
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_connected_ap(wifi_manager_h wifi, wifi_manager_ap_h *ap);

/**
 * @brief Gets the result of the scan.
 * @details This function iterates through the list of access points found during a general Wi-Fi scan and provides their details to the caller via the provided callback function.
 *          A general Wi-Fi scan searches for all available access points in the vicinity, regardless of whether they are known or unknown to the device.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_found_ap_cb().
 */
int wifi_manager_foreach_found_ap(wifi_manager_h wifi, wifi_manager_found_ap_cb callback, void *user_data);

/**
 * @brief Gets the result of specific AP scan.
 * @details This function iterates through the list of access points found during a specific access point scan
 *          and provides their details to the caller via the provided callback function.
 *          A specific access point scan is used to focus on a particular access point,
 *          rather than scanning all available access points in the vicinity.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_found_ap_cb().
 * @see wifi_manager_scan_specific_ap()
 */
int wifi_manager_foreach_found_specific_ap(wifi_manager_h wifi,
		wifi_manager_found_ap_cb callback, void *user_data);

/**
 * @brief Gets the result of the BSSID scan (i.e. BSSID, ESSID & RSSI).
 * @details This function iterates through the list of access points found during a BSSID scan and provides their details to the caller via the provided callback function.
 *          The BSSID scan focuses on identifying access points based on their unique MAC addresses (BSSID), allowing for more precise identification compared to a regular scan.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_found_ap_cb().
 * @see wifi_manager_scan()
 * @see wifi_manager_scan_specific_ap()
 */
int wifi_manager_foreach_found_bssid_ap(wifi_manager_h wifi,
				wifi_manager_found_ap_cb callback, void *user_data);


/**
 * @brief Connects to the access point asynchronously.
 * @details This function initiates a connection to the specified Wi-Fi access point.
 *          The connection process is performed asynchronously, meaning that the function returns immediately,
 *          and the result of the connection attempt is reported via the provided callback function.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] ap              The access point handle
 * @param[in] callback        The callback function to be called \n
 *            This can be @c NULL if you don't want to get the notification.
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOW_IN_PROGRESS      Now in progress
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_connected_cb().
 * @see wifi_manager_connected_cb()
 * @see wifi_manager_connect_by_wps_pbc()
 * @see wifi_manager_connect_by_wps_pin()
 * @see wifi_manager_disconnect()
 */
int wifi_manager_connect(wifi_manager_h wifi, wifi_manager_ap_h ap, wifi_manager_connected_cb callback, void *user_data);

/**
 * @brief Disconnects to the access point asynchronously.
 * @details This function disrupts the existing connection to the specified Wi-Fi access point.
 *          The disconnection process is performed asynchronously, meaning that the function returns immediately,
 *          and the result of the disconnection attempt is reported via the provided callback function.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] ap              The access point handle
 * @param[in] callback        The callback function to be called \n
 *            This can be @c NULL if you don't want to get the notification.
 * @param[in] user_data        The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_disconnected_cb().
 * @see wifi_manager_disconnected_cb()
 * @see wifi_manager_connect_by_wps_pbc()
 * @see wifi_manager_connect_by_wps_pin()
 * @see wifi_manager_connect()
 */
int wifi_manager_disconnect(wifi_manager_h wifi, wifi_manager_ap_h ap, wifi_manager_disconnected_cb callback, void *user_data);

/**
 * @brief Connects to the access point with WPS PBC asynchronously.
 * @details This function initiates a connection to a Wi-Fi access point using WPS PBC authentication.
 *          The connection process is performed asynchronously, meaning that the function returns immediately,
 *          and the result of the connection attempt is reported via the provided callback function.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] ap              The access point handle
 * @param[in] callback        The callback function to be called \n
 *            This can be NULL if you don't want to get the notification.
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOW_IN_PROGRESS      Now in progress
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_connected_cb().
 * @see wifi_manager_connected_cb()
 * @see wifi_manager_connect()
 * @see wifi_manager_disconnect()
 * @see wifi_manager_ap_is_wps_supported()
 */
int wifi_manager_connect_by_wps_pbc(wifi_manager_h wifi,
		wifi_manager_ap_h ap, wifi_manager_connected_cb callback, void *user_data);

/**
 * @brief Connects to the access point with WPS PIN asynchronously.
 * @details This function initiates a connection to a Wi-Fi access point using WPS PIN authentication.
 *          The connection process is performed asynchronously, meaning that the function returns immediately,
 *          and the result of the connection attempt is reported via the provided callback function.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] ap              The access point handle
 * @param[in] pin             The WPS PIN is a non-NULL string with length greater than 0 and less than or equal to 8
 * @param[in] callback        The callback function to be called (this can be NULL if you don't want to get the notification)
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOW_IN_PROGRESS      Now in progress
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_connected_cb().
 * @see wifi_manager_connected_cb()
 * @see wifi_manager_connect()
 * @see wifi_manager_disconnect()
 * @see wifi_manager_ap_is_wps_supported()
 */
int wifi_manager_connect_by_wps_pin(wifi_manager_h wifi,
		wifi_manager_ap_h ap, const char *pin, wifi_manager_connected_cb callback, void *user_data);

/**
 * @brief Connects to the access point with WPS PBC without entering SSID.
 * @details This function initiates a connection to a Wi-Fi access point using WPS PBC authentication.
 *          The SSID of the access point does not need to be known;
 *          the Wi-Fi module will search for and connect to the access point based on the WPS PBC signal.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get \n
 *            %http://tizen.org/privilege/network.profile
 * @remarks This function needs all listed privileges.
 * @param[in] wifi      The Wi-Fi handle
 * @param[in] callback  The callback function to be called \n
 *			This can be NULL if you don't want to get the notification.
 * @param[in] user_data The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOW_IN_PROGRESS      Now in progress
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_connected_cb().
 * @see wifi_manager_connected_cb()
 * @see wifi_manager_connect()
 * @see wifi_manager_disconnect()
 * @see wifi_manager_ap_is_wps_supported()
 */
int wifi_manager_connect_by_wps_pbc_without_ssid(wifi_manager_h wifi,
		wifi_manager_connected_cb callback, void *user_data);

/**
 * @brief Connects to the access point with WPS PIN without entering SSID.
 * @details This function initiates a connection to a Wi-Fi access point using WPS PIN authentication.
 *          The SSID of the access point does not need to be known;
 *          the Wi-Fi module will search for and connect to the access point based on the provided WPS PIN.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get \n
 *            %http://tizen.org/privilege/network.profile
 * @remarks This function needs all listed privileges.
 * @param[in] wifi      The Wi-Fi handle
 * @param[in] pin       The WPS PIN, a non-NULL string with length greater \n
 *                      than 0 and less than or equal to 8
 * @param[in] callback  The callback function to be called \n
 *                      This can be NULL if you don't want to get the notification.
 * @param[in] user_data The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOW_IN_PROGRESS      Now in progress
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_connected_cb().
 * @see wifi_manager_connected_cb()
 * @see wifi_manager_connect()
 * @see wifi_manager_disconnect()
 * @see wifi_manager_ap_is_wps_supported()
 */
int wifi_manager_connect_by_wps_pin_without_ssid(wifi_manager_h wifi,
		 const char *pin, wifi_manager_connected_cb callback, void *user_data);

/**
 * @brief Stops ongoing WPS provisioning / disconnects from the connected access point.
 * @details This function stops any ongoing WPS provisioning process initiated by `wifi_manager_start_wps()` or disconnects
 *          from the currently connected access point if there is no ongoing WPS provisioning.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi      The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
*/
int wifi_manager_cancel_wps(wifi_manager_h wifi);

/**
 * @brief Gets the WPS generated PIN code.
 * @details This function queries the Wi-Fi manager to obtain the WPS-generated PIN code.
 *          The retrieved PIN code can be used for WPS-based authentication when connecting to a Wi-Fi access point.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @remarks You must release @a wps_pin using free().
 * @param[in] wifi            The Wi-Fi handle
 * @param[out] wps_pin        The WPS PIN
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_wps_generated_pin(wifi_manager_h wifi, char **wps_pin);

/**
 * @brief Deletes the information of a stored access point and disconnects from it if it was connected to.
 * @details If an AP is connected to, then connection information will be stored.
 * This information is used when a connection to that AP is established automatically.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] ap              The access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_forget_ap(wifi_manager_h wifi, wifi_manager_ap_h ap);

/**
 * @brief Deletes stored access point's information and disconnects from it if connected, asynchronously.
 * @details This function initiates an asynchronous process to delete the stored information of the specified access point and disconnect from it if currently connected.
 *          The result of the operation is reported via the provided callback function.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] ap              The access point handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_forget_ap_finished_cb().
 */
int wifi_manager_forget_ap_async(wifi_manager_h wifi, wifi_manager_ap_h ap,
		wifi_manager_forget_ap_finished_cb callback, void *user_data);

/**
 * @brief Updates an existing AP.
 * @details When a AP is changed, these changes will be not applied to the Connection Manager immediately.
 *          When you call this function, your changes affect the Connection Manager and the existing AP is updated.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] ap              The access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_update_ap(wifi_manager_h wifi, wifi_manager_ap_h ap);

/**
 * @brief Adds the Wi-Fi Vendor Specific Information Element (VSIE) to a specific frame type.
 * @details This function adds the specified VSIE data to the given frame ID.
 *          The VSIE data will be included in the corresponding frame when transmitted by the Wi-Fi module.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 * @remarks @a vsie_str for @a frame_id will be in effect until Wi-Fi is deactivated.
 *
 * @param[in] wifi           The Wi-Fi handle
 * @param[in] frame_id       The frame ID for setting VSIE
 * @param[in] vsie_str       The VSIE data
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre Wi-Fi service must be activated by wifi_manager_activate().
 * @see wifi_manager_activate()
 * @see wifi_manager_remove_vsie()
 * @see wifi_manager_get_vsie()
 */
int wifi_manager_add_vsie(wifi_manager_h wifi,
		wifi_manager_vsie_frames_e frame_id, const char *vsie_str);

/**
 * @brief Gets the Wi-Fi Vendor Specific Information Elements (VSIE) from a specific frame.
 * @details This function retrieves the VSIE data associated with the given frame ID.
 *          The retrieved VSIE data can be used to inspect or modify the contents of the VSIE element within the specified frame.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @remarks @a vsie_str must be released with free().
 *
 * @param[in] wifi           The Wi-Fi handle
 * @param[in] frame_id       The frame ID for getting VSIE
 * @param[out] vsie_str      The VSIE data
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre Wi-Fi service must be activated by wifi_manager_activate()
 *      and VSIE for a specific frame must be set with wifi_manager_add_vsie().
 * @see wifi_manager_activate()
 * @see wifi_manager_remove_vsie()
 * @see wifi_manager_add_vsie()
 */
int wifi_manager_get_vsie(wifi_manager_h wifi,
		wifi_manager_vsie_frames_e frame_id, char **vsie_str);

/**
 * @brief Removes the Wi-Fi Vendor Specific Information Element (VSIE) from specific frame.
 * @details This function removes the specified VSIE data from the given frame ID.
 *          The VSIE data will no longer be included in the corresponding frame when transmitted by the Wi-Fi module.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 * @remarks @a vsie_str for @a frame_id will be in effect until Wi-Fi is deactivated.
 *
 * @param[in] wifi           The wifi handle
 * @param[in] frame_id       The frame ID for removing VSIE
 * @param[in] vsie_str       The VSIE data
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre Wi-Fi service must be activated by wifi_manager_activate()
 *      and VSIE for a specific frame must be set with wifi_manager_add_vsie().
 * @see wifi_manager_activate()
 * @see wifi_manager_get_vsie()
 * @see wifi_manager_add_vsie()
 */
int wifi_manager_remove_vsie(wifi_manager_h wifi,
		wifi_manager_vsie_frames_e frame_id, const char *vsie_str);

/**
* @}
*/

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_MONITOR_MODULE
* @{
*/

/**
 * @brief Gets the connection state.
 * @details This function queries the Wi-Fi manager to obtain the current connection state of the Wi-Fi network associated with the given Wi-Fi handle.
 *          The retrieved connection state can be one of the following: DISCONNECTED, CONNECTED, or CONNECTING.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi                The Wi-Fi handle
 * @param[out] connection_state   The connection state
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_connection_state(wifi_manager_h wifi,
		wifi_manager_connection_state_e *connection_state);

/**
 * @brief Registers the callback called when the device state is changed.
 * @details This function registers a callback function to monitor changes in the Wi-Fi device state.
 *          When the Wi-Fi device state changes (e.g., enabled, disabled, or enabling/disabling in progress),
 *          the registered callback function will be invoked with the updated device state and the provided user data.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_set_device_state_changed_cb(wifi_manager_h wifi,
		wifi_manager_device_state_changed_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when the device state is changed.
 * @details This function removes the previously registered callback for monitoring changes in the Wi-Fi device state.
 *          The callback will no longer be invoked when the device state changes after this function is called.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_unset_device_state_changed_cb(wifi_manager_h wifi);

/**
 * @brief Registers the callback called when the background scan is finished.
 * @details This function registers a callback function to monitor the completion of background scans initiated by the Wi-Fi manager.
 *          When a background scan finishes, the registered callback function will be invoked with the scan results and the provided user data.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_set_background_scan_cb(wifi_manager_h wifi,
		wifi_manager_scan_finished_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when the scan is finished.
 * @details This function removes the previously registered callback for monitoring the completion of background scans initiated by the Wi-Fi manager.
 *          The callback will no longer be invoked when a background scan finishes after this function is called.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_unset_background_scan_cb(wifi_manager_h wifi);

/**
 * @brief Registers the callback called when the scanning state is changed.
 * @details This function registers a callback function to monitor changes in the scanning state of the Wi-Fi network.
 *          When the scanning state changes (e.g., scan started or scan finished),
 *          the registered callback function will be invoked with the updated scanning state and the provided user data.
 * @since_tizen 4.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_set_scan_state_changed_cb(wifi_manager_h wifi,
		wifi_manager_scan_state_changed_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when the scanning state is changed.
 * @details This function removes the previously registered callback for monitoring changes in the scanning state of the Wi-Fi network.
 *          The callback will no longer be invoked when the scanning state changes after this function is called.
 * @since_tizen 4.0
 * @param[in] wifi            The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_unset_scan_state_changed_cb(wifi_manager_h wifi);

/**
 * @brief Registers the callback called when the connection state is changed.
 * @details This function registers a callback function to monitor changes in the connection state of the Wi-Fi network.
 *          When the connection state changes (e.g., connected, disconnected, or connection failure),
 *          the registered callback function will be invoked with the updated connection state and the provided user data.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_set_connection_state_changed_cb(wifi_manager_h wifi,
		wifi_manager_connection_state_changed_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when the connection state is changed.
 * @details This function removes the previously registered callback for monitoring changes in the connection state of the Wi-Fi network.
 *          The callback will no longer be invoked when the connection state changes after this function is called.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_unset_connection_state_changed_cb(wifi_manager_h wifi);

/**
 * @brief Registers callback called when the RSSI of connected Wi-Fi is changed.
 * @details This function registers a callback function to monitor changes in the Received Signal Strength Indication (RSSI) level of the currently connected Wi-Fi network.
 *          When the RSSI level changes, the registered callback function will be invoked with the updated RSSI level and the provided user data.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_set_rssi_level_changed_cb(wifi_manager_h wifi,
		wifi_manager_rssi_level_changed_cb callback, void *user_data);

/**
 * @brief Unregisters callback called when the RSSI of connected Wi-Fi is changed.
 * @details This function removes the previously registered callback for monitoring changes in the Received Signal Strength Indication (RSSI) level of the currently connected Wi-Fi network.
 *          The callback will no longer be invoked when the RSSI level changes after this function is called.
 * @since_tizen 3.0
 * @param[in] wifi            The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_unset_rssi_level_changed_cb(wifi_manager_h wifi);

/**
 * @brief Registers a callback called when the Wi-Fi Module state is changed.
 * @details This function registers a callback function to monitor changes in the Wi-Fi module state.
 *          When the state of the Wi-Fi module changes (e.g., enabled or disabled),
 *          the registered callback function will be invoked with the updated state and the provided user data.
 * @since_tizen 4.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_module_state_changed_cb()
 * @see wifi_manager_unset_module_state_changed_cb()
*/
int wifi_manager_set_module_state_changed_cb(wifi_manager_h wifi,
		wifi_manager_module_state_changed_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when the Wi-Fi Module state is changed.
 * @details This function removes the previously registered callback for monitoring changes in the Wi-Fi module state.
 *          The callback will no longer be invoked when the Wi-Fi module state changes after this function is called.
 * @since_tizen 4.0
 * @param[in] wifi            The Wi-Fi handle
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_module_state_changed_cb()
 * @see wifi_manager_set_module_state_changed_cb()
 */
int wifi_manager_unset_module_state_changed_cb(wifi_manager_h wifi);

/**
 * @brief Gets the Wi-Fi Module state.
 * @details This function retrieves the current state of the Wi-Fi module.
 *          The Wi-Fi module state indicates whether the Wi-Fi hardware is enabled or disabled.
 *          This information is useful for determining if the device is capable of connecting to wireless networks.
 * @since_tizen 4.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[out] state          The Wi-Fi Module state
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_module_state(wifi_manager_h wifi, wifi_manager_module_state_e *state);

/**
 * @brief Starts BSSID scan asynchronously.
 * @details This function starts a BSSID (Basic Service Set Identifier) scan to discover wireless access points.
 *          A BSSID scan focuses on finding specific access points based on their MAC addresses rather than scanning for all available networks.
 *          The scan is performed asynchronously, and the result is reported through the provided callback function.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set \n
 *            %http://tizen.org/privilege/network.get
 * @remarks This function needs both privileges.
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_bssid_scan_finished_cb().
 */
int wifi_manager_bssid_scan(wifi_manager_h wifi,
		wifi_manager_bssid_scan_finished_cb callback, void *user_data);

/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_MODULE
* @{
*/

/**
 * @brief Creates the access point handle.
 * @details This function creates a new access point handle for a visible wireless network.
 *          The ESSID (Extended Service Set Identifier) should be provided as a null-terminated string and can be encoded in UTF-8.
 *          The created access point handle can be used to connect to the specified wireless network using the Wi-Fi handle.
 * @since_tizen 3.0
 * @remarks You must release @a ap using wifi_manager_ap_destroy().
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] essid           The ESSID (Extended Service Set Identifier) should be null-terminated and can be UTF-8 encoded
 * @param[out] ap             The access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_ap_destroy()
 */
int wifi_manager_ap_create(wifi_manager_h wifi, const char *essid, wifi_manager_ap_h *ap);

/**
 * @brief Creates the hidden access point handle.
 * @details This function creates a new access point handle for a hidden wireless network.
 *          Hidden networks do not broadcast their SSID (Service Set Identifier),
 *          so you need to provide the ESSID (Extended Service Set Identifier) explicitly when creating the access point handle.
 *          The ESSID should be a null-terminated string and can be encoded in UTF-8.
 * @since_tizen 3.0
 * @remarks You must release @a ap using wifi_manager_ap_destroy().
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] essid           The ESSID (Extended Service Set Identifier) should be null-terminated and can be UTF-8 encoded
 * @param[out] ap             The access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_ap_destroy()
 */
int wifi_manager_ap_hidden_create(wifi_manager_h wifi, const char *essid, wifi_manager_ap_h *ap);

/**
 * @brief Destroys the access point handle.
 * @details This function releases all resources allocated for the given access point handle.
 *          After calling this function, the access point handle becomes invalid and cannot be used for any further operations.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_ap_create()
 */
int wifi_manager_ap_destroy(wifi_manager_ap_h ap);

/**
 * @brief Clones the access point handle.
 * @details This function creates a new access point handle that is a copy of the given access point handle.
 *          The cloned access point handle contains the same information as the original handle,
 *          allowing you to work with the same access point data without modifying the original handle.
 * @since_tizen 3.0
 * @remarks You must release @a cloned_ap using wifi_manager_ap_destroy().
 * @param[out] cloned_ap      The cloned access point handle
 * @param[in] origin          The origin access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_ap_destroy()
 */
int wifi_manager_ap_clone(wifi_manager_ap_h *cloned_ap, wifi_manager_ap_h origin);

/**
 * @brief Refreshes the access point information.
 * @details This function refreshes the access point information.
 *          Access point information can change over time due to various factors such as changes in the environment,
 *          network configuration updates, or other network events.
 *          By calling this function, you ensure that the latest information about the access point is retrieved and stored in the given access point handle.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @remarks You should call this function in order to get the current access point information, because the information can be changed.
 * @param[in] ap              The access point handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_refresh(wifi_manager_ap_h ap);


/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_NETWORK_MODULE
* @{
*/

/**
 * @brief Gets ESSID (Extended Service Set Identifier).
 * @details This function retrieves the Extended Service Set Identifier (ESSID) for the access point.
 *          The ESSID is a unique identifier assigned to a group of wireless access points that form an extended service set,
 *          typically represented as a sequence of characters (e.g., "MyHomeNetwork").
 *          In most cases, the ESSID is identical to the SSID of the primary access point within the extended service set.
 * @since_tizen 3.0
 * @remarks You must release @a essid using free().
 * @param[in] ap              The access point handle
 * @param[out] essid          The ESSID
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_essid(wifi_manager_ap_h ap, char **essid);

/**
 * @brief Gets raw SSID (Service Set Identifier).
 * @details This function retrieves the raw Service Set Identifier (SSID) for the access point.
 *          The SSID is a unique identifier assigned to a wireless network, typically represented as a sequence of characters (e.g., "MyHomeNetwork").
 *          This function returns the raw SSID as a byte array, allowing direct access to the underlying data without any character encoding considerations.
 * @since_tizen 4.0
 * @remarks You must release @a ssid using free().
 * @param[in] ap              The access point handle
 * @param[out] ssid           The raw SSID bytes
 * @param[out] ssid_len       The raw SSID length
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_raw_ssid(wifi_manager_ap_h ap, char **ssid, int *ssid_len);

/**
 * @brief Gets BSSID (Basic Service Set Identifier).
 * @details This function retrieves the Basic Service Set Identifier (BSSID) for the access point.
 *          The BSSID is a unique identifier assigned to each wireless access point,
 *          consisting of a MAC address formatted as a string (e.g., "00:1A:2B:3C:4D:5E").
 * @since_tizen 3.0
 * @remarks You must release @a bssid using free().
 * @param[in] ap              The access point handle
 * @param[out] bssid          The BSSID
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_bssid(wifi_manager_ap_h ap, char **bssid);

/**
 * @brief Gets the RSSI.
 * @details This function retrieves the Received Signal Strength Indication (RSSI) value for the access point.
 *          The RSSI value indicates the strength of the received signal from the access point, measured in decibels relative to milliwatts (dBm).
 *          A higher RSSI value generally indicates a stronger signal and better connection quality.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[out] rssi           The RSSI value (in dBm)
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_rssi(wifi_manager_ap_h ap, int *rssi);

/**
 * @brief Gets the RSSI level.
 * @details This function retrieves the Received Signal Strength Indication (RSSI) level for the access point.
 *          The RSSI level indicates the strength of the received signal from the access point,
 *          which can be used to determine the quality of the wireless connection.
 * @since_tizen 4.0
 * @param[in] ap              The access point handle
 * @param[out] rssi_level     The RSSI level
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_rssi_level(wifi_manager_ap_h ap, wifi_manager_rssi_level_e *rssi_level);

/**
 * @brief Gets the frequency band (MHz).
 * @details This function retrieves the frequency band (in megahertz, MHz) for the access point.
 *          The frequency band indicates the specific radio frequency range used by the access point to transmit and receive wireless signals.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[out] frequency      The frequency
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_frequency(wifi_manager_ap_h ap, int *frequency);

/**
 * @brief Gets the max speed (Mbps).
 * @details This function retrieves the maximum achievable speed (in megabits per second, Mbps) for the access point.
 *          The maximum speed indicates the highest possible data transfer rate that can be achieved when connected to the access point.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[out] max_speed      The max speed
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_max_speed(wifi_manager_ap_h ap, int *max_speed);

/**
 * @brief Checks whether the access point is favorite or not.
 * @details This function determines whether the access point is marked as a favorite or not.
 *          A favorite access point is one that the user has explicitly chosen to connect to frequently or has saved for future use.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[out] favorite  @c true if access point is favorite,
 *             otherwise @c false if access point is not favorite
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_is_favorite(wifi_manager_ap_h ap, bool *favorite);

/**
 * @brief Checks whether the access point is passpoint or not.
 * @details This function determines whether the access point is a Passpoint or not.
 *          A Passpoint access point is a type of Wi-Fi hotspot that supports the Hotspot 2.0 standard,
 *          providing seamless and secure connectivity across various networks.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[out] passpoint  @c true if access point is passpoint,
 *              otherwise @c false if access point is not passpoint.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_is_passpoint(wifi_manager_ap_h ap, bool *passpoint);

/**
 * @brief Checks whether the access point is hidden or not.
 * @details This function determines whether the access point is hidden or not.
 *          A hidden access point does not broadcast its SSID (Service Set Identifier), making it less visible to potential clients.
 * @since_tizen 5.5
 * @param[in] ap              The access point handle
 * @param[out] is_hidden  @c true if the access point is hidden,
 *                        @c false if the access point is not hidden.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_is_hidden(wifi_manager_ap_h ap, bool *is_hidden);

/**
 * @brief Gets the connection state.
 * @details This function retrieves the current connection state of the access point.
 *          The connection state indicates whether the access point is currently connected, connecting, or disconnected.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[out] state          The connection state
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_connection_state(wifi_manager_ap_h ap, wifi_manager_connection_state_e *state);

/**
 * @brief Gets the config type of IP.
 * @details This function retrieves the configuration type of the IP for the specified address family.
 *          The IP configuration type indicates whether the IP address is statically configured or dynamically assigned via DHCP.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[out] type           The type of IP config
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED           Not supported
 */
int wifi_manager_ap_get_ip_config_type(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, wifi_manager_ip_config_type_e *type);

/**
 * @brief Sets the config type of IP.
 * @details If you set IP config type to #WIFI_MANAGER_IP_CONFIG_TYPE_STATIC,
 * then IP address, Gateway and Subnet mask will be set to the initial value "0.0.0.0".
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[in] type            The type of IP config
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_ip_config_type(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, wifi_manager_ip_config_type_e type);

/**
 * @brief Gets the IP address.
 * @details This function retrieves the IP address for the specified address family.
 *          The IP address is the unique identifier assigned to a device on a network,
 *          allowing it to communicate with other devices on the same network or across different networks.
 * @since_tizen 3.0
 * @remarks You must release @a ip_address using free().
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[out] ip_address     The IP address; this value is for default address \n
 *                            e.g, IPv6 can have multiple addresses. but you can only obtain a representative address \n
 *                            if you want to get all IPv6 address, you can use the wifi_manager_ap_foreach_ipv6_address().
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NO_CONNECTION        There is no connected AP
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see  wifi_manager_ap_foreach_ipv6_address()
 */
int wifi_manager_ap_get_ip_address(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, char **ip_address);

/**
 * @brief Sets the IP address.
 * @details This function sets the IP address for the specified address family.
 *          If the IP address is set to NULL, the existing IP address for that address family will be deleted.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[in] ip_address      The IP address; if you set this value to NULL, then the existing value will be deleted
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_ip_address(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, const char *ip_address);

/**
 * @brief Called with an IPv6 address.
 * @details This callback function type is used by the `wifi_manager_ap_foreach_ipv6_address()` function
 *          to iterate over each IPv6 address assigned to the Wi-Fi interface for the specified access point.
 *          The callback function receives the IPv6 address as a parameter and returns a boolean value indicating whether to continue iterating or to stop.
 * @since_tizen 4.0
 * @remarks  If @a ipv6_address is needed outside the callback, a copy should be
 *           made. @a ipv6_address will be freed automatically after the execution
 *           of this callback.
 * @param[in]  ipv6_address  The IPv6 address
 * @param[in]  user_data  The user data passed from the foreach function
 * @return  @c true to continue with the next iteration of the loop, \n
 *          @c false to break out of the loop
 * @pre  wifi_manager_ap_foreach_ipv6_address() will invoke this callback.
 * @see  wifi_manager_ap_foreach_ipv6_address()
 */
typedef bool(*wifi_manager_ap_ipv6_address_cb)(char *ipv6_address, void *user_data);

/**
 * @brief Gets all IPv6 addresses assigned to the Wi-Fi interface.
 * @details This function retrieves all IPv6 addresses assigned to the Wi-Fi interface for the specified access point.
 *          It uses a callback function to iterate over each IPv6 address and provide it to the caller.
 * @since_tizen 4.0
 * @param[in] ap        The access point handle
 * @param[in] callback  The callback to be called for each IPv6 address
 * @param[in] user_data The user data passed to the callback function
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE               Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER  Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED   Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED      Not supported
 */
int wifi_manager_ap_foreach_ipv6_address(wifi_manager_ap_h ap,
		wifi_manager_ap_ipv6_address_cb callback, void *user_data);

/**
 * @brief Gets the subnet mask.
 * @details This function retrieves the subnet mask for the specified address family.
 *          The subnet mask is used to determine which part of an IP address represents the network portion and which part represents the host portion.
 * @since_tizen 3.0
 * @remarks You must release @a subnet_mask using free().
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[out] subnet_mask    The subnet mask
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_subnet_mask(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, char **subnet_mask);

/**
 * @brief Sets the subnet mask.
 * @details This function sets the subnet mask for the specified address family.
 *          If the subnet mask is set to NULL, the existing subnet mask for that address family will be deleted.
 * @since_tizen 3.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[in] subnet_mask     The subnet mask; if you set this value to NULL, then the existing value will be deleted
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_subnet_mask(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, const char *subnet_mask);

/**
 * @brief Gets the gateway address.
 * @details This function retrieves the gateway address for the specified address family.
 *          The gateway address is the IP address of the router or gateway that acts as the entry point to the wider internet or network.
 * @since_tizen 3.0
 * @remarks You must release @a gateway_address using free().
 * @param[in] ap                The access point handle
 * @param[in] address_family    The address family
 * @param[out] gateway_address  The gateway address
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_gateway_address(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, char **gateway_address);

/**
 * @brief Sets the gateway address.
 * @details This function sets the gateway address for the specified address family.
 *          If the gateway address is set to NULL, the existing gateway address for that address family will be deleted.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] address_family    The address family
 * @param[in] gateway_address   The gateway address \n
 *                   If you set this value to @c NULL, then the existing value will be deleted.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_gateway_address(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, const char *gateway_address);

/**
 * @brief Gets the DHCP Server address.
 * @details This function retrieves the DHCP server address for the specified address family.
 *          The DHCP server address is the IP address of the server that provides dynamic IP addresses to devices on the network.
 * @since_tizen 4.0
 * @remarks You must release @a dhcp_server using g_free().
 *          This function is supported only for IPv4 address family.
 * @param[in] ap                The access point handle
 * @param[in] address_family    The address family
 * @param[out] dhcp_server      The DHCP server address
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_dhcp_server_address(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, char **dhcp_server);

/**
 * @brief Gets the DHCP lease duration.
 * @details This function retrieves the DHCP lease duration for the specified address family.
 *          The DHCP lease duration indicates the time period for which the IP address assigned to the device through DHCP will remain valid.
 * @since_tizen 4.0
 * @param[in] ap                        The access point handle
 * @param[in] address_family            The address family
 * @param[out] dhcp_lease_duration      The DHCP lease duration in seconds
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_dhcp_lease_duration(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, int *dhcp_lease_duration);

/**
 * @brief Gets the proxy address.
 * @details This function retrieves the proxy address for the specified address family.
 *          The retrieved proxy address can be used to configure the proxy settings for the network interface.
 * @since_tizen 3.0
 * @remarks You must release @a proxy_address using free().
 * @param[in] ap                The access point handle
 * @param[in] address_family    The address family
 * @param[out] proxy_address    The proxy address
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_proxy_address(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, char **proxy_address);

/**
 * @brief Sets the proxy address.
 * @details This function sets the proxy address for the specified address family.
 *          If the proxy address is set to NULL, the existing proxy address for that address family will be deleted.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] address_family    The address family
 * @param[in] proxy_address        The proxy address \n
 *                 If you set this value to @c NULL, then the existing value will be deleted.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_proxy_address(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, const char *proxy_address);

/**
 * @brief Gets the Proxy type.
 * @details This function retrieves the proxy address type for the access point.
 *          The proxy type indicates whether the proxy settings are disabled, set to automatic, or set to manual.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[out] proxy_type       The type of proxy
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_proxy_type(wifi_manager_ap_h ap,
		wifi_manager_proxy_type_e *proxy_type);

/**
 * @brief Sets the Proxy address.
 * @details If you set Proxy type to #WIFI_MANAGER_PROXY_TYPE_AUTO or #WIFI_MANAGER_PROXY_TYPE_MANUAL, then Proxy will be restored.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] proxy_type        The type of proxy
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_proxy_type(wifi_manager_ap_h ap,
		wifi_manager_proxy_type_e proxy_type);

/**
 * @brief Gets the DNS address.
 * @details This function retrieves the DNS address for the specified order and address family.
 *          Up to two DNS addresses can be retrieved for each address family.
 * @since_tizen 3.0
 * @remarks The allowance of DNS address is @c 2.You must release @a dns_address using free().
 * @param[in] ap                The access point handle
 * @param[in] order             The order of DNS address; it starts from 1, which means first DNS address
 * @param[in] address_family    The address family
 * @param[out] dns_address      The DNS address
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_dns_address(wifi_manager_ap_h ap,
		int order, wifi_manager_address_family_e address_family, char **dns_address);

/**
 * @brief Sets the DNS address.
 * @details This function sets the DNS address for the specified order and address family.
 *          Up to two DNS addresses can be set for each address family.
 *          If the DNS address is set to NULL, the existing DNS address for that order and address family will be deleted.
 * @since_tizen 3.0
 * @remarks The allowance of DNS address is @c 2 \n
 * @param[in] ap                The access point handle
 * @param[in] order             The order of DNS address \n
 *                It starts from @c 1, which means first DNS address.
 * @param[in] address_family    The address family
 * @param[in] dns_address       The DNS address \n
 *                If you set this value to @c NULL, then the existing value will be deleted.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_dns_address(wifi_manager_ap_h ap,
		int order, wifi_manager_address_family_e address_family, const char *dns_address);

/**
 * @brief Gets the DNS config type.
 * @details This function retrieves the DNS configuration type for the specified address family.
 *          The DNS configuration type indicates how the DNS server addresses are configured for the network interface.
 * @since_tizen 4.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[out] type           The DNS config type
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_dns_config_type(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, wifi_manager_dns_config_type_e *type);

/**
 * @brief Sets the DNS config type.
 * @details This function sets the DNS configuration type for the specified address family.
 *          The DNS configuration type determines how the DNS server addresses are configured for the network interface.
 * @since_tizen 4.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[in] type            The DNS config type
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_dns_config_type(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, wifi_manager_dns_config_type_e type);

/**
 * @brief Gets the network prefix length.
 * @details This function retrieves the network prefix length for the specified address family.
 *          The network prefix length defines the subnet mask for the network interface,
 *          determining the range of IP addresses available for devices connected to the network.
 * @since_tizen 4.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[out] prefix_len     The network prefix length. \n
 *                            In case of IPv4, it means netmask length \n
 *                            (also called a prefix, e.g. 8, 16, 24, 32)
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_prefix_length(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, int *prefix_len);

/**
 * @brief Sets the network prefix length.
 * @details This function sets the network prefix length for the specified address family.
 *          The network prefix length is used to define the subnet mask for the network interface,
 *          which determines the range of IP addresses available for devices connected to the network.
 * @since_tizen 4.0
 * @param[in] ap              The access point handle
 * @param[in] address_family  The address family
 * @param[in] prefix_len      The network prefix length. \n
 *                            In case of IPv4, it means netmask length \n
 *                            (also called a prefix, e.g. 8, 16, 24, 32)
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_update_ap()
 */
int wifi_manager_ap_set_prefix_length(wifi_manager_ap_h ap,
		wifi_manager_address_family_e address_family, int prefix_len);

/**
 * @brief Gets the Wi-Fi disconnect reason from the supplicant.
 * @details This function retrieves the Wi-Fi disconnect reason from the supplicant.
 *          The disconnect reason provides detailed information about the cause of the disconnection between the device and the access point.
 * @since_tizen 3.0
 * @param[in] ap                    The access point handle
 * @param[out] disconnect_reason    The supplicant disconnect reason
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_disconnect_reason(wifi_manager_ap_h ap,
		wifi_manager_disconnect_reason_e *disconnect_reason);

/**
 * @brief Gets the error state.
 * @details This function retrieves the Wi-Fi connection error state for the access point.
 *          The error state indicates any issues encountered during the connection process to the access point.
 * @since_tizen 4.0
 * @param[in] ap              The access point handle
 * @param[out] error_state    The Wi-Fi connection error state
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_error_state(wifi_manager_ap_h ap, wifi_manager_error_e *error_state);

/**
 * @brief Gets the Wi-Fi Association Status Code from the supplicant.
 * @details This function retrieves the Wi-Fi Association Status Code from the supplicant.
 *          The status code provides detailed information about the result of the association process between the device and the access point.
 * @since_tizen 5.0
 * @param[in] ap              The access point handle
 * @param[out] status_code    The supplicant Wi-Fi association status code
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */

int wifi_manager_ap_get_assoc_status_code(wifi_manager_ap_h ap,
		wifi_manager_assoc_status_code_e *status_code);
/**
 * @brief Called with VSIE data and length of VSIE.
 * @details This callback function type is used to iterate through the Vendor-Specific Information Element (VSIE) list.
 *          It is called once for each VSIE found, providing the VSIE data and its length.
 * @since_tizen 5.0
 * @remarks  If @a vsie is needed outside the callback, a copy should be
 *           made. @a vsie will be freed automatically after the execution
 *           of this callback.
 * @param[in] vsie             The vendor specific data
 * @param[in] length         The length of vendor specific data
 * @param[in] user_data        The user data passed from the foreach function
 * @return  @c true to continue with the next iteration of the loop, \n
 *          @c false to break out of the loop
 * @pre  wifi_manager_ap_foreach_vsie() will invoke this callback.
 * @see  wifi_manager_ap_foreach_vsie().
 */
typedef bool(*wifi_manager_ap_vsie_cb)(unsigned char *vsie, int length, void *user_data);

/**
 * @brief Gets all VSIE of AP.
 * @details This function retrieves all Vendor-Specific Information Elements (VSIE) associated with the access point.
 *          VSIE are custom elements added by vendors to provide additional information about the access point.
 * @since_tizen 5.0
 * @param[in] ap               The access point handle
 * @param[in] callback         The callback to be called for each VSIE of AP
 * @param[in] user_data        The user data passed to the callback function
 * @return 0 on success, otherwise negative error value.
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_foreach_vsie(wifi_manager_ap_h ap,
		wifi_manager_ap_vsie_cb callback, void *user_data);

/**
 * @brief Gets the raw country code.
 * @details This function retrieves the raw country code associated with the access point.
 *          The country code is a two-letter ISO 3166-1 alpha-2 country code that identifies the country where the access point is located.
 * @since_tizen 5.0
 * @remarks You must release @a country_code using free().
 * @param[in] ap  The access point handle
 * @param[out] country_code  The country code, NULL if M/W does not have the country code
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_countrycode(wifi_manager_ap_h ap, char **country_code);

/**
 * @brief Called for each found BSSID.
 * @details This callback function type is used to iterate through the found BSSID list.
 *          It is called once for each BSSID found, providing information about the BSSID,
 *          its signal strength (RSSI), and its operating frequency.
 * @since_tizen 5.0
 * @remarks  The @a bssid can be used only in the callback. To use it outside, make a copy.
 *           @a bssid is managed by the platform and will be released after the execution
 *           of this callback.
 * @param[in]  bssid  The BSSID
 * @param[in]  rssi  The RSSI
 * @param[in]  freq  The frequency
 * @param[in]  user_data  The user data passed from the foreach function
 * @return  @c true to continue with the next iteration of the loop, otherwise @c false to break out of the loop
 * @pre  wifi_manager_foreach_found_bssid() will invoke this callback.
 */
typedef bool(*wifi_manager_found_bssid_cb)(const char *bssid, int rssi, int freq, void *user_data);

/**
 * @brief Gets the BSSID list.
 * @details This function retrieves the BSSID list associated with the access point.
 *          Each BSSID represents a unique identifier for a wireless network interface,
 *          typically corresponding to a specific access point or router.
 * @since_tizen 5.0
 * @param[in] ap  The access point handle
 * @param[in] callback  The callback to be called
 * @param[in] user_data The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @post This function invokes wifi_manager_found_bssid_cb().
 */
int wifi_manager_foreach_found_bssid(wifi_manager_ap_h ap, wifi_manager_found_bssid_cb callback, void *user_data);

/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_SECURITY_MODULE
* @{
*/

/**
 * @brief Gets the Wi-Fi security mode.
 * @details This function retrieves the Wi-Fi security mode for the access point.
 *          The security mode determines the level of protection provided to the Wi-Fi network against unauthorized access and attacks.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[out] type             The type of Wi-Fi security
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_security_type(wifi_manager_ap_h ap, wifi_manager_security_type_e *type);

/**
 * @brief Checks whether a given security type is available in a given AP..
 * @details An AP may support several types of security modes together.
 * You can check all supported security modes with this function.
 * @since_tizen 7.0
 * @param[in] ap             The access point handle
 * @param[in] type           Wi-Fi security type
 * @param[out] supported     true when enabled and false when disabled
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_ap_is_security_type_supported(wifi_manager_ap_h ap,
					wifi_manager_security_type_e type, bool *supported);

/**
 * @brief Sets the Wi-Fi security mode.
 * @details This function sets the Wi-Fi security mode for the access point.
 *          The security mode determines the level of protection provided to the Wi-Fi network against unauthorized access and attacks.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] type              The type of Wi-Fi security
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_security_type(wifi_manager_ap_h ap, wifi_manager_security_type_e type);

/**
 * @brief Gets the Wi-Fi encryption type.
 * @details This function retrieves the Wi-Fi encryption type for the access point.
 *          The encryption type determines how data transmitted over the Wi-Fi network is protected from unauthorized access.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[out] type             The type of Wi-Fi encryption
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_encryption_type(wifi_manager_ap_h ap, wifi_manager_encryption_type_e *type);

/**
 * @brief Sets the Wi-Fi encryption type.
 * @details This function sets the Wi-Fi encryption type for the access point.
 *          The encryption type determines how data transmitted over the Wi-Fi network is protected from unauthorized access.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] type              The type of Wi-Fi encryption
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_encryption_type(wifi_manager_ap_h ap, wifi_manager_encryption_type_e type);

/**
 * @brief Checks whether the passphrase is required or not.
 * @details This function checks whether a passphrase is required for the access point.
 *          A passphrase is necessary to authenticate and establish a secure connection with the access point.
 * @since_tizen 3.0
 * @remarks This function is not valid if security type is #WIFI_MANAGER_SECURITY_TYPE_EAP.
 * @param[in] ap                The access point handle
 * @param[out] required  @c true if passphrase is required,
 *             @c false if passphrase is not required.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_is_passphrase_required(wifi_manager_ap_h ap, bool *required);

/**
 * @brief sets the passphrase for a Wi-Fi access point
 * @details This function sets the passphrase for a Wi-Fi access point.
 *          The passphrase is used to authenticate and establish a secure connection with the access point.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] passphrase        The passphrase of access point
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_passphrase(wifi_manager_ap_h ap, const char *passphrase);

/**
 * @brief Checks whether the WPS(Wi-Fi Protected Setup) is supported or not.
 * @details This function checks whether WPS is supported by the access point.
 *          WPS is a standard that simplifies the process of setting up secure Wi-Fi connections
 *          by allowing devices to connect to each other without entering passwords manually.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[out] supported  @c true if WPS is supported,
 *              otherwise @c false is WPS is not supported.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_connect_by_wps_pbc()
 * @see wifi_manager_connect_by_wps_pin()
 */
int wifi_manager_ap_is_wps_supported(wifi_manager_ap_h ap, bool *supported);

/**
 * @brief Checks whether Protected Management Frame is required.
 * @details This function checks whether PMF is required for the access point.
 *          PMF is a security feature that protects wireless networks from various types of attacks,
 *          such as deauthentication and disassociation attacks.
 * @since_tizen 7.0
 * @param[in] ap             The access point handle
 * @param[out] required      true when required and false when not required
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_ap_is_pmf_required(wifi_manager_ap_h ap, bool *required);
/**
* @}
*/


/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_AP_SECURITY_EAP_MODULE
* @{
*/

/**
 * @brief Sets the passphrase of EAP.
 * @details You can set one of @a user_name and @a password as @c NULL.
 * In this case, the value of a parameter which is set as @c NULL will be the previous value.
 * But it is not allowed that both @a user_name and @a password are set as @c NULL.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_PEAP or #WIFI_MANAGER_EAP_TYPE_TTLS.
 * @param[in] ap                The access point handle
 * @param[in] user_name         The user name \n
 *            This value can be @c NULL.
 * @param[in] password          The password \n
 *            This value can be @c NULL.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_eap_passphrase(wifi_manager_ap_h ap, const char *user_name, const char *password);

/**
 * @brief Gets the passphrase of EAP.
 * @details This function retrieves the user name and password status used for EAP authentication.
 *          It is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_PEAP or #WIFI_MANAGER_EAP_TYPE_TTLS.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_PEAP or #WIFI_MANAGER_EAP_TYPE_TTLS.
 *            You must release @a user_name using free().
 * @param[in] ap                The access point handle
 * @param[out] user_name        The user name
 * @param[out] is_password_set  @c true if password is set,
 *                otherwise @c false if password is not set.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_eap_passphrase(wifi_manager_ap_h ap, char **user_name, bool *is_password_set);

/**
 * @brief Sets access point anonymous identity.
 * @details This function sets the anonymous identity for EAP authentication.
 *          The anonymous identity is an optional field that can be used to provide additional information about the client during the EAP authentication process.
 * @since_tizen 5.5
 * @param[in] ap                    The access point handle
 * @param[in] anonymous_identity    The anonymous identity
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION            Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED                Not supported
 */
int wifi_manager_ap_set_eap_anonymous_identity(wifi_manager_ap_h ap, const char *anonymous_identity);

/**
 * @brief Gets access point anonymous identity.
 * @details This function retrieves the anonymous identity used for EAP authentication.
 *          The anonymous identity is an optional field that can be used to provide additional information about the client during the EAP authentication process.
 * @since_tizen 5.5
 * @remarks You must release @a anonymous_identity using free().
 * @param[in] ap                    The access point handle
 * @param[out] anonymous_identity   The anonymous identity of access point
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY                Out of memory
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION            Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED                Not supported
 */
int wifi_manager_ap_get_eap_anonymous_identity(wifi_manager_ap_h ap, char **anonymous_identity);

/**
 * @brief Gets the CA Certificate of EAP.
 * @details This function retrieves the file path of the CA certificate used for EAP authentication.
 *          It is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 *            You must release @a file using free().
 * @param[in] ap                The access point handle
 * @param[out] file             The file path of CA Certificate
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_eap_ca_cert_file(wifi_manager_ap_h ap, char **file);

/**
 * @brief Sets the CA Certificate of EAP.
 * @details This function sets the CA certificate for EAP authentication.
 *          It is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 *          The CA certificate is used to verify the server's identity during the EAP authentication process.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 * @param[in] ap                The access point handle
 * @param[in] file              The file path of CA Certificate
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_eap_ca_cert_file(wifi_manager_ap_h ap, const char *file);

/**
 * @brief Gets the Client Certificate of EAP.
 * @details This function retrieves the file path of the client certificate used for EAP authentication.
 *          It is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 *            You must release @a file using free().
 * @param[in] ap                The access point handle
 * @param[out] file             The file path of Client Certificate
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_eap_client_cert_file(wifi_manager_ap_h ap, char **file);

/**
 * @brief Sets the Client Certificate of EAP.
 * @details This function sets the client certificate for EAP authentication.
 *          It is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 *          The client certificate is used to verify the client's identity during the EAP authentication process.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 * @param[in] ap                The access point handle
 * @param[in] file              The file path of Client Certificate
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_eap_client_cert_file(wifi_manager_ap_h ap, const char *file);

/**
 * @brief Gets the private key file of EAP.
 * @details This function retrieves the file path of the private key used for EAP authentication.
 *          It is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 *            You must release @a file using free().
 * @param[in] ap                The access point handle
 * @param[out] file             The file path of private key
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_eap_private_key_file(wifi_manager_ap_h ap, char **file);

/**
 * @brief Sets the private key information of EAP.
 * @details This function sets the private key information for EAP authentication.
 *          It is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 *          The private key information includes the file path of the private key and its corresponding password.
 * @since_tizen 3.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 * @param[in] ap                The access point handle
 * @param[in] file              The file path of private key
 * @param[in] password          The password
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_eap_private_key_info(wifi_manager_ap_h ap,
		const char *file, const char *password);

/**
 * @brief Gets the EAP type of Wi-Fi.
 * @details This function retrieves the EAP type set for the given Wi-Fi access point.
 *          The EAP type specifies the EAP method that should be used during the EAP authentication process.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[out] type             The type of EAP
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_eap_type(wifi_manager_ap_h ap, wifi_manager_eap_type_e *type);

/**
 * @brief Sets the EAP type of Wi-Fi.
 * @details This function sets the EAP type for the given Wi-Fi access point.
 *          The EAP type specifies the EAP method that should be used during the EAP authentication process.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] type              The type of EAP
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_eap_type(wifi_manager_ap_h ap, wifi_manager_eap_type_e type);

/**
 * @brief Gets the type of EAP phase2 authentication of Wi-Fi.
 * @details This function retrieves the EAP Phase 2 authentication type set for the given Wi-Fi access point.
 *          The EAP Phase 2 authentication type specifies how the client should authenticate itself during the second phase of the EAP authentication process.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[out] type             The type of EAP phase2 authentication
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_get_eap_auth_type(wifi_manager_ap_h ap, wifi_manager_eap_auth_type_e *type);

/**
 * @brief Sets the type of EAP phase2 authentication of Wi-Fi.
 * @details This function sets the EAP Phase 2 authentication type for the given Wi-Fi access point.
 *          The EAP Phase 2 authentication type specifies how the client should authenticate itself during the second phase of the EAP authentication process.
 * @since_tizen 3.0
 * @param[in] ap                The access point handle
 * @param[in] type              The type of EAP phase2 authentication
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ap_set_eap_auth_type(wifi_manager_ap_h ap, wifi_manager_eap_auth_type_e type);

/**
* @}
*/

/**
* @addtogroup CAPI_NETWORK_WIFI_MANAGER_CONFIG_MODULE
* @{
*/

/**
 * @brief Gets access point configuration handle.
 * @details This function creates a new access point configuration handle with the provided parameters.
 *          The handle can be used to save or update the access point configuration.
 * @since_tizen 3.0
 * @remarks You must release @a config using wifi_manager_config_destroy().
 *
 * @param[in] wifi              The Wi-Fi handle
 * @param[in] name              The access point name
 * @param[in] passphrase        The access point passphrase
 * @param[in] security_type     The access point security type
 * @param[out] config           The access point configuration handle
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre This function needs a wifi_manager_initialize() call before use
 * @see wifi_manager_config_destroy()
 */
int wifi_manager_config_create(wifi_manager_h wifi, const char *name,
		const char *passphrase, wifi_manager_security_type_e security_type, wifi_manager_config_h *config);

/**
 * @brief Clones the access point configuration handle.
 * @details This function creates a copy of the given access point configuration handle.
 *          The new handle has the same properties as the original handle.
 * @since_tizen 3.0
 * @remarks You must release @a cloned_config using wifi_manager_config_destroy().
 *
 * @param[in] origin            The origin access point configuration handle
 * @param[out] cloned_config    The cloned access point configuration handle
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_destroy()
 */
int wifi_manager_config_clone(wifi_manager_config_h origin, wifi_manager_config_h *cloned_config);

/**
 * @brief Destroys the access point configuration handle.
 * @details This function destroys the access point configuration handle and releases all its associated resources.
 *          After calling this function, the handle is no longer valid.
 * @since_tizen 3.0
 *
 * @param[in] config            The access point configuration handle
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_create()
 * @see wifi_manager_config_clone()
 */
int wifi_manager_config_destroy(wifi_manager_config_h config);

/**
 * @brief Saves Wi-Fi configuration of access point.
 * @details When a configuration is changed, these changes will be not applied to the Connection Manager immediately.\n
 *            When you call this function, your changes affect the Connection Manager and the existing configuration is updated.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile
 *
 * @param[in] wifi              The Wi-Fi handle
 * @param[in] config            The access point configuration handle
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_create()
 */
int wifi_manager_config_save(wifi_manager_h wifi, wifi_manager_config_h config);

/**
 * @brief Removes Wi-Fi configuration of access point.
 * @details This function deletes the specified access point configuration from the system.
 *          After this function is called, the access point configuration will no longer be available.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile
 *
 * @param[in] wifi              The Wi-Fi handle
 * @param[in] config            The access point configuration handle
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_create()
 * @see wifi_manager_config_foreach_configuration()
 */
int wifi_manager_config_remove(wifi_manager_h wifi, wifi_manager_config_h config);

/**
 * @brief Gets configurations of an access point.
 * @details This function retrieves all access point configurations stored in the system.
 *          Each access point configuration is represented by a handle,
 *          which can be used to get detailed information about the access point.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.profile
 *
 * @param[in] wifi              The Wi-Fi handle
 * @param[in] callback          The callback to be called
 * @param[in] user_data         The user data passed to the callback function
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @pre This function needs wifi_manager_initialize() before use.
 * @post This function invokes wifi_manager_config_list_cb().
 */
int wifi_manager_config_foreach_configuration(wifi_manager_h wifi,
		wifi_manager_config_list_cb callback, void *user_data);

/**
 * @brief Gets the name of access point from configuration.
 * @details This function retrieves the name set for the given access point configuration.
 *          The name is a user-friendly identifier for the access point.
 * @since_tizen 3.0
 * @remarks You must release @a name using free().
 *
 * @param[in] config            The access point configuration handle
 * @param[out] name             The name of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_name(wifi_manager_config_h config, char **name);

/**
 * @brief Gets the security type of access point from configuration.
 * @details This function retrieves the security type set for the given access point configuration.
 *          The security type determines the type of security mechanism used to protect the wireless network.
 * @since_tizen 3.0
 *
 * @param[in] config            The access point configuration handle
 * @param[out] security_type    The security type of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_security_type(wifi_manager_config_h config,
		wifi_manager_security_type_e *security_type);

/**
 * @brief Sets access point proxy address configuration.
 * @details This function sets the proxy address for the given access point configuration.
 *          The proxy address is used to route traffic through a proxy server.
 * @since_tizen 3.0
 *
 * @param[in] config            The access point configuration handle
 * @param[in] address_family    The address family
 * @param[in] proxy_address     The proxy address
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_proxy_address(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, const char *proxy_address);

/**
 * @brief Gets the proxy address of access point from configuration.
 * @details This function retrieves the proxy address set for the given access point configuration.
 *          The proxy address is used to route traffic through a proxy server.
 * @since_tizen 3.0
 * @remarks You must release @a proxy_address using free().
 *
 * @param[in] config            The access point configuration handle
 * @param[out] address_family   The address family
 * @param[out] proxy_address    The proxy address
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_proxy_address(wifi_manager_config_h config,
		wifi_manager_address_family_e *address_family, char **proxy_address);

/**
 * @brief Sets the hidden property of access point from the configuration.
 * @details This function sets the hidden property of the access point in the given access point configuration.
 *          The hidden property indicates whether the access point is hidden or not.
 * @since_tizen 3.0
 *
 * @param[in] config            The access point configuration handle
 * @param[in] is_hidden         The new value of the hidden property
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_hidden_ap_property(wifi_manager_config_h config, bool is_hidden);

/**
 * @brief Gets the hidden property of access point from the configuration.
 * @details This function retrieves the hidden property of the access point set in the given access point configuration.
 *          The hidden property indicates whether the access point is hidden or not.
 * @since_tizen 3.0
 *
 * @param[in] config            The access point configuration handle
 * @param[out] is_hidden        The hidden property of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_hidden_ap_property(wifi_manager_config_h config, bool *is_hidden);

/**
 * @brief Gets access point IP config type from configuration.
 * @details This function retrieves the IP configuration type set for the given access point configuration.
 *          The IP configuration type determines how the IP address and related information (e.g., subnet mask, gateway) are obtained.
 * @since_tizen 5.0
 *
 * @param[in] config		The access point configuration handle
 * @param[in] address_family	The address family
 * @param[out] type		The type of IP configuration
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE			Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED		Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER	Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED	Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED		Not supported
 */
int wifi_manager_config_get_ip_config_type(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family,
		wifi_manager_ip_config_type_e *type);

/**
 * @brief Sets access point IP config type to configuration.
 * @details This function sets the IP configuration type for the given access point configuration.
 *          The IP configuration type determines how the IP address and related information (e.g., subnet mask, gateway) are obtained.
 * @since_tizen 5.0
 *
 * @param[in] config            The access point configuration handle
 * @param[in] address_family    The address family
 * @param[in] type		The type of IP configuration
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_ip_config_type(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family,
		wifi_manager_ip_config_type_e type);

/**
 * @brief Gets access point IP address from configuration.
 * @details This function retrieves the IP address set for the given access point configuration.
 *          The IP address is used as the local address for the device when connected to the access point.
 * @since_tizen 5.0
 * @remarks You must release @a ip_address using free().
 *
 * @param[in] config		The access point configuration handle
 * @param[in] address_family	The address family
 * @param[out] ip_address	The IP address; this value is for default address
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_ip_address(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, char **ip_address);

/**
 * @brief Sets access point IP address to configuration.
 * @details This function sets the IP address for the given access point configuration.
 *          The IP address is used as the local address for the device when connected to the access point.
 * @since_tizen 5.0
 *
 * @param[in] config			The access point configuration handle
 * @param[in] address_family	The address family
 * @param[in] ip_address		The IP address; if you set this value to NULL, then
 *				the existing value will be deleted.
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_ip_address(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, const char *ip_address);

/**
 * @brief Gets access point subnet mask from configuration.
 * @details This function retrieves the subnet mask set for the given access point configuration.
 *          The subnet mask is used to specify the network portion of an IP address.
 * @since_tizen 5.0
 * @remarks You must release @a subnet_mask using free().
 *
 * @param[in] config            The access point configuration handle
 * @param[in] address_family	The address family
 * @param[out] subnet_mask	The subnet mask
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_subnet_mask(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, char **subnet_mask);

/**
 * @brief Sets access point subnet mask to configuration.
 * @details This function sets the subnet mask for the given access point configuration.
 *          The subnet mask is used to specify the network portion of an IP address.
 * @since_tizen 5.0
 *
 * @param[in] config            The access point configuration handle
 * @param[in] address_family    The address family
 * @param[in] subnet_mask	The subnet mask; if you set this value to NULL, then
 *				the existing value will be deleted.
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_subnet_mask(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, const char *subnet_mask);

/**
 * @brief Gets the network prefix length from configuration.
 * @details This function retrieves the network prefix length set for the given access point configuration.
 *          The network prefix length specifies the number of leading bits in the IP address that are used to determine the network portion of the address.
 * @since_tizen 5.0
 *
 * @param[in] config          The access point configuration handle
 * @param[in] address_family  The address family
 * @param[out] prefix_len     The network prefix length. \n
 *                            In case of IPv4, it means netmask length \n
 *                            (also called a prefix, e.g. 8, 16, 24, 32)
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_prefix_length(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, int *prefix_len);

/**
 * @brief Sets the network prefix length to configuration.
 * @details This function sets the network prefix length for the given access point configuration.
 *          The network prefix length specifies the number of leading bits in the IP address that are used to determine the network portion of the address.
 * @since_tizen 5.0
 *
 * @param[in] config          The access point configuration handle
 * @param[in] address_family  The address family
 * @param[in] prefix_len      The network prefix length. \n
 *                            In case of IPv4, it means netmask length \n
 *                            (also called a prefix, e.g. 8, 16, 24, 32)
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_prefix_length(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, int prefix_len);

/**
 * @brief Gets access point gateway address from configuration.
 * @details This function retrieves the gateway address set for the given access point configuration.
 *          The gateway address is used as the default route for the network traffic.
 * @since_tizen 5.0
 * @remarks You must release @a gateway_address using free().
 *
 * @param[in] config            The access point configuration handle
 * @param[in] address_family	The address family
 * @param[out] gateway_address	The gateway address; this value is for default address
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_gateway_address(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family, char **gateway_address);

/**
 * @brief Sets access point gateway address to configuration.
 * @details This function sets the gateway address for the given access point configuration.
 *          The gateway address is used as the default route for the network traffic.
 * @since_tizen 5.0
 *
 * @param[in] config            The access point configuration handle
 * @param[in] address_family    The address family
 * @param[in] gateway_address	The gateway address; if you set this value to NULL, then
 *				the existing value will be deleted.
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_gateway_address(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family,
		const char *gateway_address);

/**
 * @brief Gets access point dns config type from configuration.
 * @details This function retrieves the DNS configuration type set for the given access point configuration.
 *          The DNS configuration type specifies how the DNS addresses are configured for the access point.
 * @since_tizen 5.0
 *
 * @param[in] config		The access point configuration handle
 * @param[in] address_family	The address family
 * @param[out] type		The type of dns configuration
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE			Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED		Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER	Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED	Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED		Not supported
 */
int wifi_manager_config_get_dns_config_type(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family,
		wifi_manager_dns_config_type_e *type);

/**
 * @brief Sets access point dns config type to configuration.
 * @details This function sets the DNS configuration type for the given access point configuration.
 *          The DNS configuration type specifies how the DNS addresses are configured for the access point.
 * @since_tizen 5.0
 *
 * @param[in] config            The access point configuration handle
 * @param[in] address_family    The address family
 * @param[in] type		The type of DNS configuration
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_dns_config_type(wifi_manager_config_h config,
		wifi_manager_address_family_e address_family,
		wifi_manager_dns_config_type_e type);

/**
 * @brief Gets access point dns address from configuration.
 * @details This function retrieves the DNS address set for the given access point configuration.
 *          The DNS address is used to resolve domain names to IP addresses during the network communication process.
 * @since_tizen 5.0
 * @remarks The allowance of DNS address is @c 2.You must release @a dns_address using free().
 *
 * @param[in] config            The access point configuration handle
 * @param[in] order             The order of DNS address; it starts from 1, which means first DNS address
 * @param[in] address_family    The address family
 * @param[out] dns_address      The DNS address
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED  Address family not supported
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_dns_address(wifi_manager_config_h config,
		int order, wifi_manager_address_family_e address_family,
		char **dns_address);

/**
 * @brief Sets access point dns address to configuration.
 * @details This function sets the DNS address for the given access point configuration.
 *          The DNS address is used to resolve domain names to IP addresses during the network communication process.
 * @since_tizen 5.0
 *
 * @param[in] config            The access point configuration handle
 * @param[in] order             The order of DNS address \n
 *				It starts from 1, which means first DNS address.
 * @param[in] address_family    The address family
 * @param[in] dns_address	The dns address; if you set this value to NULL, then
 *				the existing value will be deleted.
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED Not supported address family
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_dns_address(wifi_manager_config_h config,
		int order, wifi_manager_address_family_e address_family,
		const char *dns_address);

/**
 * @brief Gets access point anonymous identity from configuration.
 * @details This function retrieves the anonymous identity set for the given access point configuration.
 *          The anonymous identity is used during the authentication process to identify the client anonymously when connecting to the access point.
 * @since_tizen 3.0
 * @remarks You must release @a anonymous_identity using free().
 *
 * @param[in] config                The access point configuration handle
 * @param[out] anonymous_identity   The anonymous identity of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_config_get_eap_anonymous_identity(wifi_manager_config_h config,
		char **anonymous_identity);

/**
 * @brief Sets access point anonymous identity to configuration.
 * @details This function sets the anonymous identity for the given access point configuration.
 *          The anonymous identity is used during the authentication process to identify the client anonymously when connecting to the access point.
 * @since_tizen 3.0
 *
 * @param[in] config                The access point configuration handle
 * @param[in] anonymous_identity    The anonymous identity
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_anonymous_identity(wifi_manager_config_h config,
		const char *anonymous_identity);

/**
 * @brief Gets access point cacert file from configuration.
 * @details This function retrieves the CA certificate file path set for the given access point configuration.
 *          The CA certificate file contains the trusted root certificates used to verify the server's certificate during the authentication process.
 * @since_tizen 3.0
 * @remarks You must release @a ca_cert using free().
 * @remarks The mediastorage privilege %http://tizen.org/privilege/mediastorage is needed \n
 *            if @a ca_cert is relevant to media storage.\n
 *            The externalstorage privilege %http://tizen.org/privilege/externalstorage is needed \n
 *            if @a ca_cert is relevant to external storage.
 *
 * @param[in] config           The access point configuration handle
 * @param[out] ca_cert         The certification authority(CA) certificates file of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_config_get_eap_ca_cert_file(wifi_manager_config_h config, char **ca_cert);

/**
 * @brief Sets access point cacert file to configuration.
 * @details This function sets the CA certificate file required for EAP authentication.
 *          The CA certificate file contains the trusted root certificates used to verify the server's certificate during the authentication process.
 * @since_tizen 3.0
 * @remarks The mediastorage privilege %http://tizen.org/privilege/mediastorage is needed \n
 *            if @a ca_cert is relevant to media storage.\n
 *            The externalstorage privilege %http://tizen.org/privilege/externalstorage is needed \n
 *            if @a ca_cert is relevant to external storage.
 *
 * @param[in] config           The access point configuration handle
 * @param[in] ca_cert          The certification authority(CA) certificates file of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_ca_cert_file(wifi_manager_config_h config, const char *ca_cert);

/**
 * @brief Gets access point client cert file from configuration.
 * @details This function retrieves the client certificate file path set for the given access point configuration.
 *          The client certificate file is used for EAP authentication to establish a secure connection with the access point.
 * @since_tizen 3.0
 * @remarks You must release @a client_cert using free().
 *
 * @param[in] config           The access point configuration handle
 * @param[out] client_cert     The certification authority(CA) certificates file of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_config_get_eap_client_cert_file(wifi_manager_config_h config, char **client_cert);

/**
 * @brief Sets access point client cert file to configuration.
 * @details This function sets the client certificate and private key files required for EAP authentication.
 *          These files are used to establish a secure connection with the access point.
 * @since_tizen 3.0
 *
 * @param[in] config           The access point configuration handle
 * @param[in] private_key      The private key file
 * @param[in] client_cert      The certification authority(CA) certificates file of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_client_cert_file(wifi_manager_config_h config,
		const char *private_key, const char *client_cert);

/**
 * @brief Gets the private key file of EAP.
 * @details This function retrieves the file path of the private key used for EAP-TLS authentication.
 *          This function is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 * @since_tizen 5.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 *            You must release @a file using free().
 *
 * @param[in] config           The access point configuration handle
 * @param[out] file            The file path of private key
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OUT_OF_MEMORY        Out of memory
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_eap_private_key_file(wifi_manager_config_h config, char **file);

/**
 * @brief Sets the private key information of EAP.
 * @details This function sets the private key information required for EAP-TLS authentication.
 *          It includes the file path of the private key and its corresponding password.
 *          This function is only valid if the EAP type is set to #WIFI_MANAGER_EAP_TYPE_TLS.
 * @since_tizen 5.0
 * @remarks This function is valid only if the EAP type is #WIFI_MANAGER_EAP_TYPE_TLS.
 *
 * @param[in] config           The access point configuration handle
 * @param[in] file             The file path of private key
 * @param[in] password         The password
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_private_key_info(wifi_manager_config_h config,
		const char *file, const char *password);

/**
 * @brief Gets access point identity from configuration.
 * @details This function retrieves the EAP identity set for the given access point configuration.
 *          The EAP identity is used during the authentication process to identify the client when connecting to the access point.
 * @since_tizen 3.0
 * @remarks You must release @a identity using free().
 *
 * @param[in] config           The access point configuration handle
 * @param[out] identity        The identity of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_config_get_eap_identity(wifi_manager_config_h config, char **identity);

/**
 * @brief Sets access point identity to configuration.
 * @details This function sets the EAP identity for the given access point configuration.
 *          The EAP identity is used during the authentication process to identify the client when connecting to the access point.
 * @since_tizen 3.0
 *
 * @param[in] config           The access point configuration handle
 * @param[in] identity         The identity
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_identity(wifi_manager_config_h config, const char *identity);

/**
 * @brief Gets access point EAP type from configuration.
 * @details This function retrieves the EAP type set for the given access point configuration.
 *          The EAP type specifies the EAP protocol used to authenticate the client when connecting to the access point.
 * @since_tizen 3.0
 *
 * @param[in] config           The access point configuration handle
 * @param[out] eap_type        The EAP type of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_config_get_eap_type(wifi_manager_config_h config, wifi_manager_eap_type_e *eap_type);

/**
 * @brief Sets access point EAP type to configuration.
 * @details This function sets the EAP type for the given access point configuration.
 *          The EAP type specifies the EAP protocol used to authenticate the client when connecting to the access point.
 * @since_tizen 3.0
 *
 * @param[in] config           The access point configuration handle
 * @param[in] eap_type         The EAP type
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_type(wifi_manager_config_h config, wifi_manager_eap_type_e eap_type);

/**
 * @brief Gets access point EAP auth type from configuration.
 * @details This function retrieves the EAP authentication type set for the given access point configuration.
 *          The EAP authentication type specifies the method used to authenticate the client when connecting to the access point.
 * @since_tizen 3.0
 *
 * @param[in] config           The access point configuration handle
 * @param[out] eap_auth_type   The EAP auth type of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_config_get_eap_auth_type(wifi_manager_config_h config,
		wifi_manager_eap_auth_type_e *eap_auth_type);

/**
 * @brief Sets access point EAP auth type to configuration.
 * @details This function sets the EAP authentication type for the given access point configuration.
 * The EAP authentication type specifies the method used to authenticate the client when connecting to the access point.
 * @since_tizen 3.0
 *
 * @param[in] config           The access point configuration handle
 * @param[in] eap_auth_type    The EAP auth type
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_auth_type(wifi_manager_config_h config,
		wifi_manager_eap_auth_type_e eap_auth_type);

/**
 * @brief Gets access point subject match from configuration.
 * @details This function retrieves the EAP subject match set for the given access point configuration.
 *          The EAP subject match specifies which certificates to trust when connecting to the access point.
 * @since_tizen 3.0
 * @remarks You must release @a subject_match using free().
 *
 * @param[in] config           The access point configuration handle
 * @param[out] subject_match   The subject match of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 */
int wifi_manager_config_get_eap_subject_match(wifi_manager_config_h config,
		char **subject_match);

/**
 * @brief Sets access point subject match to configuration.
 * @details This function sets the EAP subject match for the given access point configuration.
 *          The EAP subject match is used to specify which certificates to trust when connecting to the access point.
 * @since_tizen 3.0
 *
 * @param[in] config           The access point configuration handle
 * @param[in] subject_match    The subject match
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_eap_subject_match(wifi_manager_config_h config,
		const char *subject_match);

/**
 * @brief Gets access point frequency from configuration.
 * @details This function gets the frequency band (MHz) for the given access point configuration.
 *          The frequency band indicates the radio frequency used by the access point.
 * @since_tizen 10.0
 *
 * @param[in] config        The access point configuration handle
 * @param[out] frequency    The frequency of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_config_get_frequency(wifi_manager_config_h config,
		int *frequency);

/**
 * @brief Sets access point frequency to configuration.
 * @details This function sets the frequency band (MHz) for the given access point configuration.
 *          The frequency band determines the radio frequency the access point will use.
 * @since_tizen 10.0
 *
 * @param[in] config        The access point configuration handle
 * @param[in] frequency     The frequency of access point
 *
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 * @see wifi_manager_config_save()
 */
int wifi_manager_config_set_frequency(wifi_manager_config_h config,
		int frequency);

/**
 * @}
 */

/**
 * @addtogroup CAPI_NETWORK_WIFI_MANAGER_TDLS_MODULE
 * @{
 */

/**
 * @brief Registers the callback called when a TDLS device is found.
 * @details This function registers a callback function to receive notifications about discovered TDLS (802.11z) devices.
 *          When a TDLS device is found, the provided callback function will be called with the device information as an argument.
 * @since_tizen 4.0
 *
 * @param[in] wifi             The Wi-Fi handle
 * @param[in] callback         The callback function to be called
 * @param[in] user_data The user data passed to the callback function
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_tdls_set_discovered_cb(wifi_manager_h wifi,
		wifi_manager_tdls_discovered_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when TDLS device is found.
 * @details This function unregisters the callback that was previously registered using wifi_manager_tdls_set_discovered_cb().
 *          After calling this function, the application will no longer receive notifications about discovered TDLS devices.
 * @since_tizen 4.0
 *
 * @param[in] wifi             The Wi-Fi handle
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 */
int wifi_manager_tdls_unset_discovered_cb(wifi_manager_h wifi);

/**
 * @brief Discovers devices that support TDLS.
 * @details This function initiates a discovery process to find TDLS (802.11z)-capable devices within range.
 *          The MAC address of the peer device to be discovered can be optionally provided as an argument.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 *
 * @param[in] wifi             The Wi-Fi handle
 * @param[in] peer_mac_addr    The MAC address (e.g., ff:ff:ff:ff:ff:ff,
 *                             AA:BB:CC:DD:EE:FF) of the peer to be discovered
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 */
int wifi_manager_tdls_start_discovery(wifi_manager_h wifi, const char *peer_mac_addr);

/**
 * @brief Connects to a peer device.
 * @details This function establishes a TDLS (802.11z) connection with the specified peer device.
 *          The MAC address of the peer device to be connected must be provided as an argument.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 *
 * @param[in] wifi             The Wi-Fi handle
 * @param[in] peer_mac_addr    The MAC address of the peer device to be connected
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_tdls_connect(wifi_manager_h wifi, const char *peer_mac_addr);

/**
 * @brief Disconnects the connected peer.
 * @details This function disconnects the currently connected TDLS (802.11z) peer.
 *          The MAC address of the peer to be disconnected must be provided as an argument.
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 *
 * @param[in] wifi                The Wi-Fi handle
 * @param[in] peer_mac_addr       The MAC address of the connected peer
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_tdls_disconnect(wifi_manager_h wifi, const char *peer_mac_addr);

/**
 * @brief Gets Peer Mac address of Connected peer.
 * @details This function retrieves the MAC address of the currently connected TDLS (802.11z) peer.
 * The MAC address is returned as a null-terminated string and must be freed by the caller after use
 * @since_tizen 3.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 *
 * @remarks The @a peer_mac_addr should be freed using free().
 * @param[in] wifi                The Wi-Fi handle
 * @param[out] peer_mac_addr      The MAC address of the connected peer
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_NO_CONNECTION        No active TDLS Connection
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_tdls_get_connected_peer(wifi_manager_h wifi, char **peer_mac_addr);

/**
 * @brief Registers the callback called when TDLS state is changed.
 * @details This function registers a callback function to receive notifications about changes in the TDLS (802.11z) state.
 * When the TDLS state changes, the provided callback function will be called with the new state as an argument.
 * @since_tizen 3.0
 *
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_tdls_set_state_changed_cb(wifi_manager_h wifi,
		wifi_manager_tdls_state_changed_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when TDLS state is changed.
 * @details This function unregisters the callback that was previously registered using wifi_manager_tdls_set_state_changed_cb().
 *          After calling this function, the application will no longer receive notifications about changes in the TDLS state.
 * @since_tizen 3.0
 *
 * @param[in] wifi            The Wi-Fi handle
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_tdls_unset_state_changed_cb(wifi_manager_h wifi);


/**
 * @brief Enables a TDLS channel switching request.
 * @details This function enables the TDLS (802.11z) channel switching request for the specified TDLS peer.
 *          Channel switching allows the two peers to switch to a different channel to improve communication performance.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 *
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] peer_mac_addr   The MAC address of the connected TDLS peer
 * @param[in] freq            The new frequency [MHz]
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */

int wifi_manager_tdls_enable_channel_switching(wifi_manager_h wifi,
		const char *peer_mac_addr, int freq);


/**
 * @brief Disables a TDLS channel switching request.
 * @details This function disables the TDLS (802.11z) channel switching request for the specified TDLS peer.
 * Channel switching allows the two peers to switch to a different channel to improve communication performance.
 * @since_tizen 4.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 *
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] peer_mac_addr   The MAC address of the connected TDLS peer
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */

int wifi_manager_tdls_disable_channel_switching(wifi_manager_h wifi,
		const char *peer_mac_addr);


/**
 * @brief Registers the callback called when IP conflict state is changed.
 * @details This function registers a callback function to receive notifications about changes in the IP conflict state.
 *          When the IP conflict state changes, the provided callback function will be called with the new state as an argument.
 * @since_tizen 5.0
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] callback        The callback function to be called
 * @param[in] user_data       The user data passed to the callback function
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_set_ip_conflict_cb(wifi_manager_h wifi,
		wifi_manager_ip_conflict_cb callback, void *user_data);

/**
 * @brief Unregisters the callback called when IP conflict state is changed.
 * @details This function unregisters the callback that was previously registered using wifi_manager_set_ip_conflict_cb().
 *          After calling this function, the application will no longer receive notifications about changes in the IP conflict state.
 * @since_tizen 5.0
 * @param[in] wifi            The Wi-Fi handle
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_unset_ip_conflict_cb(wifi_manager_h wifi);

/**
 * @brief Enables or disables IP conflict detection.
 * @details This function enables or disables the IP conflict detection feature on the device. When enabled,
 *          the device periodically checks for IP address conflicts with other devices on the same network.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.set
 *
 * @param[in] wifi            The Wi-Fi handle
 * @param[in] detect          True or false boolean to enable or disable
 *
 * @return @c 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_INVALID_OPERATION    Invalid operation
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_set_ip_conflict_detect_enable(wifi_manager_h wifi,
		bool detect);

/**
 * @brief Checks whether IP conflict detection is enabled.
 * @details This function checks if the IP conflict detection feature is enabled on the device.
 *          If the feature is enabled, the function sets the @a state parameter to @c true;
 *          otherwise, it sets it to @c false.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi        The Wi-Fi handle
 * @param[out] state     @c true if IP conflict detection is enabled,
 *             otherwise @c false if IP conflict detection is disabled.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_ip_conflict_detect_is_enabled(wifi_manager_h wifi, bool *state);

/**
 * @brief Gets the state of the IP conflict.
 * @details This function retrieves the current state of the IP conflict detection feature.
 *          The IP conflict state indicates whether there is an IP address conflict
 *          with other devices on the same network.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi          The Wi-Fi handle
 * @param[out] state        The current state of IP conflict
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_get_ip_conflict_state(wifi_manager_h wifi,
		wifi_manager_ip_conflict_state_e *state);

/**
 * @brief Gets whether 5Ghz Wi-Fi band is supported.
 * @details This function checks if the device supports the 5Ghz Wi-Fi band.
 *          If the device supports it, the function sets the @a supported parameter to @c true;
 *          otherwise, it sets it to @c false.
 * @since_tizen 5.5
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi          The Wi-Fi handle
 * @param[out] supported   @c true if 5Ghz Wi-Fi band is supported,
 *             otherwise   @c false if 5Ghz Wi-Fi band is not supported.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_is_5ghz_band_supported(wifi_manager_h wifi, bool *supported);

/**
 * @brief Gets whether 6Ghz Wi-Fi band is supported.
 * @details This function checks if the device supports the 6Ghz Wi-Fi band.
 *          If the device supports it, the function sets the @a supported parameter to @c true;
 *          otherwise, it sets it to @c false.
 * @since_tizen 8.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/network.get
 * @param[in] wifi          The Wi-Fi handle
 * @param[out] supported   @c true if 6Ghz Wi-Fi band is supported,
 *             otherwise   @c false if 6Ghz Wi-Fi band is not supported.
 * @return 0 on success, otherwise negative error value
 * @retval #WIFI_MANAGER_ERROR_NONE                 Successful
 * @retval #WIFI_MANAGER_ERROR_NOT_INITIALIZED      Not initialized
 * @retval #WIFI_MANAGER_ERROR_INVALID_PARAMETER    Invalid parameter
 * @retval #WIFI_MANAGER_ERROR_OPERATION_FAILED     Operation failed
 * @retval #WIFI_MANAGER_ERROR_PERMISSION_DENIED    Permission Denied
 * @retval #WIFI_MANAGER_ERROR_NOT_SUPPORTED        Not supported
 */
int wifi_manager_is_6ghz_band_supported(wifi_manager_h wifi, bool *supported);

/**
 * @}
 */


#ifdef __cplusplus
}
#endif

#endif /* __TIZEN_WIFI_MANAGER_H__ */
