#!/usr/bin/env python
#
# This file is part of REPA: Release Engineering Process Assistant.
#
# Copyright (C) 2013 Intel Corporation
#
# REPA is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA 02110-1301, USA.

"""
REPA: Release Engineering Process Assistant.

Copyright (C) Intel Corporation 2013
Licence: GPL version 2
Author: Ed Bartosh <eduard.bartosh@intel.com>

Accept module.
Accept submissions.
"""

import sys
from collections import namedtuple

from repa.obs import OBS
from repa.main import sub_main
from repa.common import accept_or_reject, edit


class Reject(object):
    """Subcommand: reject submissions."""

    name = 'reject'
    description = 'Reject submission or group'
    help = description

    @staticmethod
    def add_arguments(parser, config):
        """Adds arguments to the parser. Called from [sub_]main."""
        parser.add_argument('submission', help='submission or group')
        parser.add_argument('-c', '--comment', help='comment',
                            default=config.get('reject_comment', ''))
        parser.add_argument('-j', '--jenkins', action='store_true',
                            help='trigger Jenkins job')
        parser.add_argument('-e', '--edit', action='store_true',
                            help='run editor to edit comment')

    @staticmethod
    def run(argv):
        """Command line entry point. Called from [sub_]main."""
        if argv.edit:
            argv.comment = edit(argv.comment)

        obs = OBS(argv.apiurl, argv.apiuser, argv.apipasswd)

        cred = None
        if argv.jenkins:
            cred = namedtuple('cred', ['url', 'username', 'password'])(\
                     argv.jenkins_url, argv.jenkins_user, argv.jenkins_passwd)

        return accept_or_reject(obs, argv.submission, 'declined',
                                argv.project, argv.comment, cred)

if __name__ == '__main__':
    sys.exit(sub_main(sys.argv[1:], Reject()))
